<?php

namespace Modules\Chat\Providers;

use Closure;
use Modules\Core\Support\ModuleServiceProvider;
use Modules\Core\Menu\MenuItem;
use Modules\Core\Settings\SettingsMenuItem;
use Modules\Core\Facades\Innoclapps;
use Modules\Chat\Services\EvolutionSyncService;
use Modules\Chat\Services\ChatRedisService;
use Modules\Chat\Services\WhatsAppContactService;
use Modules\Chat\Services\ContactProfilePhotoService;
use Modules\Chat\Services\EvolutionApiService;

class ChatServiceProvider extends ModuleServiceProvider
{
    public function boot(): void
    {
        $this->loadViewsFrom(module_path('Chat', 'resources/views'), 'chat');
        $this->loadRoutesFrom(module_path('Chat', 'routes/web.php'));
        $this->loadRoutesFrom(module_path('Chat', 'routes/api.php'));
        $this->loadTranslationsFrom(module_path('Chat', 'resources/lang'), 'chat');
        $this->loadMigrationsFrom(module_path('Chat', 'database/migrations'));

        $this->registerChatPermissions();
    }

    public function register(): void
    {
        if (class_exists(\Modules\Chat\Providers\RouteServiceProvider::class)) {
            $this->app->register(\Modules\Chat\Providers\RouteServiceProvider::class);
        }
        
        $this->app->singleton(EvolutionSyncService::class, function ($app) {
            return new EvolutionSyncService(
                $app->make(EvolutionApiService::class),
                $app->make(ChatRedisService::class),
                $app->make(WhatsAppContactService::class), 
                $app->make(ContactProfilePhotoService::class)
            );
        });
    }

    protected function registerChatPermissions(): void
    {
        Innoclapps::permissions(function ($manager) {
            $group = ['name' => 'chat', 'as' => 'Chat'];

            $manager->group($group, function ($manager) {
                $manager->view('access', [
                    'as' => 'Acesso ao Chat',
                    'revokeable' => true,
                    'permissions' => [
                        'access chat module' => 'Acessar módulo de Chat',
                    ],
                ]);

                $manager->view('view', [
                    'as' => 'Visualizar Conversas',
                    'permissions' => [
                        'view own conversations' => 'Ver apenas suas conversas',
                        'view all conversations' => 'Ver todas as conversas',
                        'view team conversations' => 'Ver conversas da equipe',
                        'view unassigned conversations' => 'Ver conversas não atribuídas',
                    ],
                ]);

                $manager->view('create', [
                    'as' => 'Criar Conversas',
                    'permissions' => [
                        'create conversations' => 'Criar conversas',
                        'create internal conversations' => 'Criar conversas internas',
                        'create external conversations' => 'Criar conversas externas',
                    ],
                ]);

                $manager->view('edit', [
                    'as' => 'Editar Conversas',
                    'permissions' => [
                        'edit own conversations' => 'Editar apenas suas conversas',
                        'edit all conversations' => 'Editar todas as conversas',
                        'edit team conversations' => 'Editar conversas da equipe',
                    ],
                ]);

                $manager->view('assign', [
                    'as' => 'Atribuir Conversas',
                    'permissions' => [
                        'assign own conversations' => 'Atribuir próprias conversas',
                        'assign any conversation' => 'Atribuir qualquer conversa',
                        'assign team conversations' => 'Atribuir conversas da equipe',
                    ],
                ]);

                $manager->view('messages', [
                    'as' => 'Gerenciar Mensagens',
                    'permissions' => [
                        'send messages' => 'Enviar mensagens',
                        'send internal messages' => 'Enviar mensagens internas',
                        'send external messages' => 'Enviar mensagens externas',
                        'delete own messages' => 'Deletar próprias mensagens',
                        'delete any message' => 'Deletar qualquer mensagem',
                    ],
                ]);

                $manager->view('archive', [
                    'as' => 'Arquivar Conversas',
                    'permissions' => [
                        'archive own conversations' => 'Arquivar próprias conversas',
                        'archive any conversation' => 'Arquivar qualquer conversa',
                        'view archived conversations' => 'Ver conversas arquivadas',
                        'restore archived conversations' => 'Restaurar conversas arquivadas',
                    ],
                ]);

                $manager->view('finalize', [
                    'as' => 'Finalizar Conversas',
                    'permissions' => [
                        'finalize own conversations' => 'Finalizar próprias conversas',
                        'finalize any conversation' => 'Finalizar qualquer conversa',
                        'reopen conversations' => 'Reabrir conversas finalizadas',
                    ],
                ]);

                $manager->view('delete', [
                    'as' => 'Excluir Conversas',
                    'revokeable' => true,
                    'permissions' => [
                        'delete own conversations' => 'Excluir apenas suas conversas',
                        'delete any conversation' => 'Excluir qualquer conversa',
                        'delete team conversations' => 'Excluir conversas da equipe',
                    ],
                ]);

                $manager->view('channels', [
                    'as' => 'Gerenciar Canais',
                    'permissions' => [
                        'view chat channels' => 'Ver canais de chat',
                        'create chat channels' => 'Criar canais de chat',
                        'edit chat channels' => 'Editar canais de chat',
                        'delete chat channels' => 'Deletar canais de chat',
                        'manage channel settings' => 'Gerenciar configurações de canais',
                    ],
                ]);

                $manager->view('admin', [
                    'as' => 'Administração do Chat',
                    'permissions' => [
                        'access chat settings' => 'Acessar configurações do chat',
                        'manage chat integrations' => 'Gerenciar integrações do chat',
                        'view chat analytics' => 'Ver relatórios e analytics do chat',
                        'export conversations' => 'Exportar conversas',
                        'bulk operations' => 'Operações em massa',
                        'manage chat users' => 'Gerenciar usuários do chat',
                        'view system logs' => 'Ver logs do sistema',
                    ],
                ]);
            });
        });
    }

    protected function moduleName(): string
    {
        return 'Chat';
    }

    protected function moduleNameLower(): string
    {
        return 'chat';
    }

    protected function setup(): void
    {
        \Modules\Core\Facades\Innoclapps::vite([
            'resources/js/app.js',
            'resources/js/global-chat.js',
            'resources/css/app.css'
        ], 'modules/' . $this->moduleNameLower() . '/build');
    }

    public function menu(): array
    {
        return [
            MenuItem::make('Chat', '/chat')
                ->icon('ChatBubbleLeftRight')
                ->position(25)
                ->canSee(function () {
                    if (!auth()->check()) return false;
                    
                    $user = auth()->user();
                    
                    if ($user->super_admin) return true;
                    
                    if (!$user->can('access chat module')) return false;
                    
                    return $user->can('view own conversations') ||
                           $user->can('view all conversations') ||
                           $user->can('view team conversations') ||
                           $user->can('view unassigned conversations');
                })
                // ✅ BADGE CORRIGIDO - CONTAR CONVERSAS QUE O USUÁRIO PODE VER
                ->badge(function () {
                    if (!auth()->check()) return null;
                    
                    try {
                        $user = auth()->user();
                        $userId = $user->id;
                        
                        if (!$user->super_admin && !$user->can('access chat module')) {
                            return null;
                        }
                        
                        // ✅ USAR A MESMA LÓGICA DO SCOPE forUser
                        $unreadCount = \Modules\Chat\Models\Conversation::where(function ($q) use ($userId, $user) {
                            // Suas conversas
                            $q->where('assigned_user_id', $userId)
                              ->orWhereExists(function($exists) use ($userId) {
                                  $exists->select(\DB::raw(1))
                                         ->from('conversation_assignments')
                                         ->whereColumn('conversations.id', 'conversation_assignments.conversation_id')
                                         ->where('conversation_assignments.user_id', $userId);
                              });
                              
                            // ✅ SÓ ADICIONAR PARTICIPANTS SE A TABELA EXISTIR
                            if (\DB::getSchemaBuilder()->hasTable('conversation_participants')) {
                                $q->orWhereExists(function($exists) use ($userId) {
                                    $exists->select(\DB::raw(1))
                                           ->from('conversation_participants')
                                           ->whereColumn('conversations.id', 'conversation_participants.conversation_id')
                                           ->where('conversation_participants.user_id', $userId)
                                           ->where('conversation_participants.is_active', true);
                                });
                            }
                              
                            // ✅ CONVERSAS NÃO ATRIBUÍDAS (se tiver permissão)
                            if ($user->super_admin || $user->can('view unassigned conversations')) {
                                $q->orWhere(function($unassignedQ) {
                                    $unassignedQ->where('type', 'external')
                                                ->whereNull('assigned_user_id')
                                                ->whereNotExists(function($notExists) {
                                                    $notExists->select(\DB::raw(1))
                                                             ->from('conversation_assignments')
                                                             ->whereColumn('conversations.id', 'conversation_assignments.conversation_id');
                                                });
                                });
                            }
                        })
                        ->where('status', '!=', 'archived')
                        ->whereNull('deleted_at')
                        ->where('unread_count', '>', 0) // ✅ APENAS CONVERSAS COM MENSAGENS NÃO LIDAS
                        ->count();
                        
                        return $unreadCount > 0 ? $unreadCount : null;
                        
                    } catch (\Exception $e) {
                        \Log::error('Erro ao calcular badge do chat: ' . $e->getMessage());
                        return null;
                    }
                })
                ->badgeVariant('danger')
        ];
    }

    protected function settingsMenu(): SettingsMenuItem
    {
        return SettingsMenuItem::make('chat', 'Configurações do Chat')
            ->path('/settings/chat')
            ->icon('ChatBubbleLeftRight')
            ->order(25)
            ->canSee(function () {
                if (!auth()->check()) return false;
                
                $user = auth()->user();
                
                if ($user->super_admin) return true;
                
                return $user->can('access chat module') && 
                       $user->can('access chat settings');
            });
    }

    protected function scriptData(): Closure
    {
        return function () {
            if (!auth()->check()) return [];

            $user = auth()->user();
            
            $hasAccess = $user->super_admin || $user->can('access chat module');
            
            if (!$hasAccess) {
                return [
                    'chat' => [
                        'enabled' => false,
                        'access_denied' => true,
                        'message' => 'Você não tem permissão para acessar o módulo de Chat'
                    ]
                ];
            }
            
            return [
                'chat' => [
                    'user_id' => $user->id,
                    'user_name' => $user->name,
                    'user_email' => $user->email,
                    'is_super_admin' => $user->super_admin ?? false,
                    'websocket_url' => config('chat.websocket_url', request()->getSchemeAndHttpHost() . ':9001'),
                    'enabled' => config('chat.enabled', true),
                    'notifications_enabled' => config('chat.notifications', true),
                    'sound_enabled' => config('chat.sound', true),
                    'debug' => config('app.debug', false),
                    'csrf_token' => csrf_token(),
                    'access_denied' => false,
                    
                    'user' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'avatar' => $user->avatar,
                        'avatar_url' => $user->avatar_url,
                        'super_admin' => $user->super_admin ?? false,
                        'timezone' => $user->timezone,
                        'locale' => $user->locale
                    ],

                    'permissions' => [
                        'access_chat_module' => $user->can('access chat module'),
                        'view_all_conversations' => $user->can('view all conversations'),
                        'view_own_conversations' => $user->can('view own conversations'),
                        'view_team_conversations' => $user->can('view team conversations'),
                        'view_unassigned_conversations' => $user->can('view unassigned conversations'),
                        'create_conversations' => $user->can('create conversations'),
                        'create_internal_conversations' => $user->can('create internal conversations'),
                        'create_external_conversations' => $user->can('create external conversations'),
                        'edit_all_conversations' => $user->can('edit all conversations'),
                        'edit_own_conversations' => $user->can('edit own conversations'),
                        'assign_conversations' => $user->can('assign any conversation'),
                        'archive_conversations' => $user->can('archive any conversation'),
                        'finalize_conversations' => $user->can('finalize any conversation'),
                        'delete_conversations' => $user->can('delete any conversation'),
                        'send_messages' => $user->can('send messages'),
                        'send_internal_messages' => $user->can('send internal messages'),
                        'send_external_messages' => $user->can('send external messages'),
                        'view_chat_channels' => $user->can('view chat channels'),
                        'create_chat_channels' => $user->can('create chat channels'),
                        'edit_chat_channels' => $user->can('edit chat channels'),
                        'delete_chat_channels' => $user->can('delete chat channels'),
                        'manage_channels' => $user->can('edit chat channels'),
                        'access_settings' => $user->can('access chat settings'),
                        'export_conversations' => $user->can('export conversations'),
                        'bulk_operations' => $user->can('bulk operations'),
                        'view_analytics' => $user->can('view chat analytics'),
                    ]
                ]
            ];
        };
    }
}