<?php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Http;
use Modules\Contacts\Models\Contact;
use Modules\Chat\Models\EvolutionInstance;
use Exception;

class ContactProfilePhotoService
{
    protected EvolutionApiService $evolutionApi;

    public function __construct(EvolutionApiService $evolutionApi)
    {
        $this->evolutionApi = $evolutionApi;
    }

    public function fetchAndSaveContactProfilePhoto(Contact $contact, string $phoneNumber, EvolutionInstance $instance): bool
    {
        try {
            if (!$contact->needsWhatsappProfilePhotoUpdate()) {
                return $contact->hasWhatsappProfilePhoto();
            }

            Log::info('📸 Buscando foto de perfil', [
                'contact_id' => $contact->id,
                'phone' => $phoneNumber
            ]);

            $photoUrl = $this->fetchProfilePictureUrl($phoneNumber, $instance);
            
            if (!$photoUrl) {
                $this->markAsChecked($contact, $phoneNumber, false);
                return false;
            }

            $photoPath = $this->downloadPhoto($photoUrl, $contact);

            if ($photoPath) {
                $this->updateContactPhoto($contact, $photoUrl, $photoPath, $phoneNumber);
                return true;
            }

            return false;
        } catch (Exception $e) {
            Log::error('❌ Erro ao buscar foto', [
                'contact_id' => $contact->id,
                'phone' => $phoneNumber,
                'error' => $e->getMessage()
            ]);
            
            $this->markAsChecked($contact, $phoneNumber, false, $e->getMessage());
            return false;
        }
    }

    private function fetchProfilePictureUrl(string $phoneNumber, EvolutionInstance $instance): ?string
    {
        $variations = $this->getPhoneVariations($phoneNumber);
        
        Log::debug('🔍 Testando variações', [
            'original' => $phoneNumber,
            'variations' => $variations
        ]);
        
        foreach ($variations as $phone) {
            $url = $this->tryFetchPhoto($phone, $instance);
            if ($url) {
                Log::info('✅ Foto encontrada', [
                    'phone_used' => $phone,
                    'original' => $phoneNumber
                ]);
                return $url;
            }
            usleep(300000);
        }
        
        Log::debug('📸 Nenhuma foto encontrada', [
            'phone' => $phoneNumber,
            'variations_tried' => count($variations)
        ]);
        
        return null;
    }

 private function tryFetchPhoto(string $phoneNumber, EvolutionInstance $instance): ?string
{
    try {
        $url = rtrim($instance->api_url, '/') . '/chat/fetchProfilePictureUrl/' . $instance->instance_name;
        
        Log::debug('📡 Buscando foto via API', [
            'phone' => $phoneNumber,
            'url' => $url
        ]);
        
        $response = Http::withHeaders([
            'apikey' => $instance->api_key,
            'Content-Type' => 'application/json'
        ])->timeout(15)->post($url, ['number' => $phoneNumber]);

        if ($response->successful()) {
            $data = $response->json();
            
            // ✅ LOG COMPLETO DA RESPOSTA
            Log::debug('📸 Resposta completa da API', [
                'phone' => $phoneNumber,
                'response_data' => $data,
                'profilePictureUrl_value' => $data['profilePictureUrl'] ?? 'NOT_SET',
                'profilePictureUrl_empty' => empty($data['profilePictureUrl'] ?? ''),
                'wuid' => $data['wuid'] ?? 'NOT_SET'
            ]);
            
            if (!empty($data['profilePictureUrl']) && 
                filter_var($data['profilePictureUrl'], FILTER_VALIDATE_URL)) {
                
                Log::debug('✅ URL válida encontrada', [
                    'phone' => $phoneNumber,
                    'url_preview' => substr($data['profilePictureUrl'], 0, 50) . '...'
                ]);
                
                return $data['profilePictureUrl'];
            }
            
            Log::debug('📸 Resposta sem foto válida', [
                'phone' => $phoneNumber,
                'response_keys' => array_keys($data),
                'has_profilePictureUrl' => isset($data['profilePictureUrl']),
                'profilePictureUrl_content' => $data['profilePictureUrl'] ?? 'NULL'
            ]);
        } else {
            Log::debug('⚠️ Resposta não bem-sucedida', [
                'phone' => $phoneNumber,
                'status' => $response->status(),
                'body' => $response->body()
            ]);
        }
        
        return null;
    } catch (Exception $e) {
        Log::debug('⚠️ Erro na requisição', [
            'phone' => $phoneNumber,
            'error' => $e->getMessage()
        ]);
        return null;
    }
}

    private function getPhoneVariations(string $phoneNumber): array
    {
        $clean = preg_replace('/[^0-9]/', '', $phoneNumber);
        $variations = [];
        
        // 1. Número original
        $variations[] = $clean;
        
        // 2. Com/sem código 55
        if (str_starts_with($clean, '55') && strlen($clean) >= 12) {
            $variations[] = substr($clean, 2);
        } elseif (!str_starts_with($clean, '55') && strlen($clean) >= 10) {
            $variations[] = '55' . $clean;
        }
        
        // 3. Variação com 9º dígito
        if (strlen($clean) >= 10) {
            $hasCountry = str_starts_with($clean, '55');
            $area = substr($clean, $hasCountry ? 2 : 0, 2);
            $number = substr($clean, $hasCountry ? 4 : 2);
            
            if (strlen($number) == 8 && in_array($number[0], ['6', '7', '8', '9'])) {
                $with9 = ($hasCountry ? '55' : '') . $area . '9' . $number;
                $variations[] = $with9;
            }
        }
        
        return array_unique(array_filter($variations, fn($v) => strlen($v) >= 10));
    }

    private function downloadPhoto(string $photoUrl, Contact $contact): ?string
    {
        try {
            $response = Http::withHeaders([
                'User-Agent' => 'WhatsApp/2.23.24.76',
                'Accept' => 'image/*'
            ])->timeout(30)->get($photoUrl);
            
            if (!$response->successful()) {
                throw new Exception("HTTP {$response->status()}");
            }

            $imageData = $response->body();
            
            if (strlen($imageData) < 500) {
                throw new Exception('Imagem muito pequena');
            }

            $header = bin2hex(substr($imageData, 0, 4));
            $extension = 'jpg';
            
            if (str_starts_with($header, 'ffd8')) $extension = 'jpg';
            elseif (str_starts_with($header, '8950')) $extension = 'png';
            elseif (str_starts_with($header, '5249')) $extension = 'webp';

            $fileName = "contact_profile_{$contact->id}_" . time() . ".{$extension}";
            $filePath = "contact-profile-photos/{$fileName}";

            Storage::disk('public')->makeDirectory('contact-profile-photos');
            
            if ($contact->whatsapp_profile_photo_path && 
                Storage::disk('public')->exists($contact->whatsapp_profile_photo_path)) {
                Storage::disk('public')->delete($contact->whatsapp_profile_photo_path);
            }

            if (Storage::disk('public')->put($filePath, $imageData)) {
                Log::info('✅ Foto salva', [
                    'contact_id' => $contact->id,
                    'path' => $filePath,
                    'size' => strlen($imageData)
                ]);
                return $filePath;
            }

            return null;
        } catch (Exception $e) {
            Log::error('❌ Erro ao baixar foto', [
                'contact_id' => $contact->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    private function updateContactPhoto(Contact $contact, string $photoUrl, string $photoPath, string $phoneNumber): void
    {
        $contact->update([
            'whatsapp_profile_photo_url' => $photoUrl,
            'whatsapp_profile_photo_path' => $photoPath,
            'whatsapp_profile_photo_updated_at' => now(),
            'whatsapp_profile_metadata' => [
                'last_check' => now(),
                'has_photo' => true,
                'phone_checked' => $phoneNumber,
                'file_size' => Storage::disk('public')->size($photoPath) ?? 0
            ]
        ]);
    }

    private function markAsChecked(Contact $contact, string $phoneNumber, bool $hasPhoto, ?string $error = null): void
    {
        $metadata = [
            'last_check' => now(),
            'has_photo' => $hasPhoto,
            'phone_checked' => $phoneNumber
        ];
        
        if ($error) {
            $metadata['error'] = $error;
        }

        $contact->update([
            'whatsapp_profile_photo_updated_at' => now(),
            'whatsapp_profile_metadata' => $metadata
        ]);
    }

    public function testProfilePicture(string $phoneNumber, EvolutionInstance $instance): array
    {
        $variations = $this->getPhoneVariations($phoneNumber);
        $results = [];
        
        foreach ($variations as $phone) {
            $url = $this->tryFetchPhoto($phone, $instance);
            $results[] = [
                'phone' => $phone,
                'success' => $url !== null,
                'url_preview' => $url ? substr($url, 0, 50) . '...' : null
            ];
            
            if ($url) break;
        }
        
        return [
            'original_phone' => $phoneNumber,
            'results' => $results,
            'success' => !empty(array_filter($results, fn($r) => $r['success']))
        ];
    }
}