<?php

namespace Modules\Chat\Services;

use Modules\Chat\Models\Conversation;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\ChatChannel;
use Modules\Contacts\Models\Contact;
use Modules\Users\Models\User;
use Modules\Chat\Http\Traits\FormatsConversations;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Modules\Chat\Models\EvolutionInstance;
use Exception;

class ConversationService
{
    use FormatsConversations;

    protected ChatRedisService $chatRedisService;

    public function __construct(ChatRedisService $chatRedisService)
    {
        $this->chatRedisService = $chatRedisService;
    }

public function getConversationsForUser(
    int $userId, 
    array $filters, 
    array $pagination, 
    bool $showMyOnly = false, 
    bool $showAllConversations = false, 
    bool $showMineAndUnassigned = false,
    bool $isAdmin = false
): array {
    try {
        $query = Conversation::with([
            'lastMessage',
            'contact',
            'assignedUser',
            'participants',
            'assignedUsers',
        ]);

        // ✅ EXTRAIR PARÂMETROS ESPECÍFICOS DOS FILTROS
        $showUnassignedOnly = $filters['show_unassigned_only'] ?? false;
        
        // ✅ LOG DETALHADO DOS PARÂMETROS
        Log::info('🔍 Parâmetros de filtro recebidos:', [
            'user_id' => $userId,
            'show_my_only' => $showMyOnly,
            'show_all_conversations' => $showAllConversations,
            'show_mine_and_unassigned' => $showMineAndUnassigned,
            'show_unassigned_only' => $showUnassignedOnly,
            'is_admin' => $isAdmin,
            'filters' => $filters
        ]);

        // ✅ APLICAR FILTROS NA ORDEM CORRETA
        if ($showUnassignedOnly) {
            // 🎯 FILTRO "FILA": APENAS conversas não atribuídas
            $query->where('type', 'external')
                  ->whereNull('assigned_user_id')
                  ->whereNotExists(function($exists) {
                      $exists->select(DB::raw(1))
                             ->from('conversation_assignments')
                             ->whereColumn('conversations.id', 'conversation_assignments.conversation_id');
                  });
                  
            Log::info('🎯 Aplicado filtro FILA (apenas não atribuídas)');
            
        } elseif ($showMyOnly) {
            // 🎯 FILTRO "MINHAS": APENAS conversas atribuídas ao usuário
            $query->where(function ($q) use ($userId) {
                $q->where('assigned_user_id', $userId)
                  ->orWhereExists(function($exists) use ($userId) {
                      $exists->select(DB::raw(1))
                             ->from('conversation_assignments')
                             ->whereColumn('conversations.id', 'conversation_assignments.conversation_id')
                             ->where('conversation_assignments.user_id', $userId);
                  })
                  ->orWhereExists(function($exists) use ($userId) {
                      $exists->select(DB::raw(1))
                             ->from('conversation_participants')
                             ->whereColumn('conversations.id', 'conversation_participants.conversation_id')
                             ->where('conversation_participants.user_id', $userId)
                             ->where('conversation_participants.is_active', true);
                  });
            });
            
            Log::info('🎯 Aplicado filtro MINHAS (apenas atribuídas ao usuário)');
            
        } elseif ($showAllConversations && $isAdmin) {
            // 🎯 FILTRO "TODAS": Todas as conversas (admin)
            // Sem filtros adicionais
            Log::info('🎯 Aplicado filtro TODAS (admin - sem restrições)');
            
        } elseif ($showMineAndUnassigned) {
            // 🎯 FILTRO "PADRÃO": Suas conversas + não atribuídas
            $query->where(function ($q) use ($userId) {
                // Suas conversas
                $q->where('assigned_user_id', $userId)
                  ->orWhereExists(function($exists) use ($userId) {
                      $exists->select(DB::raw(1))
                             ->from('conversation_assignments')
                             ->whereColumn('conversations.id', 'conversation_assignments.conversation_id')
                             ->where('conversation_assignments.user_id', $userId);
                  })
                  ->orWhereExists(function($exists) use ($userId) {
                      $exists->select(DB::raw(1))
                             ->from('conversation_participants')
                             ->whereColumn('conversations.id', 'conversation_participants.conversation_id')
                             ->where('conversation_participants.user_id', $userId)
                             ->where('conversation_participants.is_active', true);
                  })
                  // OU conversas não atribuídas
                  ->orWhere(function($unassignedQ) {
                      $unassignedQ->where('type', 'external')
                                  ->whereNull('assigned_user_id')
                                  ->whereNotExists(function($notExists) {
                                      $notExists->select(DB::raw(1))
                                               ->from('conversation_assignments')
                                               ->whereColumn('conversations.id', 'conversation_assignments.conversation_id');
                                  });
                  });
            });
            
            Log::info('�� Aplicado filtro PADRÃO (minhas + não atribuídas)');
            
        } else {
            // ✅ FALLBACK: usar scope padrão
            $query->forUser($userId, false, false);
            Log::info('�� Aplicado filtro FALLBACK (scope forUser)');
        }

        // ✅ APLICAR FILTROS ADICIONAIS
        $this->applyAdditionalFilters($query, $filters, $userId);
        
        // ✅ EXCLUIR ARQUIVADAS/FECHADAS POR PADRÃO
        if (!isset($filters['include_archived']) || !$filters['include_archived']) {
            $query->whereNotIn('status', ['archived', 'closed']);
        }

        // ✅ ORDENAÇÃO
        $query->orderByDesc('is_pinned')
              ->orderByDesc('last_message_at')
              ->orderByDesc('created_at');

        // ✅ DEBUG: Log da query final
        Log::info('🔍 Query SQL final:', [
            'sql' => $query->toSql(),
            'bindings' => $query->getBindings()
        ]);

        // ✅ PAGINAÇÃO
        $perPage = min($pagination['per_page'] ?? 20, 100);
        $page = max($pagination['page'] ?? 1, 1);

        $conversations = $query->paginate($perPage, ['*'], 'page', $page);

        // ✅ LOG DOS RESULTADOS
        Log::info('✅ Conversas carregadas - RESULTADO FINAL:', [
            'user_id' => $userId,
            'total' => $conversations->total(),
            'current_page_count' => $conversations->count(),
            'filter_applied' => $showUnassignedOnly ? 'fila' : ($showMyOnly ? 'minhas' : ($showAllConversations ? 'todas' : 'default')),
            'conversation_ids' => $conversations->pluck('id')->toArray(),
            'assigned_conversations' => $conversations->where('assigned_user_id', '!=', null)->pluck('id')->toArray(),
            'unassigned_conversations' => $conversations->where('assigned_user_id', null)->pluck('id')->toArray()
        ]);

        return ['conversations' => $conversations];

    } catch (Exception $e) {
        Log::error('❌ Erro ao carregar conversas para usuário', [
            'user_id' => $userId,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        throw $e;
    }
}

// ✅ MÉTODO AUXILIAR PARA FILTROS ADICIONAIS
private function applyAdditionalFilters($query, array $filters, int $userId): void
{
    // Busca por texto
    if (!empty($filters['search'])) {
        $search = trim($filters['search']);
        $query->where(function ($q) use ($search) {
            $q->whereHas('contact', function ($contactQuery) use ($search) {
                $contactQuery->where('display_name', 'like', "%{$search}%")
                             ->orWhere('email', 'like', "%{$search}%")
                             ->orWhere('phone', 'like', "%{$search}%")
                             ->orWhere('company_name', 'like', "%{$search}%");
            })
            ->orWhere('whatsapp_profile_name', 'like', "%{$search}%")
            ->orWhere('contact_phone', 'like', "%{$search}%");
        });
    }

    // Filtro por status específico
    if (!empty($filters['status'])) {
        $query->where('status', $filters['status']);
    }

    // Filtro por tipo
    if (!empty($filters['type'])) {
        $query->where('type', $filters['type']);
    }

    // Apenas não lidas
    if (isset($filters['only_unread']) && $filters['only_unread']) {
        $query->where(function ($q) use ($userId) {
            $q->where('unread_count', '>', 0)
              ->orWhereHas('messages', function ($msgQuery) use ($userId) {
                  $msgQuery->where('sender_id', '!=', $userId)
                           ->whereNull('read_at');
              });
        });
    }
}

  public function createConversation(array $data): array
{
    try {
        DB::beginTransaction();

        if ($data['type'] === 'external') {
            $result = $this->createExternalConversation($data);
        } else {
            $result = $this->createInternalConversation($data);
        }

        DB::commit();
        return $result;

    } catch (Exception $e) {
        DB::rollBack();
        Log::error('Erro ao criar conversa', [
            'type' => $data['type'],
            'error' => $e->getMessage()
        ]);
        throw $e;
    }
}

// Em ConversationService.php - CORRIGIR O MÉTODO createExternalConversation:

private function createExternalConversation(array $data): array
{
    try {
        // ✅ BUSCAR CONTATO E CANAL
        $contact = Contact::find($data['contact_id']);
        if (!$contact) {
            return [
                'success' => false,
                'error' => 'Contato não encontrado',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Contato não encontrado.'
                ]
            ];
        }

        $chatChannel = ChatChannel::find($data['channel_id']);
        if (!$chatChannel) {
            return [
                'success' => false,
                'error' => 'Canal não encontrado',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Canal de chat não encontrado.'
                ]
            ];
        }

        // ✅ VERIFICAR SE JÁ EXISTE CONVERSA ATIVA PARA ESTE CONTATO
        $existingConversation = Conversation::where('contact_id', $contact->id)
            ->where('chat_channel_id', $chatChannel->id)
            ->where('status', 'active')
            ->first();

        if ($existingConversation) {
            return [
                'success' => false,
                'error' => 'Conversa ativa já existe',
                'conversation_id' => $existingConversation->id,
                'notification' => [
                    'type' => 'warning',
                    'title' => 'Aviso',
                    'message' => 'Já existe uma conversa ativa com este contato.'
                ]
            ];
        }

        // ✅ OBTER INSTÂNCIA EVOLUTION SE FOR WHATSAPP - MÉTODO CORRIGIDO
        $evolutionInstance = null;
        if ($chatChannel->type === 'evolution_api') {
            // ✅ BUSCAR POR NOME DA INSTÂNCIA NO CONFIG DO CANAL
            $instanceName = $chatChannel->getConfig('instance_name');
            
            if ($instanceName) {
                $evolutionInstance = EvolutionInstance::where('instance_name', $instanceName)
                    ->where('status', 'open')
                    ->first();
            }
            
            // ✅ SE NÃO ENCONTROU POR NOME, BUSCAR QUALQUER INSTÂNCIA ATIVA
            if (!$evolutionInstance) {
                $evolutionInstance = EvolutionInstance::where('status', 'open')
                    ->first();
            }

            // ✅ LOG PARA DEBUG
            Log::info('Buscando instância Evolution', [
                'channel_id' => $chatChannel->id,
                'channel_config' => $chatChannel->config,
                'instance_name' => $instanceName,
                'found_instance' => $evolutionInstance ? $evolutionInstance->instance_name : 'none'
            ]);
        }

        // ✅ PREPARAR DADOS DA CONVERSA
        $conversationData = [
            'contact_id' => $contact->id,
            'chat_channel_id' => $chatChannel->id,
            'type' => 'external',
            'status' => 'active',
            'assigned_user_id' => $data['auto_assign'] ? auth()->id() : null,
            'last_message_at' => now(),
            'metadata' => [
                'priority' => $data['priority'] ?? 'normal',
                'tags' => $data['tags'] ?? [],
                'notifications_enabled' => $data['notifications_enabled'] ?? true,
                'created_by' => auth()->id(),
                'created_via' => 'manual'
            ]
        ];

        // ✅ DADOS ESPECÍFICOS DO WHATSAPP
        if ($evolutionInstance) {
            // Obter número do WhatsApp do contato
            $whatsappNumber = $contact->whatsapp_number ?? $contact->phones()->first()?->number;
            
            if ($whatsappNumber) {
                $cleanNumber = preg_replace('/[^0-9]/', '', $whatsappNumber);
                
                // Garantir formato brasileiro
                if (strlen($cleanNumber) === 11 && !str_starts_with($cleanNumber, '55')) {
                    $cleanNumber = '55' . $cleanNumber;
                }
                
                $conversationData = array_merge($conversationData, [
                    'evolution_instance_name' => $evolutionInstance->instance_name,
                    'whatsapp_remote_jid' => $cleanNumber . '@s.whatsapp.net',
                    'whatsapp_chat_id' => $cleanNumber,
                    'contact_phone' => $whatsappNumber,
                    'whatsapp_profile_name' => $contact->display_name ?? $contact->first_name,
                ]);
            }
        } else {
            // ✅ SE NÃO TEM INSTÂNCIA EVOLUTION, USAR DADOS BÁSICOS
            $whatsappNumber = $contact->whatsapp_number ?? $contact->phones()->first()?->number;
            if ($whatsappNumber) {
                $conversationData['contact_phone'] = $whatsappNumber;
                $conversationData['whatsapp_profile_name'] = $contact->display_name ?? $contact->first_name;
            }
        }

        Log::info('Criando conversa externa', [
            'conversation_data' => $conversationData,
            'evolution_instance' => $evolutionInstance ? $evolutionInstance->instance_name : 'none'
        ]);

        // ✅ CRIAR CONVERSA
        $conversation = Conversation::create($conversationData);

        // ✅ CRIAR MENSAGEM INICIAL SE FORNECIDA
        if (!empty($data['initial_message'])) {
            $this->createInitialMessage($conversation, $data['initial_message']);
        }

        // ✅ PUBLICAR EVENTO
        try {
            $this->chatRedisService->publishConversationUpdate($conversation->fresh(), 'conversation_created');
        } catch (Exception $e) {
            Log::warning('Erro ao publicar evento', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
        }

        return [
            'success' => true,
            'conversation' => $this->formatConversation($conversation->fresh()->load([
                'contact', 
                'assignedUser', 
                'chatChannel'
            ]), auth()->id()),
            'message' => 'Conversa criada com sucesso.',
            'notification' => [
                'type' => 'success',
                'title' => 'Sucesso',
                'message' => 'Conversa criada com sucesso.'
            ]
        ];

    } catch (Exception $e) {
        Log::error('Erro ao criar conversa externa', [
            'contact_id' => $data['contact_id'] ?? null,
            'channel_id' => $data['channel_id'] ?? null,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        throw $e;
    }
}

private function createInternalConversation(array $data): array
{
    try {
        // ✅ VALIDAR PARTICIPANTES
        if (empty($data['participants']) || !is_array($data['participants'])) {
            return [
                'success' => false,
                'error' => 'Participantes são obrigatórios',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'É necessário informar pelo menos um participante.'
                ]
            ];
        }

        // ✅ VERIFICAR SE USUÁRIOS EXISTEM
        $participants = User::whereIn('id', $data['participants'])->get();
        if ($participants->count() !== count($data['participants'])) {
            return [
                'success' => false,
                'error' => 'Alguns participantes não foram encontrados',
                'notification' => [
                    'type' => 'error',
                    'title' => 'Erro',
                    'message' => 'Alguns participantes não foram encontrados.'
                ]
            ];
        }

        // ✅ ADICIONAR CRIADOR AOS PARTICIPANTES SE NÃO ESTIVER
        $creatorId = auth()->id();
        $participantIds = $data['participants'];
        if (!in_array($creatorId, $participantIds)) {
            $participantIds[] = $creatorId;
        }

        // ✅ BUSCAR OU CRIAR CANAL INTERNO PADRÃO
        $chatChannel = null;
        
        // Primeiro, tentar usar canal fornecido
        if (!empty($data['channel_id'])) {
            $chatChannel = ChatChannel::where('id', $data['channel_id'])
                ->where('type', 'internal_chat')
                ->where('is_active', true)
                ->first();
        }
        
        // Se não encontrou, buscar canal interno padrão
        if (!$chatChannel) {
            $chatChannel = ChatChannel::where('type', 'internal_chat')
                ->where('is_active', true)
                ->first();
        }
        
        // Se ainda não encontrou, criar canal interno padrão
        if (!$chatChannel) {
            $chatChannel = ChatChannel::create([
                'name' => 'Chat Interno Padrão',
                'type' => 'internal_chat',
                'is_active' => true,
                'config' => [
                    'auto_created' => true,
                    'description' => 'Canal padrão para conversas internas'
                ],
                'created_by' => $creatorId
            ]);
            
            Log::info('Canal interno padrão criado automaticamente', [
                'channel_id' => $chatChannel->id,
                'created_by' => $creatorId
            ]);
        }

        // ✅ PREPARAR DADOS DA CONVERSA COM CANAL OBRIGATÓRIO
        $conversationData = [
            'type' => 'internal',
            'status' => 'active',
            'assigned_user_id' => $creatorId,
            'chat_channel_id' => $chatChannel->id, // ✅ CAMPO OBRIGATÓRIO
            'last_message_at' => now(),
            'metadata' => [
                'subject' => $data['subject'],
                'description' => $data['description'] ?? null,
                'internal_type' => $data['internal_type'] ?? 'discussion',
                'priority' => $data['priority'] ?? 'normal',
                'participants' => $participantIds,
                'created_by' => $creatorId,
                'created_via' => 'manual'
            ]
        ];

        // ✅ CRIAR CONVERSA
        $conversation = Conversation::create($conversationData);

        // ✅ ADICIONAR PARTICIPANTES
        foreach ($participantIds as $index => $userId) {
            $conversation->participants()->attach($userId, [
                'role' => $userId === $creatorId ? 'admin' : 'participant',
                'is_active' => true,
                'joined_at' => now()
            ]);
        }

        // ✅ CRIAR MENSAGEM INICIAL SE FORNECIDA
        if (!empty($data['initial_message'])) {
            $this->createInitialMessage($conversation, $data['initial_message']);
        } else {
            // Mensagem de sistema padrão
            $this->createSystemMessage($conversation, 'conversation_created', 'Conversa interna criada.');
        }

        // ✅ PUBLICAR EVENTO
        try {
            $this->chatRedisService->publishConversationUpdate($conversation->fresh(), 'conversation_created');
        } catch (Exception $e) {
            // Silenciar erro de publicação
        }

        return [
            'success' => true,
            'conversation' => $this->formatConversation($conversation->fresh()->load([
                'participants', 
                'assignedUser',
                'chatChannel' // ✅ INCLUIR CANAL
            ]), auth()->id()),
            'message' => 'Conversa interna criada com sucesso.',
            'notification' => [
                'type' => 'success',
                'title' => 'Sucesso',
                'message' => 'Conversa interna criada com sucesso.'
            ]
        ];

    } catch (Exception $e) {
        Log::error('Erro ao criar conversa interna', [
            'subject' => $data['subject'] ?? null,
            'participants' => $data['participants'] ?? null,
            'error' => $e->getMessage()
        ]);
        throw $e;
    }
}

private function createInitialMessage(Conversation $conversation, string $content): void
{
    try {
        Message::create([
            'conversation_id' => $conversation->id,
            'sender_type' => User::class,
            'sender_id' => auth()->id(),
            'content' => $content,
            'message_type' => 'text',
            'is_from_contact' => false,
            'status' => 'sent',
            'sent_at' => now(),
            'metadata' => [
                'is_initial_message' => true,
                'created_via' => 'manual'
            ]
        ]);

        $conversation->update(['last_message_at' => now()]);
    } catch (Exception $e) {
        Log::error('Erro ao criar mensagem inicial', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

private function createSystemMessage(Conversation $conversation, string $systemType, string $content, array $metadata = []): void
{
    try {
        Message::create([
            'conversation_id' => $conversation->id,
            'sender_type' => 'system',
            'sender_id' => null,
            'content' => $content,
            'message_type' => 'system',
            'is_from_contact' => false,
            'status' => 'sent',
            'sent_at' => now(),
            'metadata' => array_merge([
                'is_system_message' => true,
                'system_type' => $systemType
            ], $metadata)
        ]);

        $conversation->update(['last_message_at' => now()]);
    } catch (Exception $e) {
        Log::error('Erro ao criar mensagem de sistema', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

    public function updateConversation(Conversation $conversation, array $data): array
    {
        try {
            $updated = false;
            $dataToUpdate = [];

            // Campos simples
            foreach (['is_pinned', 'is_muted', 'assigned_user_id'] as $field) {
                if (array_key_exists($field, $data)) {
                    $dataToUpdate[$field] = $data[$field];
                    $updated = true;
                }
            }

            // Campos de metadata para conversas internas
            if ($conversation->type === 'internal') {
                $metadata = $conversation->metadata ?? [];
                
                foreach (['subject', 'description', 'priority'] as $field) {
                    if (array_key_exists($field, $data)) {
                        $metadata[$field] = $data[$field];
                        $updated = true;
                    }
                }

                if ($updated && !empty($metadata)) {
                    $dataToUpdate['metadata'] = $metadata;
                }
            }

            if ($updated) {
                $conversation->update($dataToUpdate);
                
                // Publicar atualização
                $this->chatRedisService->publishConversationUpdate(
                    $conversation->fresh(), 
                    'conversation_updated'
                );
            }

            $message = $this->getUpdateMessage($data);

            return [
                'success' => true,
                'conversation' => $this->formatConversation($conversation->fresh(), auth()->id()),
                'message' => $message,
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => $message
                ]
            ];

        } catch (Exception $e) {
            Log::error('Erro ao atualizar conversa', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

public function updateConversationStatus(Conversation $conversation, string $newStatus, ?string $reason = null, array $extraData = []): array
{
    try {
        DB::beginTransaction();
        
        $oldStatus = $conversation->status;

        $updateData = ['status' => $newStatus];

        if ($newStatus === 'closed') {
            $updateData['closed_at'] = $extraData['closed_at'] ?? now();
            $updateData['closed_by'] = auth()->id();
            $updateData['close_reason'] = $reason;
            
            // ✅ PROCESSAR DADOS EXTRAS DE FINALIZAÇÃO
            if (!empty($extraData['resolution'])) {
                $metadata = $conversation->metadata ?? [];
                $metadata['resolution_notes'] = $extraData['resolution'];
                $metadata['finalized_at'] = now()->toISOString();
                $metadata['finalized_by'] = auth()->id();
                $updateData['metadata'] = $metadata;
            }
        } elseif ($newStatus === 'active') {
            // ✅ REABERTURA
            $updateData['closed_at'] = null;
            $updateData['closed_by'] = null;
            $updateData['close_reason'] = null;
            
            // ✅ PROCESSAR DADOS EXTRAS DE REABERTURA
            if (!empty($extraData['reopen_reason'])) {
                $metadata = $conversation->metadata ?? [];
                $metadata['reopen_reason'] = $extraData['reopen_reason'];
                $metadata['reopened_at'] = $extraData['reopened_at'] ?? now()->toISOString();
                $metadata['reopened_by'] = $extraData['reopened_by'] ?? auth()->id();
                $updateData['metadata'] = $metadata;
            }
        }

        $conversation->update($updateData);

        // ✅ PROCESSAR AÇÕES EXTRAS PARA FINALIZAÇÃO
        if ($newStatus === 'closed' && !empty($extraData)) {
            $this->processFinalizationActions($conversation, $extraData, auth()->id());
        }

        $this->sendSystemStatusMessage($conversation, $newStatus, $reason, auth()->id());

        // Publicar atualização
        $this->chatRedisService->publishConversationUpdate(
            $conversation->fresh(), 
            'status_updated'
        );

        DB::commit();

        $message = match ($newStatus) {
            'archived' => 'Conversa arquivada com sucesso!',
            'closed' => 'Conversa finalizada com sucesso!',
            'active' => $oldStatus === 'closed' ? 'Conversa reaberta com sucesso!' : 'Conversa ativada com sucesso!',
            default => 'Status da conversa atualizado com sucesso!'
        };

        return [
            'success' => true,
            'conversation' => $this->formatConversation($conversation->fresh(), auth()->id()),
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'message' => $message,
            'notification' => [
                'type' => 'success',
                'title' => 'Sucesso',
                'message' => $message
            ]
        ];

    } catch (Exception $e) {
        DB::rollBack();
        
        Log::error('Erro ao atualizar status da conversa', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
        throw $e;
    }
}

// ✅ ADICIONAR MÉTODO PARA PROCESSAR AÇÕES DE FINALIZAÇÃO
private function processFinalizationActions(Conversation $conversation, array $extraData, int $userId): void
{
    try {
        // Enviar mensagem de despedida se solicitado
        if (!empty($extraData['send_farewell_message']) && !empty($extraData['farewell_message'])) {
            $this->sendFarewellMessage($conversation, $extraData['farewell_message'], $userId);
        }

        // Arquivar se solicitado
        if (!empty($extraData['archive_conversation'])) {
            $conversation->update([
                'status' => 'archived',
                'archived_at' => now()
            ]);
        }

        // Bloquear contato se solicitado (apenas para spam)
        if (!empty($extraData['block_contact']) && $conversation->contact_id) {
            $this->blockContact($conversation->contact_id);
        }

        // Notificar equipe se solicitado
        if (!empty($extraData['notify_team'])) {
            $this->notifyTeamAboutClosure($conversation, $extraData['reason'] ?? '', $userId);
        }

        // Solicitar feedback se solicitado
        if (!empty($extraData['request_feedback']) && $conversation->type !== 'internal') {
            $this->requestFeedback($conversation);
        }

    } catch (Exception $e) {
        Log::error('Erro ao processar ações de finalização', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

// ✅ ADICIONAR OS MÉTODOS AUXILIARES
private function sendFarewellMessage(Conversation $conversation, string $message, int $userId): void
{
    try {
        Message::create([
            'conversation_id' => $conversation->id,
            'sender_type' => User::class,
            'sender_id' => $userId,
            'content' => $message,
            'message_type' => 'text',
            'is_from_contact' => false,
            'status' => 'sent',
            'sent_at' => now(),
            'metadata' => [
                'is_farewell_message' => true,
                'auto_generated' => true,
                'sent_by' => $userId
            ]
        ]);

        $conversation->update(['last_message_at' => now()]);

        Log::info('Mensagem de despedida enviada', [
            'conversation_id' => $conversation->id,
            'user_id' => $userId
        ]);

    } catch (Exception $e) {
        Log::error('Erro ao enviar mensagem de despedida', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

private function blockContact(int $contactId): void
{
    try {
        Contact::where('id', $contactId)->update([
            'is_blocked' => true,
            'blocked_at' => now(),
            'blocked_reason' => 'spam'
        ]);

        Log::info('Contato bloqueado por spam', ['contact_id' => $contactId]);

    } catch (Exception $e) {
        Log::error('Erro ao bloquear contato', [
            'contact_id' => $contactId,
            'error' => $e->getMessage()
        ]);
    }
}

private function notifyTeamAboutClosure(Conversation $conversation, string $reason, int $userId): void
{
    try {
        Log::info('Equipe notificada sobre fechamento', [
            'conversation_id' => $conversation->id,
            'reason' => $reason,
            'closed_by' => $userId
        ]);

    } catch (Exception $e) {
        Log::error('Erro ao notificar equipe', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

private function requestFeedback(Conversation $conversation): void
{
    try {
        if ($conversation->type !== 'internal' && $conversation->contact_id) {
            $feedbackMessage = "Obrigado por utilizar nossos serviços! 😊\n\n" .
                             "Sua opinião é muito importante para nós. " .
                             "Como você avaliaria nosso atendimento?\n\n" .
                             "⭐ Excelente\n" .
                             "👍 Bom\n" .
                             "😐 Regular\n" .
                             "👎 Ruim\n\n" .
                             "Deixe também seus comentários e sugestões!";

            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'sender_id' => null,
                'content' => $feedbackMessage,
                'message_type' => 'feedback_request',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_feedback_request' => true,
                    'auto_generated' => true,
                    'feedback_type' => 'satisfaction_survey'
                ]
            ]);

            Log::info('Solicitação de feedback enviada', [
                'conversation_id' => $conversation->id
            ]);
        }

    } catch (Exception $e) {
        Log::error('Erro ao solicitar feedback', [
            'conversation_id' => $conversation->id,
            'error' => $e->getMessage()
        ]);
    }
}

    public function assignConversation(Conversation $conversation, ?int $userId): array
    {
        try {
            $conversation->update(['assigned_user_id' => $userId]);
            $conversation->load('assignedUser');

            // Publicar atualização
            $this->chatRedisService->publishConversationUpdate(
                $conversation->fresh(), 
                'assignment_updated'
            );

            return [
                'success' => true,
                'assigned_to' => $conversation->assignedUser ? [
                    'id' => $conversation->assignedUser->id,
                    'name' => $conversation->assignedUser->name
                ] : null,
                'message' => 'Atribuição atualizada com sucesso.',
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Atribuição atualizada com sucesso.'
                ]
            ];

        } catch (Exception $e) {
            Log::error('Erro ao atribuir conversa', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function markAsRead(int $conversationId, int $userId): array
    {
        try {
            $conversation = Conversation::findOrFail($conversationId);

            $updatedCount = Message::where('conversation_id', $conversationId)
                ->where('sender_id', '!=', $userId)
                ->whereNull('read_at')
                ->update(['read_at' => now()]);

            $conversation->update(['unread_count' => 0]);

            return [
                'success' => true,
                'messages_marked' => $updatedCount,
                'conversation_id' => $conversationId,
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Conversa marcada como lida.'
                ]
            ];

        } catch (Exception $e) {
            Log::error('Erro ao marcar conversa como lida', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function markAsUnread(int $conversationId, int $userId): array
    {
        try {
            $conversation = Conversation::findOrFail($conversationId);
            $conversation->update(['unread_count' => 1]);

            Message::where('conversation_id', $conversationId)
                ->where('sender_id', '!=', $userId)
                ->whereNotNull('read_at')
                ->orderBy('created_at', 'desc')
                ->limit(1)
                ->update(['read_at' => null]);

            return [
                'success' => true,
                'unread_count' => 1,
                'conversation_id' => $conversationId,
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Conversa marcada como não lida.'
                ]
            ];

        } catch (Exception $e) {
            Log::error('Erro ao marcar conversa como não lida', [
                'conversation_id' => $conversationId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function deleteConversation(Conversation $conversation): array
    {
        try {
            DB::beginTransaction();

            $batchSize = 100;
            
            // Excluir mensagens em lotes
            do {
                $deletedMessages = Message::where('conversation_id', $conversation->id)
                    ->limit($batchSize)
                    ->delete();
            } while ($deletedMessages > 0);

            // Remover participantes e atribuições
            $conversation->participants()->detach();
            $conversation->assignedUsers()->detach();
            
            // Excluir anexos se existirem
            if (method_exists($conversation, 'attachments')) {
                $conversation->attachments()->delete();
            }
            
            $conversation->delete();

            DB::commit();

            return [
                'success' => true,
                'message' => 'Conversa excluída com sucesso',
                'notification' => [
                    'type' => 'success',
                    'title' => 'Sucesso',
                    'message' => 'Conversa excluída permanentemente.'
                ]
            ];

        } catch (Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function getStats(int $userId): array
    {
        try {
            return [
                'total_conversations' => Conversation::count(),
                'active_conversations' => Conversation::where('status', 'active')->count(),
                'pending_conversations' => Conversation::where('status', 'pending')->count(),
                'closed_conversations' => Conversation::where('status', 'closed')->count(),
                'archived_conversations' => Conversation::where('status', 'archived')->count(),
                'total_messages' => Message::count(),
                'messages_today' => Message::whereDate('sent_at', today())->count(),
                'my_active_conversations' => Conversation::where('assigned_user_id', $userId)
                    ->where('status', 'active')->count(),
            ];

        } catch (Exception $e) {
            Log::error('Erro ao obter estatísticas', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    public function getSummary(int $userId, array $filters): array
    {
        try {
            $query = Conversation::forUser($userId);

            if (!empty($filters['status'])) {
                $query->where('status', $filters['status']);
            }

            $totalConversations = $query->count();

            $unreadQuery = Message::whereHas('conversation', function ($q) use ($userId, $filters) {
                $q->forUser($userId);
                if (!empty($filters['status'])) {
                    $q->where('status', $filters['status']);
                }
            })
                ->where('sender_id', '!=', $userId)
                ->whereNull('read_at');

            $unreadCount = $unreadQuery->count();

            return [
                'unread_count' => $unreadCount,
                'total_conversations' => $totalConversations,
                'total_archived' => Conversation::where('status', 'archived')->count(),
            ];

        } catch (Exception $e) {
            Log::error('Erro ao obter resumo', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    // ===== MÉTODOS AUXILIARES =====

    private function applyFilters($query, array $filters, int $userId): void
    {
        if (!empty($filters['status'])) {
            if (is_array($filters['status'])) {
                $query->whereIn('status', $filters['status']);
            } else {
                $query->where('status', $filters['status']);
            }
        }

        if (!empty($filters['type'])) {
            $query->where('type', $filters['type']);
        }

        if (!empty($filters['assigned']) && $filters['assigned'] !== 'all') {
            if ($filters['assigned'] === 'me') {
                $query->where('assigned_user_id', $userId);
            } elseif ($filters['assigned'] === 'unassigned') {
                $query->whereNull('assigned_user_id');
            } elseif (is_numeric($filters['assigned'])) {
                $query->where('assigned_user_id', (int) $filters['assigned']);
            }
        }

        if ($filters['only_unread']) {
            $query->where(function ($q) use ($userId) {
                $q->where(function ($sub) {
                    $sub->where('type', 'external')->where('unread_count', '>', 0);
                })->orWhere(function ($sub) use ($userId) {
                    $sub->where('type', 'internal')
                        ->whereHas('messages', function ($msg) use ($userId) {
                            $msg->whereNull('read_at')->where('sender_id', '!=', $userId);
                        });
                });
            });
        }

        if (!empty($filters['search'])) {
            $like = '%' . $filters['search'] . '%';
            $query->where(function ($q) use ($like) {
                $q->where('conversations.whatsapp_profile_name', 'LIKE', $like)
                    ->orWhere('conversations.contact_phone', 'LIKE', $like)
                    ->orWhereHas('contact', function ($c) use ($like) {
                        $c->where('first_name', 'LIKE', $like)
                            ->orWhere('last_name', 'LIKE', $like)
                            ->orWhere('email', 'LIKE', $like)
                            ->orWhereHas('phones', function ($p) use ($like) {
                                $p->where('number', 'LIKE', $like);
                            })
                            ->orWhereHas('companies', function ($comp) use ($like) {
                                $comp->where('name', 'LIKE', $like);
                            });
                    });
            });
        }

        if (!$filters['include_archived']) {
            $query->where('status', '!=', 'archived');
        }
    }


    private function sendSystemStatusMessage(Conversation $conversation, string $newStatus, ?string $reason, int $userId): void
    {
        try {
            $user = User::find($userId);
            $userName = $user ? $user->name : 'Sistema';
            
            $messageContent = $newStatus === 'closed' 
                ? "{$userName} finalizou este atendimento." . ($reason ? " Motivo: {$reason}" : '')
                : "{$userName} reabriu este atendimento.";

            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'sender_id' => null,
                'content' => $messageContent,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => [
                    'is_system_message' => true,
                    'system_type' => ($newStatus === 'closed' ? 'conversation_closed' : 'conversation_reopened'),
                    'user_id' => $userId
                ]
            ]);

            $conversation->update(['last_message_at' => now()]);
        } catch (Exception $e) {
            Log::error('Erro ao enviar mensagem de sistema', [
                'conversation_id' => $conversation->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    private function getUpdateMessage(array $data): string
    {
        if (array_key_exists('is_pinned', $data)) {
            return $data['is_pinned'] ? 'Conversa fixada com sucesso.' : 'Conversa desafixada com sucesso.';
        }
        
        if (array_key_exists('is_muted', $data)) {
            return $data['is_muted'] ? 'Conversa silenciada com sucesso.' : 'Notificações reativadas com sucesso.';
        }
        
        return 'Conversa atualizada com sucesso.';
    }
}