<?php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Modules\Chat\Models\EvolutionInstance;
use Exception;

class EvolutionApiService
{
    private const DEFAULT_EVENTS = [
        'QRCODE_UPDATED',
        'MESSAGES_UPSERT', 
        'MESSAGES_UPDATE',
        'CONNECTION_UPDATE',
        'SEND_MESSAGE'
    ];

    private const AUDIO_MIME_TYPES = ['audio/', 'video/webm'];

    protected ?EvolutionInstance $instance = null;
    protected int $timeout = 30;
    protected int $retryTimes = 3;

    public function setInstance(EvolutionInstance $instance): self
    {
        $this->instance = $instance;
        return $this;
    }

    public function getInstance(): EvolutionInstance
    {
        if (!$this->instance) {
            throw new Exception('Evolution instance not set. Use setInstance() first.');
        }
        return $this->instance;
    }

    // ===== MÉTODOS DE INSTÂNCIA =====

    public function createInstanceIfNotExists(): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Criando/verificando instância na Evolution API', [
                'instance_name' => $instance->instance_name
            ]);

            // Verificar se já existe
            try {
                $existingInstances = $this->makeRequest('GET', '/instance/fetchInstances', [], 30);

                if (is_array($existingInstances)) {
                    foreach ($existingInstances as $existingInstance) {
                        $foundInstanceName = $existingInstance['instanceName'] ?? $existingInstance['name'] ?? null;
                        if ($foundInstanceName === $instance->instance_name) {
                            Log::info('Instância já existe na Evolution API', [
                                'instance_name' => $instance->instance_name,
                                'status' => $existingInstance['status'] ?? 'unknown'
                            ]);
                            return [
                                'success' => true,
                                'message' => 'Instance already exists',
                                'instance' => $existingInstance
                            ];
                        }
                    }
                }
            } catch (Exception $e) {
                Log::debug('Erro ao verificar instâncias existentes (continuando com criação)', [
                    'error' => $e->getMessage()
                ]);
            }

            // Criar nova instância
            $payload = [
                'instanceName' => $instance->instance_name,
                'integration' => 'WHATSAPP-BAILEYS',
                'qrcode' => true
            ];

            if ($instance->webhook_url) {
                $payload['webhook'] = [
                    'url' => $instance->webhook_url,
                    'byEvents' => true,
                    'base64' => true,
                    'events' => self::DEFAULT_EVENTS
                ];
            }

            $response = $this->makeRequest('POST', '/instance/create', $payload);

            Log::info('Instância criada na Evolution API', [
                'instance_name' => $instance->instance_name,
                'instance_id' => $response['instance']['instanceId'] ?? 'unknown',
                'status' => $response['instance']['status'] ?? 'unknown'
            ]);

            return $response;

        } catch (Exception $e) {
            // Tratar erros específicos de instância já existente
            $errorMessage = $e->getMessage();

            if (str_contains($errorMessage, 'already in use') ||
                str_contains($errorMessage, 'already exists') ||
                str_contains($errorMessage, '409')) {

                Log::info('Instância já existe (detectado por erro na criação)', [
                    'instance_name' => $instance->instance_name
                ]);

                try {
                    $statusResponse = $this->makeRequest('GET', "/instance/connectionState/{$instance->instance_name}");
                    return [
                        'success' => true,
                        'message' => 'Instance already exists',
                        'instance' => $statusResponse['instance'] ?? ['instanceName' => $instance->instance_name]
                    ];
                } catch (Exception $statusException) {
                    return [
                        'success' => true,
                        'message' => 'Instance already exists (status check failed)',
                        'instance' => ['instanceName' => $instance->instance_name]
                    ];
                }
            }

            Log::error('Erro ao criar instância na Evolution API', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function getConnectionStatus(): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Verificando status da instância Evolution', [
                'instance_name' => $instance->instance_name
            ]);

            $this->createInstanceIfNotExists();
            $response = $this->makeRequest('GET', "/instance/connectionState/{$instance->instance_name}");

            $instanceData = $response['instance'] ?? $response;
            $state = $instanceData['state'] ?? $instanceData['status'] ?? 'close';
            $connected = ($state === 'open');

            Log::info('Status da instância obtido', [
                'instance_name' => $instance->instance_name,
                'state' => $state,
                'connected' => $connected
            ]);

            return [
                'state' => $state,
                'connected' => $connected,
                'instance_name' => $instance->instance_name,
                'full_response' => ['instance' => $instanceData]
            ];
        } catch (Exception $e) {
            Log::error('Erro ao verificar status Evolution', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);

            return [
                'connected' => false,
                'state' => 'disconnected',
                'full_response' => ['instance' => []]
            ];
        }
    }

    public function getQrCode(): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Obtendo QR Code', [
                'instance_name' => $instance->instance_name
            ]);

            $this->createInstanceIfNotExists();
            $response = $this->makeRequest('GET', "/instance/connect/{$instance->instance_name}", [], 60);

            $qrData = [
                'qrcode' => $response['qrcode'] ?? $response['base64'] ?? null,
                'base64' => $response['base64'] ?? $response['qrcode'] ?? null,
                'generated_at' => now()->toISOString()
            ];

            Log::info('QR Code obtido', [
                'instance_name' => $instance->instance_name,
                'has_qrcode' => !empty($qrData['qrcode'])
            ]);

            return $qrData;
        } catch (Exception $e) {
            Log::error('Erro ao obter QR Code', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function connectInstance(): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Conectando instância', [
                'instance_name' => $instance->instance_name
            ]);

            $this->createInstanceIfNotExists();
            $response = $this->makeRequest('GET', "/instance/connect/{$instance->instance_name}");

            Log::info('Comando de conexão enviado', [
                'instance_name' => $instance->instance_name
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao conectar instância', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function disconnectInstance(): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Desconectando instância', [
                'instance_name' => $instance->instance_name
            ]);

            $response = $this->makeRequest('DELETE', "/instance/logout/{$instance->instance_name}");

            Log::info('Instância desconectada', [
                'instance_name' => $instance->instance_name
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao desconectar instância', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function setupWebhook(?array $events = null): array
    {
        $instance = $this->getInstance();
        
        try {
            $events = $events ?: self::DEFAULT_EVENTS;
            $webhookUrl = $instance->webhook_url ?? $instance->getWebhookEndpoint();

            Log::info('Configurando webhook', [
                'instance_name' => $instance->instance_name,
                'webhook_url' => $webhookUrl,
                'events' => $events
            ]);

            $this->createInstanceIfNotExists();

            $payload = [
                'url' => $webhookUrl,
                'byEvents' => true,
                'base64' => true,
                'events' => $events
            ];

            $response = $this->makeRequest('POST', "/webhook/set/{$instance->instance_name}", $payload);

            Log::info('Webhook configurado', [
                'instance_name' => $instance->instance_name,
                'events' => $events
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao configurar webhook', [
                'instance_name' => $instance->instance_name,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function testConnection(): array
    {
        $instance = $this->getInstance();
        $startTime = microtime(true);

        try {
            $this->createInstanceIfNotExists();
            $response = $this->makeRequest('GET', "/instance/connectionState/{$instance->instance_name}", [], 10);
            $responseTime = round((microtime(true) - $startTime) * 1000, 2);

            return [
                'success' => true,
                'connected' => ($response['instance']['status'] ?? 'close') === 'open',
                'state' => $response['instance']['status'] ?? 'close',
                'response_time' => $responseTime,
                'instance_name' => $instance->instance_name,
                'message' => 'Conexão bem-sucedida',
                'full_response' => $response
            ];
        } catch (Exception $e) {
            $responseTime = round((microtime(true) - $startTime) * 1000, 2);

            return [
                'success' => false,
                'connected' => false,
                'response_time' => $responseTime,
                'instance_name' => $instance->instance_name,
                'message' => 'Erro de conexão: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ];
        }
    }

    // ===== MÉTODOS DE ENVIO UNIFICADOS =====

    public function sendTextMessage(string $remoteJid, string $text): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Enviando mensagem de texto', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'text_preview' => substr($text, 0, 50)
            ]);

            $this->validateConnection();

            $payload = [
                'number' => $remoteJid,
                'text' => $text,
                'delay' => 0,
                'linkPreview' => false,
            ];

            $response = $this->makeRequest('POST', "/message/sendText/{$instance->instance_name}", $payload);

            Log::info('Mensagem de texto enviada', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'message_id' => $response['key']['id'] ?? 'unknown'
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao enviar mensagem de texto', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function sendFile(string $remoteJid, string $filePath, string $caption = '', array $options = []): array
    {
        try {
            $this->validateConnection();
            $this->validateFile($filePath);

            $fileInfo = $this->getFileInfo($filePath);

            Log::info('Enviando arquivo (auto-detectado)', [
                'file_path' => basename($filePath),
                'mime_type' => $fileInfo['mime_type'],
                'media_type' => $fileInfo['media_type'],
                'file_size' => $fileInfo['file_size']
            ]);

            // Detectar se é áudio e usar endpoint apropriado
            if ($this->isAudioFile($fileInfo['mime_type'])) {
                Log::info('Detectado áudio, usando endpoint sendWhatsAppAudio', [
                    'mime_type' => $fileInfo['mime_type']
                ]);
                return $this->sendAudio($remoteJid, $filePath, $options);
            }

            return $this->sendMedia($remoteJid, $filePath, $caption, $options);
        } catch (Exception $e) {
            Log::error('Erro ao enviar arquivo', [
                'file_path' => basename($filePath),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    public function sendMediaMessage(string $remoteJid, string $mediaPath, string $caption = '', array $options = []): array
    {
        return $this->sendFile($remoteJid, $mediaPath, $caption, $options);
    }

    public function sendWhatsAppAudio(string $remoteJid, string $audioPath, array $options = []): array
    {
        return $this->sendFile($remoteJid, $audioPath, '', $options);
    }

    private function sendMedia(string $remoteJid, string $filePath, string $caption, array $options): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Enviando arquivo/mídia', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'media_path' => basename($filePath),
                'caption_preview' => substr($caption, 0, 50)
            ]);

            $fileInfo = $this->getFileInfo($filePath);
            $mediaBase64 = base64_encode(file_get_contents($filePath));

            $payload = [
                'number' => $remoteJid,
                'mediatype' => $fileInfo['media_type'],
                'mimetype' => $fileInfo['mime_type'],
                'media' => $mediaBase64,
                'fileName' => $fileInfo['filename'],
                'delay' => $options['delay'] ?? 0,
                'linkPreview' => $options['linkPreview'] ?? false,
            ];

            if (!empty($caption)) {
                $payload['caption'] = $caption;
            }

            $this->addOptionalFields($payload, $options);

            $response = $this->makeRequest('POST', "/message/sendMedia/{$instance->instance_name}", $payload);

            Log::info('Arquivo/mídia enviado', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'message_id' => $response['key']['id'] ?? 'unknown',
                'file_size' => $fileInfo['file_size'],
                'mime_type' => $fileInfo['mime_type']
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao enviar arquivo/mídia', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'media_path' => basename($filePath),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    private function sendAudio(string $remoteJid, string $filePath, array $options): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Enviando áudio do WhatsApp', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'audio_path' => basename($filePath)
            ]);

            $fileInfo = $this->getFileInfo($filePath);
            
            if (!$this->isAudioFile($fileInfo['mime_type'])) {
                throw new Exception("Arquivo não é um áudio válido: {$fileInfo['mime_type']}");
            }

            $audioBase64 = base64_encode(file_get_contents($filePath));

            $payload = [
                'number' => $remoteJid,
                'audio' => $audioBase64,
                'delay' => $options['delay'] ?? 0,
                'linkPreview' => $options['linkPreview'] ?? false,
            ];

            $this->addOptionalFields($payload, $options);

            $response = $this->makeRequest('POST', "/message/sendWhatsAppAudio/{$instance->instance_name}", $payload);

            Log::info('Áudio do WhatsApp enviado', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'message_id' => $response['key']['id'] ?? 'unknown',
                'file_size' => $fileInfo['file_size']
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao enviar áudio do WhatsApp', [
                'instance_name' => $instance->instance_name,
                'remote_jid' => $this->maskJid($remoteJid),
                'audio_path' => basename($filePath),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    // ===== MÉTODOS DE VALIDAÇÃO =====

    public function checkSingleWhatsAppNumber(string $number): array
    {
        try {
            $results = $this->checkWhatsAppNumbers([$number]);
            return $results[0] ?? ['exists' => false, 'number' => $number, 'jid' => null];
        } catch (Exception $e) {
            Log::error('Erro ao verificar número único WhatsApp', [
                'number' => substr($number, 0, 6) . '****',
                'error' => $e->getMessage()
            ]);

            return [
                'exists' => false,
                'number' => $number,
                'jid' => null,
                'error' => $e->getMessage()
            ];
        }
    }

    public function checkWhatsAppNumbers(array $numbers): array
    {
        $instance = $this->getInstance();
        
        try {
            Log::info('Verificando se números existem no WhatsApp', [
                'instance_name' => $instance->instance_name,
                'numbers_count' => count($numbers),
                'numbers_preview' => array_map(fn($n) => substr($n, 0, 6) . '****', $numbers)
            ]);

            $this->validateConnection();

            $response = $this->makeRequest('POST', "/chat/whatsappNumbers/{$instance->instance_name}", [
                'numbers' => $numbers
            ]);

            Log::info('Verificação de números concluída', [
                'instance_name' => $instance->instance_name,
                'total_checked' => count($response),
                'results' => array_map(fn($r) => [
                    'number' => substr($r['number'] ?? '', 0, 6) . '****',
                    'exists' => $r['exists'] ?? false,
                    'jid' => $r['jid'] ? substr($r['jid'], 0, 10) . '****' : null
                ], $response)
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao verificar números WhatsApp', [
                'instance_name' => $instance->instance_name,
                'numbers_count' => count($numbers),
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    // ===== MÉTODOS DE GRUPOS =====

    public function getGroupInfo(string $groupJid): array
    {
        try {
            $instance = $this->getInstance();

            Log::info('Obtendo informações do grupo', [
                'group_jid' => $this->maskJid($groupJid),
                'instance' => $instance->instance_name
            ]);

            $endpoint = "/group/findGroupInfos/{$instance->instance_name}";
            $queryParams = ['groupJid' => $groupJid];

            $response = $this->makeRequest('GET', $endpoint, $queryParams);

            if (empty($response) || (!isset($response['subject']) && !isset($response['name']))) {
                Log::warning('Resposta da API não contém informações do grupo', [
                    'group_jid' => $this->maskJid($groupJid),
                    'response' => $response,
                    'response_keys' => is_array($response) ? array_keys($response) : 'not_array'
                ]);

                throw new Exception('Informações do grupo não encontradas na resposta da API');
            }

            Log::info('Informações do grupo obtidas via API', [
                'group_jid' => $this->maskJid($groupJid),
                'group_subject' => $response['subject'] ?? 'N/A',
                'participants_count' => $response['size'] ?? 0,
                'instance' => $instance->instance_name
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao obter informações do grupo: ' . $e->getMessage(), [
                'group_jid' => $this->maskJid($groupJid),
                'instance' => $this->instance->instance_name,
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    public function getContactInfo(string $contactJid): array
    {
        try {
            if (str_ends_with($contactJid, '@g.us')) {
                Log::debug('Detectado grupo, redirecionando para getGroupInfo', [
                    'contact_jid' => $this->maskJid($contactJid)
                ]);
                return $this->getGroupInfo($contactJid);
            }

            Log::info('Obtendo informações do contato individual', [
                'contact_jid' => $this->maskJid($contactJid),
                'instance' => $this->instance->instance_name
            ]);

            $response = $this->makeRequest('GET', "/chat/findContact/{$contactJid}");

            Log::info('Informações do contato obtidas via API', [
                'contact_jid' => $this->maskJid($contactJid),
                'contact_name' => $response['pushName'] ?? $response['name'] ?? 'N/A',
                'instance' => $this->instance->instance_name
            ]);

            return $response;
        } catch (Exception $e) {
            Log::error('Erro ao obter informações do contato: ' . $e->getMessage(), [
                'contact_jid' => $this->maskJid($contactJid),
                'instance' => $this->instance->instance_name,
                'error' => $e->getMessage()
            ]);

            throw $e;
        }
    }

    public function getAllGroups(): array
    {
        try {
            $response = $this->makeRequest('GET', "/group/findGroupInfos/{$this->instance->instance_name}");

            Log::info('Lista de grupos obtida', [
                'groups_count' => is_array($response) ? count($response) : 0,
                'instance' => $this->instance->instance_name
            ]);

            return is_array($response) ? $response : [];
        } catch (Exception $e) {
            Log::error('Erro ao listar grupos: ' . $e->getMessage(), [
                'instance' => $this->instance->instance_name,
                'error' => $e->getMessage()
            ]);

            return [];
        }
    }

    public function getGroupParticipants(string $groupJid): array
    {
        try {
            $groupInfo = $this->getGroupInfo($groupJid);
            return $groupInfo['participants'] ?? [];
        } catch (Exception $e) {
            Log::error('Erro ao obter participantes do grupo: ' . $e->getMessage(), [
                'group_jid' => $this->maskJid($groupJid),
                'error' => $e->getMessage()
            ]);

            return [];
        }
    }

    // ===== MÉTODOS ESTÁTICOS =====

    public static function fetchInstancesFromEvolutionApi(): array
    {
        try {
            Log::info('Buscando instâncias diretamente da Evolution API');

            $apiUrl = config('chat.evolution.api_url') ?? env('EVOLUTION_API_URL');
            $apiKey = config('chat.evolution.api_key') ?? env('EVOLUTION_API_KEY');

            if (!$apiUrl || !$apiKey) {
                throw new Exception('Evolution API URL ou API Key não configuradas');
            }

            if (!str_starts_with($apiUrl, 'http://') && !str_starts_with($apiUrl, 'https://')) {
                $apiUrl = 'https://' . $apiUrl;
            }

            $url = rtrim($apiUrl, '/') . '/instance/fetchInstances';

            $response = Http::withHeaders([
                'apikey' => $apiKey,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ])
                ->timeout(15)
                ->retry(2, 1000)
                ->get($url);

            if (!$response->successful()) {
                throw new Exception("Erro na Evolution API - HTTP {$response->status()}: " . $response->body());
            }

            $instances = $response->json() ?? [];

            Log::info('Instâncias obtidas da Evolution API', [
                'count' => count($instances),
                'instances' => array_map(fn($i) => [
                    'name' => $i['name'] ?? 'unknown',
                    'connectionStatus' => $i['connectionStatus'] ?? 'unknown',
                    'profileName' => $i['profileName'] ?? null
                ], $instances)
            ]);

            return $instances;
        } catch (Exception $e) {
            Log::error('Erro ao buscar instâncias da Evolution API', [
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    // ===== MÉTODOS AUXILIARES =====

    private function validateConnection(): void
    {
        if (!$this->instance->is_online) {
            throw new Exception("Instância '{$this->instance->instance_name}' não está conectada");
        }
    }

    private function validateFile(string $filePath): void
    {
        if (!file_exists($filePath)) {
            throw new Exception("Arquivo não encontrado: {$filePath}");
        }
    }

    private function isAudioFile(string $mimeType): bool
    {
        foreach (self::AUDIO_MIME_TYPES as $audioType) {
            if (str_starts_with($mimeType, $audioType)) {
                return true;
            }
        }
        return false;
    }

    private function getFileInfo(string $filePath): array
    {
        if (!file_exists($filePath)) {
            throw new Exception("Arquivo não encontrado: {$filePath}");
        }

        $fileSize = filesize($filePath);
        $mimeType = mime_content_type($filePath) ?: 'application/octet-stream';
        $filename = basename($filePath);

        $mediaType = match (true) {
            str_starts_with($mimeType, 'image/') => 'image',
            str_starts_with($mimeType, 'video/') => 'video',
            str_starts_with($mimeType, 'audio/') => 'audio',
            $mimeType === 'application/pdf' => 'document',
            str_starts_with($mimeType, 'application/') => 'document',
            str_starts_with($mimeType, 'text/') => 'document',
            default => 'document'
        };

        return [
            'filename' => $filename,
            'file_size' => $fileSize,
            'mime_type' => $mimeType,
            'media_type' => $mediaType,
            'file_path' => $filePath
        ];
    }

    private function addOptionalFields(array &$payload, array $options): void
    {
        if (!empty($options['mentioned']) && is_array($options['mentioned'])) {
            $payload['mentioned'] = $options['mentioned'];
        }

        if (!empty($options['mentionsEveryOne'])) {
            $payload['mentionsEveryOne'] = true;
        }

        if (!empty($options['quoted'])) {
            $payload['quoted'] = $options['quoted'];
        }
    }

    private function maskJid(string $jid): string
    {
        if (empty($jid)) return '';

        $parts = explode('@', $jid);
        $number = $parts[0] ?? '';
        $domain = $parts[1] ?? '';

        if (strlen($number) >= 8) {
            $masked = substr($number, 0, 4) . '****' . substr($number, -2);
            return $masked . ($domain ? "@{$domain}" : '');
        }

        return '****' . ($domain ? "@{$domain}" : '');
    }

    private function maskUrl(string $url): string
    {
        if (empty($url)) return '';

        $maskedUrl = preg_replace('/apikey=[^&\\s]+/', 'apikey=****', $url);
        $maskedUrl = preg_replace('/(\\d{4})\\d{4,}(\\d{2})/', '$1****$2', $maskedUrl);

        return $maskedUrl;
    }

    // ===== GETTERS =====

    public function getInstanceName(): string
    {
        return $this->getInstance()->instance_name;
    }

    public function getInstanceModel(): EvolutionInstance
    {
        return $this->getInstance();
    }

    public function isConnected(): bool
    {
        return $this->getInstance()->is_online;
    }

    // ===== MÉTODOS DE REQUISIÇÃO =====

    protected function makeRequest(string $method, string $endpoint, array $data = [], ?int $timeout = null): array
    {
        try {
            if (!$this->instance) {
                throw new Exception('Instância Evolution não configurada');
            }

            $url = rtrim($this->instance->api_url, '/') . $endpoint;
            $timeout = $timeout ?? $this->timeout;

            $queryParams = [];
            $bodyData = [];

            if (str_contains($endpoint, '/group/findGroupInfos/') && !empty($data['groupJid'])) {
                $queryParams['groupJid'] = $data['groupJid'];
                $bodyData = array_diff_key($data, ['groupJid' => '']);
            } else {
                $bodyData = $data;
            }

            Log::debug('Fazendo requisição Evolution API', [
                'instance' => $this->instance->instance_name,
                'method' => $method,
                'url' => $this->maskUrl($url),
                'endpoint' => $endpoint,
                'has_query_params' => !empty($queryParams),
                'has_body_data' => !empty($bodyData)
            ]);

            $httpClient = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'apikey' => $this->instance->api_key,
            ])
                ->timeout($timeout)
                ->retry($this->retryTimes, 1000);

            $response = match (strtoupper($method)) {
                'GET' => $httpClient->get($url, $queryParams),
                'POST' => $httpClient->post($url, $bodyData),
                'PUT' => $httpClient->put($url, $bodyData),
                'PATCH' => $httpClient->patch($url, $bodyData),
                'DELETE' => $httpClient->delete($url, $bodyData),
                default => throw new Exception("Método HTTP não suportado: {$method}")
            };

            if (!$response->successful()) {
                throw new Exception("HTTP {$response->status()}: {$response->body()}");
            }

            $decodedResponse = $response->json();

            if (is_null($decodedResponse)) {
                throw new Exception('Resposta JSON inválida');
            }

            Log::debug('Resposta Evolution API recebida', [
                'instance' => $this->instance->instance_name,
                'status' => $response->status(),
                'response_size' => strlen($response->body()),
                'has_data' => !empty($decodedResponse)
            ]);

            return $decodedResponse;
        } catch (Exception $e) {
            Log::error('Erro na requisição Evolution API', [
                'instance' => $this->instance->instance_name ?? 'unknown',
                'method' => $method,
                'endpoint' => $endpoint,
                'error' => $e->getMessage(),
                'status' => isset($response) ? $response->status() : 'unknown',
                'response_body' => isset($response) ? $response->body() : null,
                'url' => isset($url) ? $this->maskUrl($url) : 'N/A'
            ]);

            throw new Exception("Erro na comunicação com a Evolution API: " . $e->getMessage());
        }
    }
}