<?php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Modules\Chat\Models\EvolutionInstance;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\MessageAttachment;
use Modules\Contacts\Models\Contact;
use Carbon\Carbon;
use Exception;

class EvolutionSyncService
{
    private const MESSAGE_TYPE_MAPPING = [
        'conversation' => 'text',
        'extendedTextMessage' => 'text',
        'imageMessage' => 'image',
        'audioMessage' => 'audio',
        'videoMessage' => 'video',
        'documentMessage' => 'document',
        'stickerMessage' => 'sticker',
        'locationMessage' => 'location',
        'contactMessage' => 'contact',
        'pollMessage' => 'poll',
        'reactionMessage' => 'reaction',
        'ephemeral' => 'ephemeral'
    ];

    private const MIME_TYPE_EXTENSIONS = [
        'image/jpeg' => 'jpg',
        'image/png' => 'png',
        'image/gif' => 'gif',
        'image/webp' => 'webp',
        'audio/ogg' => 'ogg',
        'audio/mpeg' => 'mp3',
        'audio/mp4' => 'm4a',
        'audio/wav' => 'wav',
        'video/mp4' => 'mp4',
        'video/quicktime' => 'mov',
        'application/pdf' => 'pdf',
        'application/msword' => 'doc',
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx'
    ];

    private const DEFAULT_MIME_TYPES = [
        'image' => 'image/jpeg',
        'audio' => 'audio/ogg',
        'video' => 'video/mp4',
        'document' => 'application/octet-stream'
    ];

    private const MEDIA_MESSAGE_TYPES = ['imageMessage', 'audioMessage', 'videoMessage', 'documentMessage', 'stickerMessage'];

    protected ?EvolutionInstance $instance = null;
    protected EvolutionApiService $evolutionApi;
    protected ChatRedisService $chatRedis;
    protected WhatsAppContactService $whatsappContactService;
    protected ?ContactProfilePhotoService $contactProfilePhotoService = null;

    public function __construct(
        EvolutionApiService $evolutionApi,
        ChatRedisService $chatRedis,
        WhatsAppContactService $whatsappContactService,
        ContactProfilePhotoService $contactProfilePhotoService = null
    ) {
        $this->evolutionApi = $evolutionApi;
        $this->chatRedis = $chatRedis;
        $this->whatsappContactService = $whatsappContactService;
        $this->contactProfilePhotoService = $contactProfilePhotoService;
    }

    public function setInstance(EvolutionInstance $instance): void
    {
        $this->instance = $instance;
        $this->evolutionApi->setInstance($instance);
    }

    // ===== SINCRONIZAÇÃO DE CHATS =====

    public function syncSingleChat(array $chatData): array
    {
        try {
            $originalJid = $chatData['id'];
            $normalizedJid = $this->normalizeJid($originalJid);
            $cleanNumber = preg_replace('/[^0-9]/', '', $normalizedJid);

            Log::debug('Sincronizando chat', [
                'normalized_jid' => $this->maskJid($normalizedJid),
                'is_group' => $chatData['isGroup'] ?? false
            ]);

            // Buscar conversa existente
            $conversation = $this->findExistingConversation($normalizedJid, $cleanNumber);

            if (!$conversation) {
                // Criar nova conversa
                $phoneNumber = $this->extractPhoneFromJid($normalizedJid);
                $contact = $this->findOrCreateContact($phoneNumber, $chatData);

                if (!$contact) {
                    throw new Exception('Não foi possível criar/encontrar contato');
                }

                $conversation = $this->createConversation($normalizedJid, $contact, $chatData);

                Log::info('Nova conversa criada', [
                                        'conversation_id' => $conversation->id,
                    'contact_id' => $contact->id
                ]);
            }

            return [
                'conversation_id' => $conversation->id,
                'contact_id' => $conversation->contact_id,
                'action' => 'synced'
            ];
        } catch (Exception $e) {
            Log::error('Erro ao sincronizar chat', [
                'jid' => $this->maskJid($chatData['id'] ?? ''),
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Buscar conversa existente
     */
    private function findExistingConversation(string $normalizedJid, string $cleanNumber): ?Conversation
    {
        return Conversation::where('evolution_instance_name', $this->instance->instance_name)
            ->where(function ($query) use ($normalizedJid, $cleanNumber) {
                $query->where('whatsapp_remote_jid', $normalizedJid)
                    ->orWhere('whatsapp_chat_id', $normalizedJid);

                if (strlen($cleanNumber) >= 10) {
                    $query->orWhere('whatsapp_chat_id', $cleanNumber)
                        ->orWhere('whatsapp_remote_jid', $cleanNumber);
                }
            })
            ->first();
    }

    // ===== GESTÃO DE CONTATOS =====

    private function findOrCreateContact(string $phoneNumber, array $chatData): ?Contact
    {
        try {
            $isGroup = $chatData['isGroup'] ?? false;

            if ($isGroup) {
                return $this->handleGroupContact($chatData);
            }

            // Usar o WhatsAppContactService
            $pushName = $chatData['name'] ?? $chatData['pushName'] ?? null;

            // Não usar pushName se for mensagem sua
            $fromMe = $chatData['fromMe'] ?? false;
            if ($fromMe) {
                $pushName = null;
            }

            $contact = $this->whatsappContactService->findOrCreateContact($phoneNumber, $pushName);

            if ($contact) {
                Log::info('Contato processado via service', [
                    'contact_id' => $contact->id,
                    'name' => $contact->first_name,
                    'phone' => $this->maskPhone($phoneNumber),
                    'action' => $contact->wasRecentlyCreated ? 'created' : 'found'
                ]);
            }

            return $contact;
        } catch (Exception $e) {
            Log::error('Erro ao processar contato', [
                'phone' => $this->maskPhone($phoneNumber),
                'is_group' => $chatData['isGroup'] ?? false,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Manter apenas lógica de grupo
     */
    private function handleGroupContact(array $chatData): Contact
    {
        $groupName = $chatData['groupName'] ?? $chatData['name'] ?? 'Grupo WhatsApp';
        $groupId = str_replace(['@g.us', '@s.whatsapp.net'], '', $chatData['id'] ?? '');

        $contact = Contact::where('first_name', $groupName)
            ->whereNull('whatsapp_number')
            ->first();

        if (!$contact) {
            $contact = Contact::create([
                'first_name' => $groupName,
                'last_name' => '',
                'whatsapp_number' => null,
                'whatsapp_opt_in' => true,
                'last_whatsapp_interaction' => now(),
                'created_by' => 1,
                'user_id' => 1,
                'job_title' => "Grupo WhatsApp - ID: " . substr($groupId, 0, 15),
            ]);

            Log::info('Contato de grupo criado', [
                'contact_id' => $contact->id,
                'group_name' => $groupName
            ]);
        }

        return $contact;
    }

    // ===== GESTÃO DE CONVERSAS =====

    private function createConversation(string $normalizedJid, Contact $contact, array $chatData): Conversation
    {
        $isGroup = str_ends_with($normalizedJid, '@g.us');
        $phoneNumber = $isGroup ? null : $this->extractPhoneFromJid($normalizedJid);

        $chatChannelId = \Modules\Chat\Models\ChatChannel::where('type', 'evolution_api')
            ->where('is_active', true)
            ->whereRaw("JSON_EXTRACT(config, '$.instance_name') = ?", [$this->instance->instance_name])
            ->value('id') ?? 2;

        $conversationName = $isGroup
            ? ($chatData['groupName'] ?? $chatData['name'] ?? 'Grupo WhatsApp')
            : ($contact->display_name ?? $contact->first_name ?? $this->formatPhoneForDisplay($phoneNumber));

        $conversation = Conversation::create([
            'whatsapp_chat_id' => str_replace(['@s.whatsapp.net', '@g.us'], '', $normalizedJid),
            'whatsapp_remote_jid' => $normalizedJid,
            'contact_id' => $contact->id,
            'contact_name' => $conversationName,
            'phone' => $phoneNumber,
            'type' => 'external',
            'status' => 'active',
            'evolution_instance_name' => $this->instance->instance_name,
            'chat_channel_id' => $chatChannelId,
            'whatsapp_profile_name' => $conversationName,
            'whatsapp_status' => 'active',
            'last_whatsapp_activity' => now(),
            'created_by' => 1,
            'assigned_user_id' => null, // ✅ DEIXAR SEM USUÁRIO ATRIBUÍDO AUTOMATICAMENTE
            'metadata' => [
                'is_group' => $isGroup,
                'created_from' => 'webhook',
                'whatsapp_type' => 'whatsapp',
                'group_data' => $isGroup ? $chatData : null
            ]
        ]);

        return $conversation;
    }

    // ===== CRIAÇÃO DE MENSAGENS =====

    public function createMessageFromApiData(Conversation $conversation, array $messageData): Message
    {
        $key = $messageData['key'] ?? [];
        $message = $messageData['message'] ?? [];
        $isFromMe = $messageData['fromMe'] ?? $key['fromMe'] ?? false;
        $whatsappMessageId = $key['id'] ?? null; // ID único do WhatsApp/Evolution API

        // ✅ DEDUPLICAÇÃO PRINCIPAL: Verificar se a mensagem já existe no BD
        if ($whatsappMessageId) {
            $existingMessage = Message::where('conversation_id', $conversation->id)
                                      ->where('whatsapp_message_id', $whatsappMessageId)
                                      ->first();
            if ($existingMessage) {
                Log::info('Mensagem WhatsApp já existe no BD, ignorando criação duplicada.', [
                    'whatsapp_message_id' => $whatsappMessageId,
                    'conversation_id' => $conversation->id,
                    'existing_message_id' => $existingMessage->id
                ]);
                // Se a mensagem já existe, apenas atualizamos o `updated_at` para manter a conversa recente.
                // Não incrementamos `unread_count` novamente, nem processamos anexos se já existirem.
                $existingMessage->touch(); // Atualiza updated_at
                return $existingMessage;
            }
        }

        if ($isFromMe) {
            $senderId = auth()->id() ?? null;
            $senderType = '\Modules\Users\Models\User';
        } else {
            $contact = $this->getOrCreateContactFromMessage($messageData, $conversation);
            $senderId = $contact->id;
            $senderType = '\Modules\Contacts\Models\Contact';
        }

        $messageType = $this->detectMessageType($message);
        $content = $this->extractMessageContent($message, $messageType);
        $whatsappMessageType = $this->mapWhatsAppMessageTypeToEnum($messageData['messageType'] ?? 'conversation');

        $messageModel = Message::create([
            'conversation_id' => $conversation->id,
            'sender_id' => $senderId,
            'sender_type' => $senderType,
            'content' => $content,
            'message_type' => $messageType,
            'whatsapp_message_id' => $whatsappMessageId, // Armazenar o ID único do WhatsApp
            'whatsapp_key_id' => $whatsappMessageId,
            'whatsapp_key_remote_jid' => $key['remoteJid'] ?? null,
            'whatsapp_key_from_me' => $isFromMe,
            'whatsapp_message_type' => $whatsappMessageType,
            'whatsapp_timestamp' => isset($messageData['messageTimestamp']) ?
                Carbon::createFromTimestamp($messageData['messageTimestamp']) : now(),
            'delivery_status' => $isFromMe ? 'sent' : 'pending',
            'status' => $isFromMe ? 'sent' : 'received',
            'is_from_contact' => !$isFromMe,
            'metadata' => json_encode([
                'push_name' => $messageData['pushName'] ?? null,
                'original_message_type' => $messageData['messageType'] ?? null,
                'source' => $messageData['source'] ?? 'webhook',
                'original_key' => $key,
                'from_me' => $isFromMe
            ]),
            'sent_at' => now()
        ]);

        if ($this->hasMedia($message) || !empty($messageData['base64'])) {
            $this->processMessageAttachments($messageModel, $message, $messageData);
        }

        // Incremento do unread_count (apenas se for uma nova mensagem recebida)
        if (
            $conversation->type === 'external' &&
            !$isFromMe && // Mensagem não foi enviada pelo CRM
            $conversation->status !== 'closed' &&
            $conversation->status !== 'archived'
        ) {
            $conversation->increment('unread_count');
            $conversation->touch();
            $conversation->refresh();

            Log::info('Unread count incrementado', [
                'conversation_id' => $conversation->id,
                'new_count' => $conversation->unread_count
            ]);
        }

        Log::info('Mensagem WhatsApp criada', [
            'message_id' => $messageModel->id,
            'conversation_id' => $conversation->id,
            'from_me' => $isFromMe
        ]);

        return $messageModel;
    }

    /**
     * Obter/criar contato da mensagem
     */
    protected function getOrCreateContactFromMessage(array $messageData, Conversation $conversation): Contact
    {
        if ($conversation->contact_id) {
            $contact = Contact::find($conversation->contact_id);
            if ($contact) {
                return $contact;
            }
        }

        $key = $messageData['key'] ?? [];
        $remoteJid = $key['remoteJid'] ?? '';
        $phoneNumber = $this->extractPhoneFromJid($remoteJid);

        if (!$phoneNumber) {
            throw new Exception('Não foi possível extrair telefone da mensagem');
        }

        // Usar o WhatsAppContactService
        $pushName = $messageData['pushName'] ?? null;
        $contact = $this->whatsappContactService->findOrCreateContact($phoneNumber, $pushName);

        if (!$contact) {
            throw new Exception('Não foi possível criar contato para mensagem');
        }

        return $contact;
    }

    // ===== PROCESSAMENTO DE ANEXOS =====

    protected function processMessageAttachments(Message $message, array $messageContent, array $messageData): void
    {
        try {
            $messageType = $this->detectMessageType($messageContent);
            $base64FromPayload = $messageData['base64'] ?? null;

            $mediaKey = $messageType . 'Message';
            $mediaData = $messageContent[$mediaKey] ?? null;

            if (!$mediaData) {
                Log::warning('Dados de mídia não encontrados', [
                    'message_id' => $message->id,
                    'expected_key' => $mediaKey
                ]);
                return;
            }

            // ✅ Verificar se o anexo já existe para esta mensagem (evitar duplicação de anexos)
            if (MessageAttachment::where('message_id', $message->id)->exists()) {
                 Log::info('Anexo já existe para a mensagem, ignorando criação duplicada.', ['message_id' => $message->id]);
                 return;
            }

            $attachmentData = $this->extractAttachmentData($mediaData, $messageType);
            $attachment = $this->createMessageAttachment($message, $attachmentData);

            if ($attachment) {
                $this->downloadAttachment($attachment, $mediaData, $base64FromPayload);
            }
        } catch (Exception $e) {
            Log::error('Erro ao processar anexos', [
                'message_id' => $message->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    protected function extractAttachmentData(array $mediaData, string $messageType): array
    {
        // Converter campos array para string/null
        $mediaKey = $mediaData['mediaKey'] ?? null;
        if (is_array($mediaKey)) {
            $mediaKey = null;
        }
        
        $directPath = $mediaData['directPath'] ?? null;
        if (is_array($directPath)) {
            $directPath = null;
        }
        
        $duration = $mediaData['seconds'] ?? null;
        if (is_array($duration)) {
            $duration = null;
        }
        
        // Garantir que metadata seja JSON string limpo
        $metadata = [
            'fileSha256' => $this->convertArrayFieldToString($mediaData['fileSha256'] ?? null),
            'fileEncSha256' => $this->convertArrayFieldToString($mediaData['fileEncSha256'] ?? null),
            'mediaKeyTimestamp' => $mediaData['mediaKeyTimestamp'] ?? null,
            'caption' => $mediaData['caption'] ?? null,
            'thumbnail_base64' => $this->extractThumbnail($mediaData),
        ];

        return [
            'file_name' => $this->generateFileName($mediaData, $messageType),
            'file_type' => $messageType,
            'mime_type' => $mediaData['mimetype'] ?? $this->getMimeTypeByMessageType($messageType),
            'file_size' => $this->extractFileSize($mediaData),
            'external_url' => $mediaData['url'] ?? null,
            'whatsapp_media_key' => $mediaKey,
            'whatsapp_direct_path' => $directPath,
            'whatsapp_media_url' => $mediaData['url'] ?? null,
            'whatsapp_metadata' => json_encode($metadata),
            'width' => $mediaData['width'] ?? null,
            'height' => $mediaData['height'] ?? null,
            'duration' => $duration,
            'is_animated' => $mediaData['gifPlayback'] ?? false,
            'download_status' => 'pending'
        ];
    }

    /**
     * Converter campo array para string
     */
    private function convertArrayFieldToString($field): ?string
    {
        if (is_array($field)) {
            // Se for array de bytes, converter para base64
            if (isset($field[0]) && is_numeric($field[0])) {
                try {
                    return base64_encode(pack('C*', ...$field));
                } catch (Exception $e) {
                    return null;
                }
            }
            return null;
        }
        
        return is_string($field) ? $field : null;
    }

    /**
     * Extrair tamanho do arquivo corretamente
     */
    private function extractFileSize(array $mediaData): int
    {
        $fileLength = $mediaData['fileLength'] ?? 1;
        
        if (is_array($fileLength)) {
            // Se for objeto com 'low' e 'high' (formato Evolution)
            return (int)($fileLength['low'] ?? 1);
        }
        
        return (int)$fileLength ?: 1;
    }

    protected function createMessageAttachment(Message $message, array $attachmentData): ?MessageAttachment
    {
        try {
            return MessageAttachment::create([
                'message_id' => $message->id,
                'file_name' => $attachmentData['file_name'],
                'file_type' => $attachmentData['file_type'],
                'file_size' => $attachmentData['file_size'],
                'mime_type' => $attachmentData['mime_type'],
                'external_url' => $attachmentData['external_url'],
                'whatsapp_media_key' => $attachmentData['whatsapp_media_key'],
                'whatsapp_direct_path' => $attachmentData['whatsapp_direct_path'],
                'whatsapp_media_url' => $attachmentData['whatsapp_media_url'],
                'whatsapp_metadata' => $attachmentData['whatsapp_metadata'],
                'width' => $attachmentData['width'],
                'height' => $attachmentData['height'],
                'duration' => $attachmentData['duration'],
                'is_animated' => $attachmentData['is_animated'],
                'download_status' => $attachmentData['download_status'],
                'download_attempts' => 0
            ]);
        } catch (Exception $e) {
            Log::error('Erro ao criar anexo', [
                'message_id' => $message->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    protected function downloadAttachment(MessageAttachment $attachment, array $mediaData, ?string $base64FromPayload = null): void
    {
        try {
            if (!empty($base64FromPayload)) {
                $this->saveBase64Attachment($attachment, $base64FromPayload);
                return;
            }

            if (!empty($mediaData['base64'])) {
                $this->saveBase64Attachment($attachment, $mediaData['base64']);
                return;
            }

            $thumbnailData = $mediaData['jpegThumbnail'] ?? null;
            if ($thumbnailData) {
                $this->saveBase64Attachment($attachment, $thumbnailData);
                return;
            }

            $attachment->update([
                'download_status' => 'failed',
                'download_error' => 'Nenhuma fonte de base64 encontrada'
            ]);
        } catch (Exception $e) {
            $attachment->update([
                'download_status' => 'failed',
                'download_error' => $e->getMessage()
            ]);
        }
    }

    protected function saveBase64Attachment(MessageAttachment $attachment, string $base64Data): void
    {
        try {
            $attachment->update(['download_status' => 'downloading']);

            $fileData = base64_decode($base64Data, true);
            if ($fileData === false) {
                throw new Exception('Falha ao decodificar base64');
            }

            if (strlen($fileData) < 10) {
                throw new Exception('Dados de arquivo muito pequenos');
            }

            $fileName = time() . '_' . uniqid() . '.' . $this->getFileExtension($attachment->mime_type);
            $filePath = 'chat-media/' . $fileName;

            $saved = Storage::disk('public')->put($filePath, $fileData);

            if (!$saved) {
                throw new Exception('Falha ao salvar arquivo');
            }

            $attachment->update([
                'file_path' => $filePath,
                'download_status' => 'completed',
                'downloaded_at' => now(),
                'file_size' => strlen($fileData)
            ]);
        } catch (Exception $e) {
            $attachment->update([
                'download_status' => 'failed',
                'download_error' => $e->getMessage()
            ]);
        }
    }

    // ===== MÉTODOS AUXILIARES =====

    private function normalizeJid(string $jid): string
    {
        if (str_contains($jid, '@lid')) {
            $numbers = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);

            $numberMappings = [
                '90744203251776' => '556696322859',
                '236485311766672' => '556696322859',
            ];

            if (isset($numberMappings[$numbers])) {
                $numbers = $numberMappings[$numbers];
                Log::info('LID mapeado', [
                    'mapped' => substr($numberMappings[$numbers], 0, 4) . '****'
                ]);
            }

            if (strlen($numbers) >= 10) {
                return $numbers . '@s.whatsapp.net';
            }
        }

        if (str_ends_with($jid, '@s.whatsapp.net') || str_ends_with($jid, '@g.us')) {
            return $jid;
        }

        $numbers = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
        if (strlen($numbers) >= 10) {
            return $numbers . '@s.whatsapp.net';
        }

        return $jid;
    }

    private function extractPhoneFromJid(string $jid): string
    {
        $phone = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);

        if (strlen($phone) === 11 && !str_starts_with($phone, '55')) {
            $phone = '55' . $phone;
        }

        return $phone;
    }

    protected function formatPhoneForDisplay(string $phone): string
    {
        $clean = preg_replace('/[^0-9]/', '', $phone);

        if (strlen($clean) >= 10) {
            return '+55 ' . substr($clean, 0, 2) . ' ' . substr($clean, 2, 5) . '-' . substr($clean, 7);
        }

        return $phone;
    }

    protected function detectMessageType(array $message): string
    {
        if (isset($message['imageMessage'])) return 'image';
        if (isset($message['audioMessage'])) return 'audio';
        if (isset($message['videoMessage'])) return 'video';
        if (isset($message['documentMessage'])) return 'document';
        if (isset($message['stickerMessage'])) return 'sticker';
        if (isset($message['locationMessage'])) return 'location';
        if (isset($message['contactMessage'])) return 'contact';

        return 'text';
    }

    protected function extractMessageContent(array $message, string $messageType = 'text'): string
    {
        if (isset($message['conversation'])) {
            return $message['conversation'];
        }

        if (isset($message['extendedTextMessage']['text'])) {
            return $message['extendedTextMessage']['text'];
        }

        $mediaKey = $messageType . 'Message';
        if (isset($message[$mediaKey])) {
            $caption = $message[$mediaKey]['caption'] ?? '';
            if (!empty($caption)) {
                return $caption;
            }

            return match ($messageType) {
                'image' => '[Imagem]',
                'audio' => '[Áudio]',
                'video' => '[Vídeo]',
                'document' => '[Documento]',
                'sticker' => '[Sticker]',
                'location' => '[Localização]',
                'contact' => '[Contato]',
                default => '[Mídia]'
            };
        }

        return '';
    }

    protected function mapWhatsAppMessageTypeToEnum(string $whatsappType): string
    {
        return self::MESSAGE_TYPE_MAPPING[$whatsappType] ?? 'text';
    }

    protected function hasMedia(array $message): bool
    {
        foreach (self::MEDIA_MESSAGE_TYPES as $type) {
            if (isset($message[$type])) {
                return true;
            }
        }

        return false;
    }

    protected function generateFileName(array $mediaData, string $messageType): string
    {
        if ($messageType === 'audio') {
            $isPtt = $mediaData['ptt'] ?? false;
            $duration = $mediaData['seconds'] ?? 0;
            $extension = $this->getFileExtension($mediaData['mimetype'] ?? 'audio/ogg');

            return $isPtt ? "voice_message_{$duration}s.{$extension}" : "audio_{$duration}s.{$extension}";
        }

        $caption = $mediaData['caption'] ?? null;
        if ($caption && !empty(trim($caption))) {
            $safeName = preg_replace('/[^a-zA-Z0-9_-]/', '', $caption);
            if (strlen($safeName) > 3) {
                return $safeName . '.' . $this->getFileExtension($mediaData['mimetype'] ?? null);
            }
        }

        return $messageType . '.' . $this->getFileExtension($mediaData['mimetype'] ?? null);
    }

    protected function getMimeTypeByMessageType(string $messageType): string
    {
        return self::DEFAULT_MIME_TYPES[$messageType] ?? 'application/octet-stream';
    }

    protected function getFileExtension(?string $mimeType): string
    {
        if (!$mimeType) return 'ogg';

        return self::MIME_TYPE_EXTENSIONS[$mimeType] ?? 'ogg';
    }

    /**
     * Extrair thumbnail limpo
     */
    protected function extractThumbnail(array $mediaData): ?string
    {
        $thumbnail = $mediaData['jpegThumbnail'] ?? $mediaData['thumbnail'] ?? null;
        
        if (is_array($thumbnail)) {
            // Se for array de bytes, converter para base64
            if (isset($thumbnail[0]) && is_numeric($thumbnail[0])) {
                try {
                    $binaryData = pack('C*', ...$thumbnail);
                    return base64_encode($binaryData);
                } catch (Exception $e) {
                    return null;
                }
            }
            return null;
        }
        
        // Se for string, limpar e validar
        if (is_string($thumbnail) && !empty($thumbnail)) {
            // Limpar escapes desnecessários
            $cleanThumbnail = str_replace(['\/', '"'], ['/', '"'], $thumbnail);
            
            // Validar se é base64 válido (básico)
            if (preg_match('/^[A-Za-z0-9+\/]*={0,2}$/', $cleanThumbnail)) {
                return $cleanThumbnail;
            }
        }
        
        return null;
    }

    private function maskJid(string $jid): string
    {
        if (empty($jid)) return '';
        $parts = explode('@', $jid);
        $number = $parts[0] ?? '';
        $domain = $parts[1] ?? '';

        if (strlen($number) >= 8) {
            $masked = substr($number, 0, 4) . '****' . substr($number, -2);
            return $masked . ($domain ? "@{$domain}" : '');
        }

        return '****' . ($domain ? "@{$domain}" : '');
    }

    protected function maskPhone(string $phone): string
    {
        if (strlen($phone) >= 8) {
            return substr($phone, 0, 4) . '****' . substr($phone, -2);
        }
        return '****';
    }

    // ===== MÉTODOS PARA OUTROS EVENTOS =====

    public function handleMessageStatusUpdate(array $data): array
    {
        return ['action' => 'status_updated'];
    }

    public function handleConnectionStatusUpdate(array $data): array
    {
        return ['action' => 'connection_updated'];
    }

    public function handleQrCodeUpdate(array $data): array
    {
        return ['action' => 'qrcode_updated'];
    }

    public function handleInstanceStatusUpdate(array $data): array
    {
        return ['action' => 'instance_updated'];
    }

    public function handleChatUpdateEvent(array $data): array
    {
        return ['action' => 'chat_updated'];
    }

    public function handleContactUpdateEvent(array $data): array
    {
        return ['action' => 'contact_updated'];
    }

    /**
     * Testar canal Evolution
     */
    public function testChannel(string $instanceName): array
    {
        try {
            Log::info('Testando canal Evolution', ['instance_name' => $instanceName]);
            
            // Buscar instância
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            
            if (!$instance) {
                return [
                    'success' => false,
                    'message' => 'Instância não encontrada',
                    'instance_name' => $instanceName,
                    'connected' => false
                ];
            }
            
            // Configurar API
            $this->evolutionApi->setInstance($instance);
            
            // Testar conexão
            $connectionTest = $this->evolutionApi->testConnection();
            
            if (!$connectionTest['success']) {
                return [
                    'success' => false,
                    'message' => 'Falha na conexão com a Evolution API',
                    'details' => $connectionTest,
                    'connected' => false,
                    'instance_name' => $instanceName
                ];
            }
            
            // Obter status da instância
            $statusData = $this->evolutionApi->getConnectionStatus();
            
            Log::info('Teste de canal concluído', [
                'instance_name' => $instanceName,
                'connected' => $statusData['connected'],
                'state' => $statusData['state']
            ]);
            
            return [
                'success' => true,
                'message' => 'Canal testado com sucesso',
                'instance_name' => $instanceName,
                'connected' => $statusData['connected'],
                'connection_state' => $statusData['state'],
                'connection_test' => $connectionTest,
                'status' => $statusData,
                'instance_data' => [
                    'id' => $instance->id,
                    'display_name' => $instance->display_name,
                    'api_url' => $instance->api_url,
                    'status' => $instance->status,
                    'connection_state' => $instance->connection_state,
                    'whatsapp_name' => $instance->whatsapp_name,
                    'whatsapp_number' => $instance->whatsapp_number
                ]
            ];
            
        } catch (Exception $e) {
            Log::error('Erro ao testar canal Evolution', [
                'instance_name' => $instanceName,
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'message' => 'Erro ao testar canal: ' . $e->getMessage(),
                'instance_name' => $instanceName,
                'connected' => false,
                'error_details' => $e->getMessage()
            ];
        }
    }

    /**
     * Sincronizar canal Evolution
     */
    public function syncChannel(string $instanceName): array
    {
        try {
            Log::info('Sincronizando canal Evolution', ['instance_name' => $instanceName]);
            
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            
            if (!$instance) {
                return [
                    'success' => false,
                    'message' => 'Instância não encontrada',
                    'instance_name' => $instanceName
                ];
            }
            
            // Configurar instância
            $this->setInstance($instance);
            
            // Executar sincronização
            $syncResult = $this->sync('incremental', []);
            
            Log::info('Sincronização de canal concluída', [
                'instance_name' => $instanceName,
                'result' => $syncResult
            ]);
            
            return [
                'success' => true,
                'message' => 'Canal sincronizado com sucesso',
                'instance_name' => $instanceName,
                'sync_result' => $syncResult
            ];
            
        } catch (Exception $e) {
            Log::error('Erro ao sincronizar canal Evolution', [
                'instance_name' => $instanceName,
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'message' => 'Erro ao sincronizar canal: ' . $e->getMessage(),
                'instance_name' => $instanceName
            ];
        }
    }

    /**
     * Método de sincronização genérico (placeholder)
     */
    public function sync(string $type, array $options): array
    {
        // Implementar lógica de sincronização conforme necessário
        return ['status' => 'completed', 'type' => $type];
    }
}