<?php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Modules\Chat\Models\EvolutionWebhook;
use Modules\Chat\Models\EvolutionInstance;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\Conversation;
use Modules\Contacts\Models\Contact;
use Modules\Contacts\Models\Phone;
use Illuminate\Support\Facades\Cache; 
use Exception;

class EvolutionWebhookProcessorService
{
    private const EVENT_TYPE_MAPPING = [
        'messages_upsert' => ['messages_upsert', 'messages', 'messages.upsert', 'MESSAGES-UPSERT'],
        'messages_update' => ['messages_update', 'message_status', 'messages.update', 'MESSAGES-UPDATE'],
        'connection_update' => ['connection_update', 'connection_status', 'CONNECTION-UPDATE'],
        'qrcode_updated' => ['qrcode_updated', 'QRCODE-UPDATED'],
        'application_startup' => ['application_startup', 'instance_status', 'APPLICATION-STARTUP'],
        'chats_update' => ['chats_update', 'CHATS-UPDATE'],
        'chats_upsert' => ['chats_upsert', 'CHATS-UPSERT'],
        'contacts_update' => ['contacts_update', 'CONTACTS-UPDATE'],
        'contacts_upsert' => ['contacts_upsert', 'CONTACTS-UPSERT'],
    ];

    private const IGNORED_EVENTS = [
        'messages_set', 'contacts_set', 'chats_set', 'presence_update', 'send_message'
    ];

    private const WHATSAPP_STATUS_MAPPING = [
        'PENDING' => 'sending',
        'SERVER_ACK' => 'sent',
        'DELIVERY_ACK' => 'delivered',
        'READ' => 'read',
        'ERROR' => 'failed',
        'FAILED' => 'failed'
    ];

    private const STATUS_RANK = [
        'sending' => 10,
        'sent' => 20,
        'delivered' => 30,
        'read' => 40,
        'failed' => 100
    ];

    private const YOUR_LIDS = ['90744203251776', '236485311766672'];

    protected ChatRedisService $chatRedis;
    protected EvolutionApiService $evolutionApi;
    protected EvolutionSyncService $evolutionSyncService;

    public function __construct(
        ChatRedisService $chatRedis,
        EvolutionApiService $evolutionApi,
        EvolutionSyncService $evolutionSyncService
    ) {
        $this->chatRedis = $chatRedis;
        $this->evolutionApi = $evolutionApi;
        $this->evolutionSyncService = $evolutionSyncService;
    }

    public function processWebhook(EvolutionWebhook $webhook): array
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $webhook->instance_name)->first();

            if (!$instance) {
                return ['action' => 'error', 'reason' => "Instância não encontrada: {$webhook->instance_name}"];
            }

            $this->evolutionSyncService->setInstance($instance);
            $normalizedEventType = $this->normalizeEventType($webhook->event_type);

            $result = match ($normalizedEventType) {
                'messages_upsert', 'messages' => $this->processMessageEvent($webhook),
                'messages_update', 'message_status' => $this->processMessageUpdateEvent($webhook),
                'connection_update', 'connection_status' => $this->processConnectionUpdateEvent($webhook),
                'qrcode_updated' => $this->processQrCodeUpdateEvent($webhook),
                'application_startup', 'instance_status' => $this->processInstanceStatusEvent($webhook),
                'chats_update', 'chats_upsert' => $this->processChatUpdateEvent($webhook),
                'contacts_update', 'contacts_upsert' => $this->processContactUpdateEvent($webhook),

                'messages_set', 'contacts_set', 'chats_set' => ['action' => 'ignored', 'reason' => 'bulk_event_not_processed'],
                'presence_update' => ['action' => 'ignored', 'reason' => 'presence_not_tracked'],
                'send_message' => ['action' => 'ignored', 'reason' => 'outbound_message_confirmation'],

                default => ['action' => 'unsupported', 'reason' => "Tipo de evento não suportado: {$webhook->event_type}"],
            };

            $this->updateWebhookRelationships($webhook, $result);

            if (in_array($result['action'], ['outbound_message_created', 'message_created', 'status_updated'])) {
                $this->publishToFrontend($webhook, $result);
            }

            return $result;
        } catch (Exception $e) {
            Log::error('Erro ao processar webhook', [
                'webhook_db_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    protected function processMessageEvent(EvolutionWebhook $webhook): array
    {
        try {
            $payload = $webhook->payload;
            $data = $payload['data'] ?? [];
            $key = $data['key'] ?? [];
            $message = $data['message'] ?? [];

            $originalFromMe = $key['fromMe'] ?? false;
            $originalRemoteJid = $key['remoteJid'] ?? '';
            $participant = $key['participant'] ?? null;

            $isFromMe = $this->determineIfFromMe($originalFromMe, $participant, $webhook->instance_name);

            if (empty($originalRemoteJid)) {
                return ['action' => 'error', 'reason' => 'remoteJid não encontrado'];
            }

            $normalizedRemoteJid = $this->normalizeJid($originalRemoteJid, $data);
            $isGroup = str_ends_with($normalizedRemoteJid, '@g.us');

            if ($isFromMe) {
                return $this->handleOutgoingMessage($webhook, $data, $key, $message, $normalizedRemoteJid, $isGroup);
            }

            return $this->handleIncomingMessage($webhook, $data, $key, $message, $normalizedRemoteJid, $isGroup);
        } catch (Exception $e) {
            Log::error('Erro ao processar mensagem', [
                'webhook_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    protected function handleOutgoingMessage(EvolutionWebhook $webhook, array $data, array $key, array $message, string $normalizedRemoteJid, bool $isGroup): array
    {
        try {
            // Ignorar mensagens para próprios LIDs
            $cleanNumber = preg_replace('/[^0-9]/', '', $normalizedRemoteJid);

            if (in_array($cleanNumber, self::YOUR_LIDS)) {
                return ['action' => 'ignored', 'reason' => 'self_to_self_message'];
            }

            // Buscar ou criar conversa
            $conversation = $this->findOrCreateConversation($webhook, $normalizedRemoteJid, $cleanNumber, $isGroup, $data, $key, true);

            if (!$conversation) {
                return ['action' => 'error', 'reason' => 'Falha ao criar/encontrar conversa'];
            }

            // Criar mensagem
            $messageModel = $this->createMessage($conversation, $data, $key, $message, $isGroup, true);

            // Disparar job de foto se necessário
            if (!$isGroup && $conversation->contact_id) {
                $contact = Contact::find($conversation->contact_id);
                if ($contact) {
                    $this->dispatchContactPhotoJobIfNeeded($contact, $this->extractPhoneFromJid($normalizedRemoteJid), $conversation);
                }
            }

            return [
                'action' => 'outbound_message_created',
                'conversation_id' => $conversation->id,
                'message_id' => $messageModel->id,
                'contact_id' => $conversation->contact_id,
                'from_me' => true,
                'is_group' => $isGroup
            ];
        } catch (Exception $e) {
            Log::error('Erro ao processar mensagem enviada', [
                'webhook_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    // ✅ NOVA LÓGICA IMPLEMENTADA
    protected function handleIncomingMessage(EvolutionWebhook $webhook, array $data, array $key, array $message, string $normalizedRemoteJid, bool $isGroup): array
    {
        try {
            // ✅ EXTRAIR NÚMERO LIMPO PARA BUSCAR CONTATO
            $cleanNumber = preg_replace('/[^0-9]/', '', explode('@', $normalizedRemoteJid)[0]);
            
            Log::info('🔍 Processando mensagem recebida', [
                'normalized_jid' => $this->maskJid($normalizedRemoteJid),
                'clean_number' => $this->maskNumber($cleanNumber),
                'instance' => $webhook->instance_name
            ]);
            
            // ✅ BUSCAR CONTATO PELO NÚMERO
            $contact = $this->findContactByNumber($cleanNumber);
            
            if (!$contact) {
                // ✅ CRIAR NOVA CONVERSA (CONTATO NÃO EXISTE)
                Log::info('📞 Contato não encontrado, criando nova conversa', [
                    'clean_number' => $this->maskNumber($cleanNumber)
                ]);
                
                $conversation = $this->findOrCreateConversation(
                    $webhook, 
                    $normalizedRemoteJid, 
                    $cleanNumber, 
                    $isGroup, 
                    $data, 
                    $key, 
                    false
                );
            } else {
                // ✅ APLICAR LÓGICA: ATIVA → FECHADA → NOVA
                Log::info('👤 Contato encontrado, aplicando lógica de conversas', [
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->first_name
                ]);
                
                $conversation = $this->findOrCreateConversationForContact(
                    $contact, 
                    $webhook, 
                    $normalizedRemoteJid, 
                    $isGroup, 
                    $data, 
                    $key
                );
            }

            if (!$conversation) {
                return ['action' => 'error', 'reason' => 'Falha ao criar/encontrar conversa'];
            }

            $messageModel = $this->createMessage($conversation, $data, $key, $message, $isGroup, false);

            if (!$isGroup && $conversation->contact_id) {
                $this->dispatchContactPhotoJobIfNeeded($conversation->contact, $this->extractPhoneFromJid($normalizedRemoteJid), $conversation);
            }

            return [
                'action' => 'message_created',
                'conversation_id' => $conversation->id,
                'message_id' => $messageModel->id,
                'contact_id' => $messageModel->sender_id,
                'from_me' => false,
                'is_group' => $isGroup,
                'conversation_status' => $conversation->status
            ];
            
        } catch (Exception $e) {
            Log::error('Erro ao processar mensagem recebida', [
                'webhook_id' => $webhook->id,
                'error' => $e->getMessage()
            ]);

            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    // ✅ MÉTODO PRINCIPAL - IMPLEMENTA SUA REGRA EXATA
    private function findOrCreateConversationForContact($contact, EvolutionWebhook $webhook, string $normalizedRemoteJid, bool $isGroup, array $data, array $key): ?Conversation
    {
        $chatChannelId = $this->getChatChannelId($webhook->instance_name);
        
        // ✅ 1. BUSCAR CONVERSA ATIVA PRIMEIRO
        $activeConversation = Conversation::where('contact_id', $contact->id)
            ->where('chat_channel_id', $chatChannelId)
            ->where('status', 'active')
            ->orderBy('updated_at', 'desc')
            ->first();

        if ($activeConversation) {
            Log::info('✅ Usando conversa ativa existente', [
                'conversation_id' => $activeConversation->id,
                'contact_id' => $contact->id,
                'contact_name' => $contact->first_name
            ]);
            return $activeConversation;
        }

        // ✅ 2. BUSCAR ÚLTIMA CONVERSA FECHADA
        $closedConversation = Conversation::where('contact_id', $contact->id)
            ->where('chat_channel_id', $chatChannelId)
            ->whereIn('status', ['closed', 'waiting', 'archived'])
            ->orderBy('closed_at', 'desc') 
            ->orderBy('updated_at', 'desc')   
            ->first();

        if ($closedConversation) {
            // ✅ 3. VALIDAR TEMPO DE FECHAMENTO
            $closedAt = $closedConversation->closed_at ?: $closedConversation->updated_at;
            $oneHourAgo = now()->subHour();

            Log::info('📋 Validando conversa fechada', [
                'conversation_id' => $closedConversation->id,
                'status' => $closedConversation->status,
                'closed_at' => $closedAt->toDateTimeString(),
                'one_hour_ago' => $oneHourAgo->toDateTimeString(),
                'should_create_new' => $closedAt->lt($oneHourAgo),
                'contact_name' => $contact->first_name
            ]);

            if ($closedAt->lt($oneHourAgo)) {
                // ✅ PASSOU DE 1 HORA - CRIAR NOVA
                Log::info('🆕 Criando nova conversa (passou de 1 hora)', [
                    'old_conversation_id' => $closedConversation->id,
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->first_name
                ]);
                
                return $this->createNewConversationForContact($contact, $webhook, $normalizedRemoteJid, $isGroup, $data, $key);
            } else {
                // ✅ MENOS DE 1 HORA - REABRIR
                Log::info('🔄 Reabrindo conversa fechada recentemente', [
                    'conversation_id' => $closedConversation->id,
                    'contact_id' => $contact->id,
                    'contact_name' => $contact->first_name
                ]);
                
                $closedConversation->update([
                    'status' => 'active',
                    'closed_by' => null,
                    'closed_at' => null,
                    'close_reason' => null,
                    'assigned_user_id' => null,
                    'updated_at' => now()
                ]);

                $this->createSystemMessage($closedConversation, 'conversation_auto_reopened', 'Conversa reaberta automaticamente.');
                
                return $closedConversation;
            }
        }

        // ✅ 4. NÃO EXISTE NENHUMA CONVERSA - CRIAR NOVA
        Log::info('🆕 Criando primeira conversa para o contato', [
            'contact_id' => $contact->id,
            'contact_name' => $contact->first_name
        ]);
        
        return $this->createNewConversationForContact($contact, $webhook, $normalizedRemoteJid, $isGroup, $data, $key);
    }

    // ✅ BUSCAR CONTATO POR NÚMERO
    private function findContactByNumber(string $cleanNumber): ?Contact
    {
        if (strlen($cleanNumber) < 10) return null;

        // ✅ VARIAÇÕES DO NÚMERO PARA BUSCA
        $variations = $this->getAllNumberVariations($cleanNumber);

        Log::debug('🔍 Buscando contato por variações', [
            'clean_number' => $this->maskNumber($cleanNumber),
            'variations_count' => count($variations)
        ]);

        // ✅ BUSCAR POR whatsapp_number
        $contact = Contact::whereIn('whatsapp_number', $variations)->first();
        if ($contact) {
            Log::debug('✅ Contato encontrado por whatsapp_number', [
                'contact_id' => $contact->id,
                'name' => $contact->first_name
            ]);
            return $contact;
        }

        // ✅ BUSCAR POR telefones relacionados
        $phone = Phone::whereIn('number', $variations)
            ->where('phoneable_type', Contact::class)
            ->with('phoneable')
            ->first();
            
        if ($phone && $phone->phoneable instanceof Contact) {
            Log::debug('✅ Contato encontrado por telefone relacionado', [
                'contact_id' => $phone->phoneable->id,
                'name' => $phone->phoneable->first_name
            ]);
            return $phone->phoneable;
        }

        Log::debug('❌ Contato não encontrado', [
            'clean_number' => $this->maskNumber($cleanNumber)
        ]);
        
        return null;
    }

    // ✅ GERAR VARIAÇÕES DO NÚMERO
    private function getAllNumberVariations(string $number): array
    {
        $clean = preg_replace('/[^0-9]/', '', $number);
        $variations = [$clean];

        // ✅ VARIAÇÕES COM/SEM CÓDIGO DO PAÍS (55)
        if (str_starts_with($clean, '55') && strlen($clean) >= 12) {
            $withoutCountry = substr($clean, 2);
            $variations[] = $withoutCountry;
        } elseif (!str_starts_with($clean, '55') && strlen($clean) >= 10) {
            $withCountry = '55' . $clean;
            $variations[] = $withCountry;
        }

        // ✅ VARIAÇÕES COM/SEM 9º DÍGITO
        foreach ($variations as $variation) {
            if (strlen($variation) >= 10) {
                if (str_starts_with($variation, '55')) {
                    $area = substr($variation, 2, 2);
                    $rest = substr($variation, 4);
                } else {
                    $area = substr($variation, 0, 2);
                    $rest = substr($variation, 2);
                }

                if (strlen($rest) == 9 && in_array($rest[0], ['6', '7', '8', '9'])) {
                    // Remover 9º dígito
                    $without9 = (str_starts_with($variation, '55') ? '55' : '') . $area . substr($rest, 1);
                    $variations[] = $without9;
                } elseif (strlen($rest) == 8 && in_array($rest[0], ['6', '7', '8', '9'])) {
                    // Adicionar 9º dígito
                    $with9 = (str_starts_with($variation, '55') ? '55' : '') . $area . '9' . $rest;
                    $variations[] = $with9;
                }
            }
        }

        return array_unique(array_filter($variations, function($v) {
            return strlen($v) >= 10;
        }));
    }

    // ✅ CRIAR NOVA CONVERSA PARA CONTATO EXISTENTE
    private function createNewConversationForContact($contact, EvolutionWebhook $webhook, string $normalizedRemoteJid, bool $isGroup, array $data, array $key): Conversation
    {
        $chatChannelId = $this->getChatChannelId($webhook->instance_name);
        $phoneNumber = $isGroup ? null : $this->extractPhoneFromJid($normalizedRemoteJid);

        $conversation = Conversation::create([
            'contact_id' => $contact->id,
            'chat_channel_id' => $chatChannelId,
            'whatsapp_chat_id' => str_replace(['@s.whatsapp.net', '@g.us'], '', $normalizedRemoteJid),
            'whatsapp_remote_jid' => $normalizedRemoteJid,
            'contact_name' => $contact->display_name ?? $contact->first_name,
            'contact_phone' => $phoneNumber,
            'type' => 'external',
            'status' => 'active',
            'evolution_instance_name' => $webhook->instance_name,
            'whatsapp_profile_name' => $contact->first_name,
            'whatsapp_status' => 'active',
            'last_whatsapp_activity' => now(),
            'created_by' => 1,
            'assigned_user_id' => null,
            'metadata' => [
                'is_group' => $isGroup,
                'created_from' => 'webhook',
                'whatsapp_type' => 'whatsapp'
            ]
        ]);

        $this->createSystemMessage($conversation, 'new_conversation_created', 'Nova conversa iniciada.');

        Log::info('✅ Nova conversa criada para contato existente', [
            'conversation_id' => $conversation->id,
            'contact_id' => $contact->id,
            'contact_name' => $contact->first_name,
            'remote_jid' => $this->maskJid($normalizedRemoteJid)
        ]);

        return $conversation;
    }

    // ✅ OBTER CHAT_CHANNEL_ID
    private function getChatChannelId(string $instanceName): int
    {
        $channelId = \Modules\Chat\Models\ChatChannel::where('type', 'evolution_api')
            ->where('is_active', true)
            ->whereRaw("JSON_EXTRACT(config, '$.instance_name') = ?", [$instanceName])
            ->value('id');
            
        return $channelId ?? 4; // Fallback
    }

    private function findOrCreateConversation(EvolutionWebhook $webhook, string $normalizedRemoteJid, ?string $cleanNumber, bool $isGroup, array $data, array $key, bool $fromMe): ?Conversation
    {
        // Buscar conversa existente
        $conversation = Conversation::where('evolution_instance_name', $webhook->instance_name)
            ->where(function ($query) use ($normalizedRemoteJid, $cleanNumber) {
                $query->where('whatsapp_remote_jid', $normalizedRemoteJid)
                    ->orWhere('whatsapp_chat_id', $normalizedRemoteJid);

                if ($cleanNumber && strlen($cleanNumber) >= 10) {
                    $query->orWhere('whatsapp_chat_id', $cleanNumber)
                        ->orWhere('whatsapp_remote_jid', $cleanNumber);
                }
            })
            ->first();

        if ($conversation) {
            return $conversation;
        }

        // Criar nova conversa
        $chatData = [
            'id' => $normalizedRemoteJid,
            'name' => $fromMe ? null : ($data['pushName'] ?? null),
            'isGroup' => $isGroup,
            'participant' => $key['participant'] ?? null,
            'groupName' => $isGroup ? $this->getGroupName($normalizedRemoteJid, $webhook->instance_name) : null,
            'senderName' => $isGroup && !$fromMe ? ($data['pushName'] ?? null) : null,
            'fromMe' => $fromMe,
        ];

        $chatSyncResult = $this->evolutionSyncService->syncSingleChat($chatData);
        $conversation = Conversation::find($chatSyncResult['conversation_id']);

        // ✅ VALIDAÇÃO: Para mensagens recebidas, deixar conversa sem usuário atribuído
        if ($conversation && !$fromMe) {
            $conversation->update(['assigned_user_id' => null]);
        }

        return $conversation;
    }

    // ✅ CORREÇÃO DO BUG - PASSAR $data COMPLETO
    private function createNewConversationFromClosed(Conversation $oldConversation, EvolutionWebhook $webhook, array $data, array $key, bool $isGroup): ?Conversation
    {
        try {
            // ✅ CORREÇÃO: PASSAR $data COMPLETO PARA normalizeJid
            $normalizedRemoteJid = $this->normalizeJid($key['remoteJid'] ?? '', $data);

            Log::info('🔧 Criando conversa a partir de fechada', [
                'old_conversation_id' => $oldConversation->id,
                'original_jid' => $this->maskJid($key['remoteJid'] ?? ''),
                'normalized_jid' => $this->maskJid($normalizedRemoteJid)
            ]);

            $chatData = [
                'id' => $normalizedRemoteJid,
                'name' => $data['pushName'] ?? $oldConversation->whatsapp_profile_name ?? null,
                'isGroup' => $isGroup,
                'participant' => $key['participant'] ?? null,
                'groupName' => $isGroup ? $this->getGroupName($normalizedRemoteJid, $webhook->instance_name) : null,
                'senderName' => $isGroup ? ($data['pushName'] ?? null) : null,
                'fromMe' => false,
            ];

            $chatSyncResult = $this->evolutionSyncService->syncSingleChat($chatData);
            $newConversation = Conversation::find($chatSyncResult['conversation_id']);

            if ($newConversation) {
                // Copiar informações relevantes, mas deixar assigned_user_id como null
                $updateData = [];
                if ($oldConversation->contact_id) {
                    $updateData['contact_id'] = $oldConversation->contact_id;
                }

                if (!empty($updateData)) {
                    $newConversation->update($updateData);
                }

                $this->createSystemMessage($newConversation, 'new_conversation_from_closed', 'Nova conversa iniciada.');
            }

            return $newConversation;
        } catch (Exception $e) {
            Log::error('Erro ao criar nova conversa', ['error' => $e->getMessage()]);
            return null;
        }
    }

    private function createMessage(Conversation $conversation, array $data, array $key, array $message, bool $isGroup, bool $fromMe): Message
    {
        $messageDataForSync = [
            'key' => $key,
            'message' => $message,
            'messageTimestamp' => $data['messageTimestamp'] ?? time(),
            'pushName' => $data['pushName'] ?? null,
            'status' => $data['status'] ?? null,
            'messageType' => $data['messageType'] ?? 'conversation',
            'instanceId' => $data['instanceId'] ?? null,
            'source' => $data['source'] ?? 'webhook',
            'base64' => $message['base64'] ?? null,
            'fromMe' => $fromMe,
            'participant' => $key['participant'] ?? null,
            'isGroup' => $isGroup,
            'groupName' => $isGroup ? $this->getGroupName($this->normalizeJid($key['remoteJid'] ?? '', $data), $data['instanceId'] ?? '') : null,
            'senderName' => $isGroup && !$fromMe ? ($data['pushName'] ?? null) : null
        ];

        return $this->evolutionSyncService->createMessageFromApiData($conversation, $messageDataForSync);
    }

    private function createSystemMessage(Conversation $conversation, string $systemType, string $content, array $metadata = []): void
    {
        try {
            Message::create([
                'conversation_id' => $conversation->id,
                'sender_type' => 'system',
                'sender_id' => null,
                'content' => $content,
                'message_type' => 'system',
                'is_from_contact' => false,
                'status' => 'sent',
                'sent_at' => now(),
                'metadata' => array_merge(['is_system_message' => true, 'system_type' => $systemType], $metadata)
            ]);

            $conversation->update(['last_message_at' => now()]);
        } catch (Exception $e) {
            // Silenciar erro
        }
    }

    protected function getGroupName(string $groupJid, string $instanceName): ?string
    {
        try {
            $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
            if (!$instance) return 'Grupo WhatsApp';

            $this->evolutionApi->setInstance($instance);
            $groupInfo = $this->evolutionApi->getGroupInfo($groupJid);
            return $groupInfo['subject'] ?? $groupInfo['name'] ?? 'Grupo WhatsApp';
        } catch (Exception $e) {
            // Fallback para banco local
            $conversation = Conversation::where('whatsapp_chat_id', $groupJid)->where('type', 'external')->first();
            return $conversation?->whatsapp_profile_name ?? 'Grupo WhatsApp';
        }
    }

    protected function determineIfFromMe(bool $originalFromMe, ?string $participant, string $instanceName): bool
    {
        if ($originalFromMe) return true;
        if (!$participant) return $originalFromMe;

        $participantNumber = preg_replace('/[^0-9]/', '', $participant);

        if (in_array($participantNumber, self::YOUR_LIDS)) {
            return true;
        }

        $instance = EvolutionInstance::where('instance_name', $instanceName)->first();
        if ($instance) {
            $instanceNumber = preg_replace('/[^0-9]/', '', $instance->whatsapp_number ?? '');
            $yourNumbers = [$instanceNumber, '556696322859', '6696322859'];

            foreach ($yourNumbers as $yourNumber) {
                if (!empty($yourNumber) && $participantNumber === $yourNumber) {
                    return true;
                }
            }
        }

        return $originalFromMe;
    }

    protected function processMessageUpdateEvent(EvolutionWebhook $webhook): array
    {
        try {
            $data = $webhook->payload['data'] ?? [];
            $whatsappMessageId = $data['keyId'] ?? $data['messageId'] ?? null;
            $newEvolutionStatus = $data['status'] ?? null;

            if (!$whatsappMessageId || !$newEvolutionStatus) {
                return ['action' => 'ignored', 'reason' => 'dados_insuficientes'];
            }

            $message = Message::where('whatsapp_message_id', $whatsappMessageId)->first();
            if (!$message) {
                return ['action' => 'message_not_found', 'whatsapp_message_id' => $whatsappMessageId];
            }

            $mappedNewStatus = $this->mapWhatsAppStatusToEnum($newEvolutionStatus);
            $currentDbStatus = $message->delivery_status;

            if ($mappedNewStatus === 'failed' && $currentDbStatus !== 'failed') {
                $this->updateMessageStatus($message, $mappedNewStatus);
                return ['action' => 'status_updated', 'message_id' => $message->id, 'old_status' => $currentDbStatus, 'new_status' => $mappedNewStatus];
            }

            if ($this->getStatusRank($mappedNewStatus) <= $this->getStatusRank($currentDbStatus)) {
                return ['action' => 'no_change', 'current_status' => $currentDbStatus];
            }

            $this->updateMessageStatus($message, $mappedNewStatus);
            return ['action' => 'status_updated', 'message_id' => $message->id, 'old_status' => $currentDbStatus, 'new_status' => $mappedNewStatus];
        } catch (Exception $e) {
            return ['action' => 'error', 'reason' => $e->getMessage()];
        }
    }

    private function updateMessageStatus(Message $message, string $newStatus): void
    {
        $message->update([
            'delivery_status' => $newStatus,
            'status' => $newStatus,
            'delivered_at' => ($newStatus === 'delivered' && !$message->delivered_at) ? now() : $message->delivered_at,
            'read_at' => ($newStatus === 'read' && !$message->read_at) ? now() : $message->read_at
        ]);

        try {
            $this->chatRedis->publishWhatsAppStatusUpdate($message, $message->getOriginal('delivery_status'), $newStatus);
        } catch (Exception $e) {
            // Silenciar erro de publicação
        }
    }

    protected function publishToFrontend(EvolutionWebhook $webhook, array $result): void
    {
        if (!$this->chatRedis->isRedisAvailable()) return;

        try {
            $eventData = [
                'original_webhook_event' => $webhook->event_type,
                'instance_name' => $webhook->instance_name,
                'result_action' => $result['action'] ?? 'processed',
                'timestamp' => now()->toISOString(),
                'webhook_db_id' => $webhook->id,
            ];

            if (in_array($webhook->event_type, ['messages', 'messages_upsert', 'messages.upsert']) && isset($result['message_id'])) {
                $message = Message::with(['conversation', 'attachments'])->find($result['message_id']);
                if ($message && $message->conversation) {
                    $this->publishMessageEvent($message, $eventData);
                }
            }

            if (in_array($webhook->event_type, ['messages_update', 'messages.update']) && isset($result['message_id'])) {
                $message = Message::find($result['message_id']);
                if ($message) {
                    $this->publishStatusEvent($message, $result, $eventData);
                }
            }
        } catch (Exception $e) {
            // Silenciar erro de publicação
        }
    }

    private function publishMessageEvent(Message $message, array $eventData): void
    {
        $conversation = $message->conversation;
        $isFromMe = $message->whatsapp_key_from_me;
        $messageMetadata = is_string($message->metadata) ? json_decode($message->metadata, true) : $message->metadata;
        $messageSource = $messageMetadata['source'] ?? 'unknown';

        $externalAppSources = ['desktop', 'android', 'ios', 'web'];
        
        if ($isFromMe && in_array($messageSource, $externalAppSources)) {
            $eventTypeToPublish = $this->chatRedis->getEventType('OUTBOUND_WHATSAPP_MESSAGE');
            $sourceIdentifier = 'whatsapp_external_app';
        } else {
            $eventTypeToPublish = $this->chatRedis->getEventType('NEW_WHATSAPP_MESSAGE');
            $sourceIdentifier = $isFromMe ? 'crm_sent_message' : 'whatsapp_incoming';
        }

        $eventData['event_type'] = $eventTypeToPublish;
        $eventData['source'] = $sourceIdentifier;
        $eventData['message'] = $message->toFrontendArray();
        $eventData['conversation'] = $conversation->toArray();
        $eventData['conversation_id'] = $conversation->id;

        $this->chatRedis->publishToStream($eventTypeToPublish, $eventData);
        $this->chatRedis->publishToStream($eventTypeToPublish, $eventData, "conversation_{$conversation->id}");
    }

    private function publishStatusEvent(Message $message, array $result, array $eventData): void
    {
        $eventData['event_type'] = $this->chatRedis->getEventType('WHATSAPP_STATUS_UPDATE');
        $eventData['message_id'] = $result['message_id'];
        $eventData['conversation_id'] = $message->conversation_id;
        $eventData['old_status'] = $result['old_status'] ?? null;
        $eventData['new_status'] = $result['new_status'] ?? null;

        $this->chatRedis->publishToStream($eventData['event_type'], $eventData);
        $this->chatRedis->publishToStream($eventData['event_type'], $eventData, "conversation_{$message->conversation_id}");
    }

    protected function dispatchContactPhotoJobIfNeeded(?Contact $contact, string $phoneNumber, Conversation $conversation): void
    {
        if (!$contact || !$conversation->chat_channel_id) return;
        
        if (!$contact->needsWhatsappProfilePhotoUpdate()) return;
        
        $cacheKey = "photo_job_pending_{$contact->id}";
        if (Cache::get($cacheKey, false)) return;

        Cache::put($cacheKey, true, 600);

        try {
            $cleanPhone = preg_replace('/[^0-9]/', '', $phoneNumber);
            
            if (strlen($cleanPhone) < 10) {
                $cleanPhone = $this->extractRealPhoneFromConversation($conversation) ?: $cleanPhone;
            }

            \Modules\Chat\Jobs\FetchContactPhotoJob::dispatch($contact, $cleanPhone, $conversation->id, false)
                ->onQueue('photos')
                ->delay(now()->addSeconds(5));
                
        } catch (Exception $e) {
            Cache::forget($cacheKey);
        }
    }

    private function extractRealPhoneFromConversation(Conversation $conversation): ?string
    {
        if ($conversation->whatsapp_remote_jid) {
            $phone = preg_replace('/[^0-9]/', '', $conversation->whatsapp_remote_jid);
            if (strlen($phone) >= 10) return $phone;
        }
        
        if ($conversation->contact_phone) {
            $phone = preg_replace('/[^0-9]/', '', $conversation->contact_phone);
            if (strlen($phone) >= 10) return $phone;
        }
        
        return null;
    }

    // ===== MÉTODOS AUXILIARES =====

    private function normalizeJid(string $jid, array $data = []): string
    {
        if (str_contains($jid, '@lid')) {
            Log::info('🔍 Processando LID', [
                'original_lid' => substr($jid, 0, 10) . '****'
            ]);
            
            // ✅ 1. TENTAR remoteJidAlt PRIMEIRO
            $key = $data['key'] ?? [];
            $remoteJidAlt = $key['remoteJidAlt'] ?? null;
            
            if ($remoteJidAlt && str_ends_with($remoteJidAlt, '@s.whatsapp.net')) {
                Log::info('✅ LID mapeado via remoteJidAlt', [
                    'original_lid' => substr($jid, 0, 10) . '****',
                    'mapped_to' => substr($remoteJidAlt, 0, 10) . '****'
                ]);
                return $remoteJidAlt;
            }
            
            // ✅ 2. BUSCAR CONVERSA EXISTENTE COM MESMO CONTATO
            $lidNumber = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
            
            // Buscar conversa existente que pode ter o número real
            $existingConversation = Conversation::where('whatsapp_chat_id', $lidNumber)
                ->orWhere('whatsapp_remote_jid', 'LIKE', "%{$lidNumber}%")
                ->whereNotNull('contact_id')
                ->with('contact')
                ->first();
                
            if ($existingConversation && $existingConversation->contact && $existingConversation->contact->whatsapp_number) {
                $realNumber = $existingConversation->contact->whatsapp_number;
                Log::info('✅ LID mapeado via conversa existente', [
                    'original_lid' => substr($jid, 0, 10) . '****',
                    'mapped_to' => substr($realNumber, 0, 4) . '****',
                    'contact_id' => $existingConversation->contact_id
                ]);
                return $realNumber . '@s.whatsapp.net';
            }
            
            // ✅ 3. BUSCAR POR pushName SE DISPONÍVEL
            $pushName = $data['pushName'] ?? null;
            if ($pushName) {
                $contactByName = Contact::where('first_name', 'LIKE', "%{$pushName}%")
                    ->whereNotNull('whatsapp_number')
                    ->first();
                    
                if ($contactByName) {
                    Log::info('✅ LID mapeado via pushName', [
                        'original_lid' => substr($jid, 0, 10) . '****',
                        'push_name' => $pushName,
                        'mapped_to' => substr($contactByName->whatsapp_number, 0, 4) . '****',
                        'contact_id' => $contactByName->id
                    ]);
                    return $contactByName->whatsapp_number . '@s.whatsapp.net';
                }
            }
            
            // ❌ 4. SE NÃO CONSEGUIR MAPEAR, RETORNAR O LID ORIGINAL
            Log::warning('⚠️ Não foi possível mapear LID para número real', [
                'original_lid' => substr($jid, 0, 10) . '****',
                'push_name' => $pushName
            ]);
            
            return $jid; // Manter o LID original para tratamento posterior
        }

        // ✅ RESTO DO CÓDIGO MANTIDO
        if (str_ends_with($jid, '@s.whatsapp.net') || str_ends_with($jid, '@g.us')) {
            return $jid;
        }

        $numbers = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
        if (strlen($numbers) >= 10) {
            return $numbers . '@s.whatsapp.net';
        }

        return $jid;
    }

    protected function normalizeEventType(string $eventType): string
    {
        // Buscar mapeamento direto
        foreach (self::EVENT_TYPE_MAPPING as $normalized => $variants) {
            if (in_array($eventType, $variants)) {
                return $normalized;
            }
        }

        // Fallback para normalização padrão
        return str_replace(['-', '.'], '_', strtolower($eventType));
    }

    private function getStatusRank(string $status): int
    {
        return self::STATUS_RANK[$status] ?? 0;
    }

    private function mapWhatsAppStatusToEnum(string $whatsappStatus): string
    {
        return self::WHATSAPP_STATUS_MAPPING[$whatsappStatus] ?? 'sent';
    }

    protected function updateWebhookRelationships(EvolutionWebhook $webhook, array $result): void
    {
        $updates = [];
        if (isset($result['message_id'])) $updates['message_id'] = $result['message_id'];
        if (isset($result['conversation_id'])) $updates['conversation_id'] = $result['conversation_id'];
        if (isset($result['contact_id'])) $updates['contact_id'] = $result['contact_id'];

        if (!empty($updates)) {
            $webhook->update($updates);
        }
    }

    protected function maskJid(string $jid): string
    {
        if (empty($jid)) return '';
        $parts = explode('@', $jid);
        $number = $parts[0] ?? '';
        $domain = $parts[1] ?? '';

        if (strlen($number) >= 8) {
            $masked = substr($number, 0, 4) . '****' . substr($number, -2);
            return $masked . ($domain ? "@{$domain}" : '');
        }

        return '****' . ($domain ? "@{$domain}" : '');
    }

    private function maskNumber(string $number): string
    {
        if (strlen($number) <= 4) return '****';
        return substr($number, 0, 4) . '****' . substr($number, -2);
    }

    private function extractPhoneFromJid(string $jid): string
    {
        $phone = preg_replace('/[^0-9]/', '', explode('@', $jid)[0]);
        if (strlen($phone) === 11 && !str_starts_with($phone, '55')) {
            $phone = '55' . $phone;
        }
        return $phone;
    }

    // ===== MÉTODOS DE EVENTOS SIMPLIFICADOS =====

        protected function processConnectionUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleConnectionStatusUpdate($webhook->payload['data'] ?? []);
    }

    protected function processQrCodeUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleQrCodeUpdate($webhook->payload['data'] ?? []);
    }

    protected function processInstanceStatusEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleInstanceStatusUpdate($webhook->payload['data'] ?? []);
    }

    protected function processChatUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleChatUpdateEvent($webhook->payload['data'] ?? []);
    }

    protected function processContactUpdateEvent(EvolutionWebhook $webhook): array
    {
        return $this->evolutionSyncService->handleContactUpdateEvent($webhook->payload['data'] ?? []);
    }
}