<?php
// modules/Chat/Services/FileService.php - VERSÃO COM VALIDAÇÃO DE ÁUDIO

namespace Modules\Chat\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Exception;

class FileService
{
    private const MAX_FILE_SIZE = 16 * 1024 * 1024; // 16MB
    private const MAX_AUDIO_DURATION = 180; // 3 minutos
    
    private const ALLOWED_EXTENSIONS = [
        'jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp',
        'mp4', 'avi', 'mov', 'wmv', 'webm', '3gp',
        'mp3', 'wav', 'ogg', 'aac', 'm4a', 'webm', // ✅ WEBM para áudio
        'pdf', 'doc', 'docx', 'txt', 'zip', 'rar'
    ];

    /**
     * ✅ PROCESSAR UPLOAD COM VALIDAÇÃO DE ÁUDIO
     */
    public function processUpload(UploadedFile $file): array
    {
        $this->validateFile($file);

        $originalName = $file->getClientOriginalName();
        $mimeType = $file->getMimeType();
        $size = $file->getSize();
        $type = $this->detectType($mimeType);

        // ✅ VALIDAÇÃO ESPECÍFICA PARA ÁUDIO
        if ($type === 'audio') {
            $this->validateAudioFile($file);
        }

        $safeName = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
        $path = $file->storeAs('chat-media', $safeName, 'public');
        $fullPath = Storage::disk('public')->path($path);

        // ✅ EXTRAIR DURAÇÃO DO ÁUDIO SE POSSÍVEL
        $duration = $type === 'audio' ? $this->getAudioDuration($fullPath) : null;

        Log::info('✅ Arquivo processado com sucesso', [
            'original_name' => $originalName,
            'stored_name' => $safeName,
            'type' => $type,
            'size_mb' => round($size / 1024 / 1024, 2),
            'duration' => $duration
        ]);

        return [
            'type' => $type,
            'original_name' => $originalName,
            'full_path' => $fullPath,
            'data' => [
                'file_name' => $originalName,
                'file_path' => $path,
                'file_type' => $type,
                'mime_type' => $mimeType,
                'file_size' => $size,
                'duration' => $duration,
                'download_status' => 'completed',
                'downloaded_at' => now(),
                'width' => $this->getImageWidth($fullPath, $type),
                'height' => $this->getImageHeight($fullPath, $type),
                'whatsapp_metadata' => $type === 'audio' ? ['is_voice_note' => true] : null
            ]
        ];
    }

    /**
     * ✅ VALIDAR ARQUIVO DE ÁUDIO
     */
    private function validateAudioFile(UploadedFile $file): void
    {
        $mimeType = $file->getMimeType();
        
        // ✅ MIME TYPES PERMITIDOS PARA ÁUDIO
        $allowedAudioMimes = [
            'audio/webm',
            'audio/ogg',
            'audio/mpeg',
            'audio/mp4',
            'audio/wav',
            'audio/aac',
            'audio/m4a'
        ];

        if (!in_array($mimeType, $allowedAudioMimes)) {
            throw new Exception("Formato de áudio não suportado: {$mimeType}");
        }

        // ✅ VALIDAR TAMANHO MÍNIMO (evitar arquivos vazios)
        if ($file->getSize() < 1024) { // 1KB mínimo
            throw new Exception('Arquivo de áudio muito pequeno');
        }

        Log::info('✅ Áudio validado', [
            'mime_type' => $mimeType,
            'size_kb' => round($file->getSize() / 1024, 2)
        ]);
    }

    /**
     * ✅ OBTER DURAÇÃO DO ÁUDIO (OPCIONAL)
     */
    private function getAudioDuration(string $filePath): ?int
    {
        try {
            // ✅ USANDO GETID3 SE DISPONÍVEL
            if (class_exists('getID3')) {
                $getID3 = new \getID3();
                $fileInfo = $getID3->analyze($filePath);
                return isset($fileInfo['playtime_seconds']) ? (int)$fileInfo['playtime_seconds'] : null;
            }

            // ✅ FALLBACK: TENTAR COM FFPROBE SE DISPONÍVEL
            if (function_exists('shell_exec')) {
                $command = "ffprobe -v quiet -show_entries format=duration -of csv=p=0 " . escapeshellarg($filePath);
                $output = shell_exec($command);
                if ($output && is_numeric(trim($output))) {
                    return (int)trim($output);
                }
            }

            return null;
        } catch (Exception $e) {
            Log::warning('⚠️ Não foi possível obter duração do áudio', [
                'file' => basename($filePath),
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    private function validateFile(UploadedFile $file): void
    {
        if (!$file->isValid()) {
            throw new Exception('Arquivo inválido ou corrompido');
        }

        if ($file->getSize() > self::MAX_FILE_SIZE) {
            $sizeMB = round($file->getSize() / 1024 / 1024, 2);
            throw new Exception("Arquivo muito grande: {$sizeMB}MB (máximo 16MB)");
        }

        $extension = strtolower($file->getClientOriginalExtension());
        if (!in_array($extension, self::ALLOWED_EXTENSIONS)) {
            throw new Exception("Tipo de arquivo não permitido: .{$extension}");
        }

        $mimeType = $file->getMimeType();
        if (empty($mimeType)) {
            throw new Exception('Não foi possível determinar o tipo do arquivo');
        }
    }

    private function detectType(string $mimeType): string
    {
        return match (true) {
            str_starts_with($mimeType, 'image/') => 'image',
            str_starts_with($mimeType, 'video/') => 'video',
            str_starts_with($mimeType, 'audio/') => 'audio',
            default => 'document'
        };
    }

    private function getImageWidth(string $filePath, string $type): ?int
    {
        if ($type !== 'image') return null;
        try {
            $imageInfo = getimagesize($filePath);
            return $imageInfo[0] ?? null;
        } catch (Exception $e) {
            return null;
        }
    }

    private function getImageHeight(string $filePath, string $type): ?int
    {
        if ($type !== 'image') return null;
        try {
            $imageInfo = getimagesize($filePath);
            return $imageInfo[1] ?? null;
        } catch (Exception $e) {
            return null;
        }
    }
}