<?php
// modules/Chat/Services/MessageService.php - VERSÃO ATUALIZADA PARA MÚLTIPLOS ANEXOS

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Exception;
use Modules\Chat\Models\Conversation;
use Modules\Chat\Models\Message;
use Modules\Chat\Models\MessageAttachment;
use Modules\Users\Models\User;
use Modules\Chat\Jobs\SendWhatsAppMessageJob;
use Illuminate\Http\UploadedFile; // ✅ NOVO IMPORT

class MessageService
{
    public function __construct(
        protected ChatRedisService $redis,
        protected EvolutionApiService $evolution,
        protected FileService $fileService
    ) {}

    /**
     * ✅ OBTER MENSAGENS - SEM CARREGAR SENDER VIA WITH
     */
    public function getMessages(int $conversationId, array $params): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        $perPage = min((int) ($params['per_page'] ?? 50), 100);

        $query = Message::where('conversation_id', $conversationId)
            ->with([
                'attachments' // ✅ Carregar tudo para o acessor 'url' funcionar
            ])
            ->select([
                'id',
                'conversation_id',
                'sender_id',
                'sender_type',
                'content',
                'message_type',
                'is_from_contact',
                'sent_at',
                'read_at',
                'status',
                'delivery_status',
                'metadata',
                'whatsapp_message_id'
            ]);

        if ($beforeId = $params['before_id'] ?? null) {
            $query->where('id', '<', $beforeId);
        }

        $messages = $query->orderByDesc('sent_at')->limit($perPage + 1)->get();
        $hasMore = $messages->count() > $perPage;

        if ($hasMore) $messages = $messages->take($perPage);
        $messages = $messages->reverse()->values();

        return [
            'data' => $messages->map(fn($m) => $this->formatMessage($m))->toArray(),
            'pagination' => [
                'has_more' => $hasMore,
                'before_id' => $messages->first()?->id
            ]
        ];
    }

    /**
     * ✅ ENVIAR MENSAGEM (TEXTO E/OU MÚLTIPLOS ANEXOS)
     */
    public function sendMessage(int $conversationId, array $data, array $files = []): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        $messageContent = trim($data['content'] ?? '');
        $uploadedFiles = $files['attachments'] ?? []; // Este é o array de UploadedFile

        if (empty($messageContent) && empty($uploadedFiles)) {
            throw new Exception('Conteúdo da mensagem ou anexos são obrigatórios.');
        }

        return DB::transaction(function () use ($conversation, $messageContent, $uploadedFiles) {
            $processedAttachmentsData = [];
            
            // ✅ 1. PROCESSAR CADA ARQUIVO UPLOADED INDIVIDUALMENTE
            if (!empty($uploadedFiles)) {
                foreach ($uploadedFiles as $uploadedFile) {
                    if ($uploadedFile instanceof UploadedFile) { // Garante que é um arquivo válido
                        try {
                            // Chama o FileService para cada arquivo
                            $processedAttachmentsData[] = $this->fileService->processUpload($uploadedFile);
                        } catch (Exception $e) {
                            Log::error('❌ MessageService: Erro ao processar anexo individual', [
                                'file_name' => $uploadedFile->getClientOriginalName(),
                                'error' => $e->getMessage()
                            ]);
                            throw new Exception("Falha ao processar anexo '{$uploadedFile->getClientOriginalName()}': {$e->getMessage()}");
                        }
                    } else {
                         Log::warning('❌ MessageService: Item no array de anexos não é UploadedFile', ['item' => $uploadedFile]);
                    }
                }
            }

            // ✅ 2. DETERMINAR TIPO PRINCIPAL DA MENSAGEM
            $primaryMessageType = 'text';
            if (!empty($processedAttachmentsData)) {
                // Se houver apenas um anexo, usa o tipo dele
                if (count($processedAttachmentsData) === 1) {
                    $primaryMessageType = $processedAttachmentsData[0]['type'];
                } else {
                    // Se múltiplos anexos, tenta inferir ou usa 'document'
                    $types = array_column($processedAttachmentsData, 'type');
                    if (count(array_unique($types)) === 1) { // Todos são do mesmo tipo
                        $primaryMessageType = $types[0];
                    } else { // Tipos mistos
                        $primaryMessageType = 'document';
                    }
                }
            }
            
            // ✅ 3. CRIAR A MENSAGEM PRINCIPAL
            $message = $conversation->messages()->create([
                'content' => $messageContent, // O texto principal da mensagem
                'sender_id' => Auth::id(),
                'sender_type' => User::class, // ✅ Corrigido para User::class
                'message_type' => $primaryMessageType,
                'is_from_contact' => false,
                'status' => 'sent', 
                'delivery_status' => 'pending', // Assume que vai para um serviço externo
                'metadata' => [
                    'source' => 'api',
                    'sender_name' => Auth::user()->name,
                    'attachment_count' => count($processedAttachmentsData),
                ]
            ]);
            
            // ✅ 4. CRIAR REGISTROS MessageAttachment PARA CADA ANEXO PROCESSADO
            $createdAttachments = [];
            foreach ($processedAttachmentsData as $pa) {
                $createdAttachments[] = $message->attachments()->create($pa['data']);
            }

            // ✅ 5. DESPACHAR JOBS PARA WHATSAPP SE NECESSÁRIO
            if ($conversation->isWhatsApp()) {
                $this->dispatchWhatsAppJobs($conversation, $message, $createdAttachments);
            }

            // ✅ 6. NOTIFICAR REDIS E RETORNAR
            $message->load('attachments'); // Recarrega com os anexos criados
            $this->redis->notifyNewMessage($message, $conversation);

            return [
                'success' => true,
                'message' => $this->formatMessage($message),
                'conversation_id' => $conversation->id
            ];
        });
    }

    /**
     * Criar mensagem de sistema
     */
    public function createSystemMessage(int $conversationId, string $content, string $systemType = 'system', array $metadata = []): array
    {
        $conversation = Conversation::findOrFail($conversationId);

        $message = Message::create([
            'conversation_id' => $conversationId,
            'sender_type' => 'system',
            'sender_id' => null,
            'content' => $content,
            'message_type' => 'system',
            'is_from_contact' => false,
            'status' => 'sent',
            'sent_at' => now(),
            'metadata' => array_merge([
                'is_system_message' => true,
                'system_type' => $systemType,
            ], $metadata)
        ]);

        $conversation->update(['last_message_at' => now()]);

        $this->redis->notifyNewMessage($message, $conversation);

        return [
            'success' => true,
            'message' => $this->formatMessage($message),
            'conversation_id' => $conversationId
        ];
    }

    /**
     * ✅ DESPACHA JOBS DE WHATSAPP PARA MENSAGEM E SEUS ANEXOS
     */
    private function dispatchWhatsAppJobs(Conversation $conversation, Message $message, array $attachments): void
    {
        // 1. Envia o texto da mensagem principal, se existir (sem anexo)
        if (!empty($message->content)) {
            SendWhatsAppMessageJob::dispatch(
                $message,
                $conversation,
                null // Sem anexo para a parte de texto
            );
        }

        // 2. Envia cada anexo individualmente
        foreach ($attachments as $attachment) {
            SendWhatsAppMessageJob::dispatch(
                $message,
                $conversation,
                $attachment // Passa o objeto MessageAttachment
            );
        }

        $message->update(['delivery_status' => 'pending', 'status' => 'sending']);
        Log::info('📋 Job(s) de WhatsApp despachado(s) com sucesso', [
            'message_id' => $message->id,
            'conversation_id' => $conversation->id,
            'attachment_count' => count($attachments)
        ]);
    }

    /**
     * ✅ MARCAR COMO LIDA
     */
    public function markAsRead(int $conversationId): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        $count = $conversation->messages()
            ->where('sender_id', '!=', Auth::id())
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        $conversation->update(['unread_count' => 0]);
        $this->redis->notifyConversationRead($conversationId, Auth::id());

        return ['success' => true, 'messages_marked' => $count];
    }

    /**
     * ✅ INDICADOR DE DIGITAÇÃO
     */
    public function handleTyping(int $conversationId, bool $isTyping): array
    {
        $conversation = Conversation::findOrFail($conversationId);
        $this->validateAccess($conversation);

        $this->redis->notifyUserTyping(Auth::id(), Auth::user()->name, $conversationId, $isTyping);

        return ['success' => true, 'is_typing' => $isTyping];
    }

    /**
     * ✅ VALIDAR ACESSO - SIMPLIFICADO
     */
    private function validateAccess(Conversation $conversation): void
    {
        if ($conversation->isInternal()) {
            if (!$conversation->participants()->where('user_id', Auth::id())->where('is_active', true)->exists()) {
                throw new Exception('Acesso negado à conversa');
            }
        }
        // TODO: Adicionar lógica de permissão para conversas externas (ex: assignedUser, permissão geral)
    }

    /**
     * ✅ FORMATAR MENSAGEM
     */
    private function formatMessage(Message $message): array
    {
        $senderData = $this->getSenderData($message);

        // ✅ GARANTIR QUE METADATA É SEMPRE UM ARRAY
        $metadata = $message->metadata;
        if (is_string($metadata)) {
            try {
                $metadata = json_decode($metadata, true) ?? [];
            } catch (\Exception $e) {
                $metadata = [];
            }
        } elseif (!is_array($metadata)) {
            $metadata = [];
        }

        return [
            'id' => $message->id,
            'conversation_id' => $message->conversation_id,
            'content' => $message->content,
            'message_type' => $message->message_type,
            'is_from_contact' => $message->is_from_contact,
            'sender_id' => $message->sender_id,
            'sender_name' => $senderData['name'],
            'sender_avatar' => $senderData['avatar'],
            'sent_at' => $message->sent_at?->toISOString(),
            'read_at' => $message->read_at?->toISOString(),
            'status' => $message->delivery_status ?? $message->status,
            'whatsapp_message_id' => $message->whatsapp_message_id,
            'attachments' => $message->attachments->map(fn($a) => [
                'id' => $a->id,
                'file_name' => $a->file_name,
                // 'file_path' => $a->file_path, // Removido para segurança e otimização
                'file_url' => $a->file_url, // ✅ Usa o acessor 'url' do model
                'file_type' => $a->file_type,
                'file_size' => $a->file_size,
                'mime_type' => $a->mime_type,
                'download_status' => $a->download_status,
                'width' => $a->width, // ✅ Adicionado
                'height' => $a->height, // ✅ Adicionado
                'duration' => $a->duration, // ✅ Adicionado
            ])->toArray(),
            'metadata' => array_merge($metadata, [
                'sender_avatar' => $senderData['avatar']
            ])
        ];
    }

    /**
     * ✅ OBTER DADOS DO REMETENTE
     */
    private function getSenderData(Message $message): array
    {
        if (!$message->is_from_contact && $message->sender_id) {
            try {
                $user = User::find($message->sender_id);
                if ($user) {
                    return [
                        'name' => $user->name,
                        'avatar' => $user->avatar_url
                    ];
                }
            } catch (\Exception $e) {
                Log::warning('Erro ao buscar dados do usuário', [
                    'user_id' => $message->sender_id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        if ($message->is_from_contact) {
            $conversation = $message->conversation;

            $metadata = $message->metadata;
            if (is_string($metadata)) {
                try {
                    $metadata = json_decode($metadata, true) ?? [];
                } catch (\Exception $e) {
                    $metadata = [];
                }
            } elseif (!is_array($metadata)) {
                $metadata = [];
            }

            return [
                'name' => $metadata['sender_name'] ?? $conversation->contact_name ?? 'Contato',
                'avatar' => $conversation->processed_avatar ?? $conversation->avatar_url // Supondo que conversation tenha um avatar_url
            ];
        }

        $metadata = $message->metadata;
        if (is_string($metadata)) {
            try {
                $metadata = json_decode($metadata, true) ?? [];
            } catch (\Exception $e) {
                $metadata = [];
            }
        } elseif (!is_array($metadata)) {
            $metadata = [];
        }

        return [
            'name' => $metadata['sender_name'] ?? 'Usuário',
            'avatar' => null
        ];
    }
}