<?php

namespace Modules\Chat\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Modules\Contacts\Models\Contact;
use Modules\Contacts\Models\Phone;
use Modules\Contacts\Enums\PhoneType;
use Exception;

class WhatsAppContactService
{
    /**
     * ✅ BUSCAR OU CRIAR CONTATO POR NÚMERO WHATSAPP
     */
    public function findOrCreateContact(string $whatsappNumber, ?string $pushName = null): ?Contact
    {
        try {
            $normalizedNumber = $this->normalizeNumber($whatsappNumber);
            
            if (!$normalizedNumber) {
                Log::warning('❌ Número WhatsApp inválido', [
                    'original' => $this->maskNumber($whatsappNumber)
                ]);
                return null;
            }

            Log::debug('🔍 Buscando contato', [
                'normalized_number' => $this->maskNumber($normalizedNumber),
                'push_name' => $pushName
            ]);

            // ✅ BUSCAR CONTATO EXISTENTE COM VARIAÇÕES
            $contact = $this->findExistingContactWithVariations($normalizedNumber);
            
            if ($contact) {
                Log::info('✅ Contato existente encontrado', [
                    'contact_id' => $contact->id,
                    'name' => $contact->display_name ?? $contact->full_name,
                    'number' => $this->maskNumber($normalizedNumber)
                ]);
                
                // ✅ ATUALIZAR NOME SE VIER pushName E CONTATO NÃO TIVER NOME PERSONALIZADO
                $this->updateContactIfNeeded($contact, $normalizedNumber, $pushName);
                return $contact;
            }

            // ✅ CRIAR NOVO CONTATO COM pushName
            return $this->createNewContactWithRelationship($normalizedNumber, $pushName);

        } catch (Exception $e) {
            Log::error('❌ Erro ao buscar/criar contato', [
                'number' => $this->maskNumber($whatsappNumber),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * ✅ BUSCAR CONTATO COM TODAS AS VARIAÇÕES POSSÍVEIS
     */
    private function findExistingContactWithVariations(string $normalizedNumber): ?Contact
    {
        // ✅ 1. BUSCAR NÚMERO EXATO PRIMEIRO
        $contact = Contact::where('whatsapp_number', $normalizedNumber)->first();
        if ($contact) {
            Log::debug('✅ Contato encontrado por whatsapp_number exato', ['contact_id' => $contact->id]);
            return $contact;
        }

        // ✅ 2. GERAR TODAS AS VARIAÇÕES POSSÍVEIS
        $variations = $this->getAllNumberVariations($normalizedNumber);
        
        Log::debug('🔍 Buscando por variações de número', [
            'variations_count' => count($variations),
            'variations' => array_map([$this, 'maskNumber'], $variations)
        ]);

        // ✅ 3. BUSCAR POR whatsapp_number COM VARIAÇÕES
        $contact = Contact::whereIn('whatsapp_number', $variations)->first();
        if ($contact) {
            Log::debug('✅ Contato encontrado por whatsapp_number (variação)', [
                'contact_id' => $contact->id,
                'found_number' => $this->maskNumber($contact->whatsapp_number)
            ]);
            return $contact;
        }

        // ✅ 4. BUSCAR POR TELEFONES RELACIONADOS
        $phone = Phone::whereIn('number', $variations)
            ->where('phoneable_type', Contact::class)
            ->with('phoneable')
            ->first();
            
        if ($phone && $phone->phoneable instanceof Contact) {
            Log::debug('✅ Contato encontrado por telefone relacionado', [
                'contact_id' => $phone->phoneable->id,
                'phone_number' => $this->maskNumber($phone->number),
                'phone_type' => $phone->type->name ?? 'unknown'
            ]);
            return $phone->phoneable;
        }

        // ✅ 5. BUSCAR COM LIKE (NÚMEROS FORMATADOS DIFERENTES)
        foreach ($variations as $variation) {
            $cleanVariation = preg_replace('/[^0-9]/', '', $variation);
            
            if (strlen($cleanVariation) >= 10) {
                // Buscar em whatsapp_number com LIKE
                $contact = Contact::where('whatsapp_number', 'LIKE', "%{$cleanVariation}%")->first();
                if ($contact) {
                    Log::debug('✅ Contato encontrado por LIKE whatsapp_number', [
                        'contact_id' => $contact->id,
                        'variation_used' => $this->maskNumber($cleanVariation)
                    ]);
                    return $contact;
                }
                
                // Buscar em telefones com LIKE
                $phone = Phone::where('number', 'LIKE', "%{$cleanVariation}%")
                    ->where('phoneable_type', Contact::class)
                    ->with('phoneable')
                    ->first();
                    
                if ($phone && $phone->phoneable instanceof Contact) {
                    Log::debug('✅ Contato encontrado por LIKE telefone', [
                        'contact_id' => $phone->phoneable->id,
                        'variation_used' => $this->maskNumber($cleanVariation)
                    ]);
                    return $phone->phoneable;
                }
            }
        }

        Log::debug('❌ Nenhum contato encontrado para todas as variações');
        return null;
    }

    /**
     * ✅ GERAR TODAS AS VARIAÇÕES POSSÍVEIS DO NÚMERO
     */
    private function getAllNumberVariations(string $number): array
    {
        $clean = preg_replace('/[^0-9]/', '', $number);
        $variations = [$clean, $number]; // Incluir original também

        // ✅ VARIAÇÕES COM/SEM CÓDIGO DO PAÍS (55)
        if (str_starts_with($clean, '55') && strlen($clean) >= 12) {
            // Tem 55, criar versão sem 55
            $withoutCountry = substr($clean, 2);
            $variations[] = $withoutCountry;
            
            // Versões formatadas
            $variations[] = '+55' . $withoutCountry;
            $variations[] = '+' . $clean;
        } elseif (!str_starts_with($clean, '55') && strlen($clean) >= 10) {
            // Não tem 55, criar versão com 55
            $withCountry = '55' . $clean;
            $variations[] = $withCountry;
            
            // Versões formatadas
            $variations[] = '+55' . $clean;
            $variations[] = '+' . $withCountry;
        }

        // ✅ VARIAÇÕES COM/SEM 9º DÍGITO (CELULARES)
        if (str_starts_with($clean, '55') && strlen($clean) >= 12) {
            $area = substr($clean, 2, 2);
            $restOfNumber = substr($clean, 4);
            
            // Se tem 9 dígitos no resto (13 total), criar versão sem 9º dígito
            if (strlen($restOfNumber) == 9 && in_array($restOfNumber[0], ['6', '7', '8', '9'])) {
                $withoutNinth = '55' . $area . substr($restOfNumber, 1);
                $variations[] = $withoutNinth;
                $variations[] = substr($withoutNinth, 2); // Sem 55 também
                $variations[] = '+' . $withoutNinth;
                $variations[] = '+55' . substr($withoutNinth, 2);
            }
            
            // Se tem 8 dígitos no resto (12 total), criar versão com 9º dígito
            if (strlen($restOfNumber) == 8 && in_array($restOfNumber[0], ['6', '7', '8', '9'])) {
                $withNinth = '55' . $area . '9' . $restOfNumber;
                $variations[] = $withNinth;
                $variations[] = substr($withNinth, 2); // Sem 55 também
                $variations[] = '+' . $withNinth;
                $variations[] = '+55' . substr($withNinth, 2);
            }
        } elseif (!str_starts_with($clean, '55') && strlen($clean) >= 10) {
            // Número sem código do país
            if (strlen($clean) >= 10) {
                $area = substr($clean, 0, 2);
                $restOfNumber = substr($clean, 2);
                
                // Se tem 9 dígitos no resto, criar versão sem 9º dígito
                if (strlen($restOfNumber) == 9 && in_array($restOfNumber[0], ['6', '7', '8', '9'])) {
                    $withoutNinth = $area . substr($restOfNumber, 1);
                    $variations[] = $withoutNinth;
                    $variations[] = '55' . $withoutNinth;
                    $variations[] = '+55' . $withoutNinth;
                }
                
                // Se tem 8 dígitos no resto, criar versão com 9º dígito
                if (strlen($restOfNumber) == 8 && in_array($restOfNumber[0], ['6', '7', '8', '9'])) {
                    $withNinth = $area . '9' . $restOfNumber;
                    $variations[] = $withNinth;
                    $variations[] = '55' . $withNinth;
                    $variations[] = '+55' . $withNinth;
                }
            }
        }

        // ✅ REMOVER DUPLICATAS E RETORNAR
        return array_unique(array_filter($variations, function($v) {
            return !empty($v) && strlen(preg_replace('/[^0-9]/', '', $v)) >= 10;
        }));
    }

    /**
     * ✅ ATUALIZAR CONTATO SE NECESSÁRIO
     */
    private function updateContactIfNeeded(Contact $contact, string $normalizedNumber, ?string $pushName): void
    {
        try {
            $updates = [];
            
            // ✅ ATUALIZAR whatsapp_number SE VAZIO
            if (empty($contact->whatsapp_number)) {
                $updates['whatsapp_number'] = $normalizedNumber;
            }
            
            // ✅ ATUALIZAR NOME SE VIER pushName E CONTATO FOR "GENÉRICO"
            if ($pushName && trim($pushName)) {
                $isGenericName = in_array(trim($contact->first_name), [
                    'Contato WhatsApp', 
                    'WhatsApp Contact', 
                    'Contato',
                    'Contact'
                ]) || empty(trim($contact->first_name));
                
                if ($isGenericName) {
                    $nameParts = explode(' ', trim($pushName), 2);
                    $updates['first_name'] = trim($nameParts[0]);
                    if (isset($nameParts[1])) {
                        $updates['last_name'] = trim($nameParts[1]);
                    }
                    
                    Log::info('✅ Nome atualizado com pushName', [
                        'contact_id' => $contact->id,
                        'old_name' => $contact->first_name,
                        'new_name' => $pushName
                    ]);
                }
            }
            
            // ✅ ATUALIZAR ÚLTIMA INTERAÇÃO
            if (!$contact->last_whatsapp_interaction || 
                $contact->last_whatsapp_interaction->lt(now()->subHour())) {
                $updates['last_whatsapp_interaction'] = now();
            }
            
            // ✅ ATIVAR OPT-IN
            if (!$contact->whatsapp_opt_in) {
                $updates['whatsapp_opt_in'] = true;
            }
            
            if (!empty($updates)) {
                $contact->update($updates);
                Log::debug('✅ Contato atualizado', [
                    'contact_id' => $contact->id,
                    'updates' => array_keys($updates)
                ]);
            }
            
        } catch (Exception $e) {
            Log::warning('⚠️ Erro ao atualizar contato', [
                'contact_id' => $contact->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * ✅ CRIAR NOVO CONTATO COM pushName
     */
    private function createNewContactWithRelationship(string $normalizedNumber, ?string $pushName): Contact
    {
        return DB::transaction(function () use ($normalizedNumber, $pushName) {
            try {
                Log::info('➕ Criando novo contato WhatsApp', [
                    'number' => $this->maskNumber($normalizedNumber),
                    'push_name' => $pushName
                ]);

                // ✅ USAR pushName SE DISPONÍVEL
                if ($pushName && trim($pushName)) {
                    $nameParts = explode(' ', trim($pushName), 2);
                    $firstName = trim($nameParts[0]);
                    $lastName = isset($nameParts[1]) ? trim($nameParts[1]) : '';
                } else {
                    $firstName = 'Contato WhatsApp';
                    $lastName = '';
                }

                // ✅ CRIAR CONTATO
                $contact = Contact::create([
                    'first_name' => $firstName,
                    'last_name' => $lastName,
                    'whatsapp_number' => $normalizedNumber,
                    'whatsapp_opt_in' => true,
                    'last_whatsapp_interaction' => now(),
                    'created_by' => $this->getCurrentUserId(),
                    'user_id' => $this->getDefaultOwnerId(),
                    'source_id' => $this->getWhatsAppSourceId(),
                ]);

                // ✅ CRIAR TELEFONE RELACIONADO
                $phone = $contact->phones()->create([
                    'type' => PhoneType::mobile,
                    'number' => $normalizedNumber,
                ]);

                Log::info('✅ Novo contato WhatsApp criado', [
                    'contact_id' => $contact->id,
                    'name' => $contact->full_name,
                    'number' => $this->maskNumber($normalizedNumber),
                    'phone_id' => $phone->id,
                    'used_push_name' => !empty($pushName)
                ]);

                return $contact;

            } catch (Exception $e) {
                Log::error('❌ Erro na criação de contato', [
                    'number' => $this->maskNumber($normalizedNumber),
                    'name' => $pushName,
                    'error' => $e->getMessage()
                ]);
                throw $e;
            }
        });
    }

    /**
     * ✅ NORMALIZAR NÚMERO (CORRIGIDO)
     */
    private function normalizeNumber(string $number): ?string
    {
        $clean = preg_replace('/[^0-9]/', '', $number);

        if (strlen($clean) < 10) {
            return null;
        }

        // ✅ SE JÁ TEM 55 E ESTÁ MUITO LONGO, PODE SER DUPLICAÇÃO
        if (str_starts_with($clean, '5555') && strlen($clean) > 13) {
            $clean = '55' . substr($clean, 4); // Remove duplicação
        }

        // ✅ SE TEM 55 NO INÍCIO E TAMANHO ADEQUADO, MANTER
        if (str_starts_with($clean, '55') && strlen($clean) >= 12 && strlen($clean) <= 13) {
            return $clean;
        }

        // ✅ SE NÃO TEM 55 E TEM 10-11 DÍGITOS, ADICIONAR 55
        if (!str_starts_with($clean, '55') && strlen($clean) >= 10 && strlen($clean) <= 11) {
            return '55' . $clean;
        }

        // ✅ SE TEM 55 MAS TAMANHO ESTRANHO, TENTAR CORRIGIR
        if (str_starts_with($clean, '55') && strlen($clean) > 13) {
            // Pegar apenas os primeiros 13 dígitos
            return substr($clean, 0, 13);
        }

        return $clean;
    }

    // ✅ MÉTODOS AUXILIARES MANTIDOS
    private function getCurrentUserId(): int
    {
        try {
            return auth()->check() ? auth()->id() : 1;
        } catch (Exception $e) {
            return 1;
        }
    }

    private function getDefaultOwnerId(): ?int
    {
        return $this->getCurrentUserId();
    }

    private function getWhatsAppSourceId(): ?int
    {
        try {
            $source = \Modules\Contacts\Models\Source::where('name', 'LIKE', '%WhatsApp%')
                ->orWhere('name', 'LIKE', '%whatsapp%')
                ->first();
                
            return $source?->id;
        } catch (Exception $e) {
            return null;
        }
    }

    private function maskNumber(string $number): string
    {
        if (strlen($number) <= 4) {
            return '****';
        }
        return substr($number, 0, 4) . '****' . substr($number, -2);
    }

    // ✅ MÉTODOS PÚBLICOS
    public function isValidWhatsAppNumber(string $number): bool
    {
        $normalized = $this->normalizeNumber($number);
        return $normalized !== null && strlen($normalized) >= 12;
    }

    public function getNormalizedNumber(string $number): ?string
    {
        return $this->normalizeNumber($number);
    }

    public function findContactByNumber(string $number): ?Contact
    {
        $normalized = $this->normalizeNumber($number);
        if (!$normalized) return null;
        
        return $this->findExistingContactWithVariations($normalized);
    }
}