<?php
// modules/Chat/database/migrations/xxxx_create_evolution_webhooks_table.php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('evolution_webhooks', function (Blueprint $table) {
            $table->id();
            
            // ✅ IDENTIFICAÇÃO
            $table->string('instance_name')
                ->comment('Nome da instância que enviou o webhook');
            
            $table->string('webhook_id')->nullable()
                ->comment('ID único do webhook (se fornecido pela Evolution)');
            
            // ✅ USAR STRING EM VEZ DE ENUM PARA FLEXIBILIDADE
            $table->string('event_type', 50)
                ->comment('Tipo do evento recebido da Evolution API');
            
            $table->string('event_subtype')->nullable()
                ->comment('Subtipo específico do evento');
            
            // ✅ DADOS DO WEBHOOK
            $table->json('payload')
                ->comment('Payload completo recebido do webhook');
            
            $table->json('headers')->nullable()
                ->comment('Headers HTTP do webhook');
            
            $table->string('source_ip')->nullable()
                ->comment('IP de origem do webhook');
            
            $table->text('user_agent')->nullable() // ✅ TEXT para user-agents longos
                ->comment('User-Agent do webhook');
            
            // ✅ PROCESSAMENTO
            $table->enum('status', [
                'pending', 'processing', 'processed', 'failed', 'ignored', 'duplicate'
            ])->default('pending')
                ->comment('Status do processamento');
            
            $table->timestamp('received_at')->useCurrent()
                ->comment('Quando o webhook foi recebido');
            
            $table->timestamp('processed_at')->nullable()
                ->comment('Quando foi processado');
            
            $table->integer('processing_attempts')->default(0)
                ->comment('Número de tentativas de processamento');
            
            $table->timestamp('next_attempt_at')->nullable()
                ->comment('Próxima tentativa de processamento');
            
            // ✅ RESULTADOS E ERROS
            $table->json('processing_result')->nullable()
                ->comment('Resultado do processamento');
            
            $table->text('error_message')->nullable()
                ->comment('Mensagem de erro se falhou');
            
            $table->json('error_details')->nullable()
                ->comment('Detalhes técnicos do erro');
            
            $table->string('error_code')->nullable()
                ->comment('Código do erro');
            
            // ✅ RELACIONAMENTOS CRIADOS (SEM FOREIGN KEYS POR ENQUANTO)
            $table->unsignedBigInteger('conversation_id')->nullable()
                ->comment('ID da conversa relacionada (se aplicável)');
            
            $table->unsignedBigInteger('message_id')->nullable()
                ->comment('ID da mensagem relacionada (se aplicável)');
            
            $table->unsignedBigInteger('contact_id')->nullable()
                ->comment('ID do contato relacionado (se aplicável)');
            
            // ✅ METADADOS WHATSAPP
            $table->string('whatsapp_chat_id')->nullable()
                ->comment('ID do chat WhatsApp relacionado');
            
            $table->string('whatsapp_message_id')->nullable()
                ->comment('ID da mensagem WhatsApp relacionada');
            
            $table->text('whatsapp_remote_jid')->nullable() // ✅ TEXT para JIDs longos
                ->comment('JID do contato relacionado');
            
            $table->boolean('is_duplicate')->default(false)
                ->comment('Indica se é um webhook duplicado');
            
            $table->string('duplicate_hash', 64)->nullable() // ✅ Tamanho específico para hash MD5
                ->comment('Hash para detectar duplicatas');
            
            $table->integer('priority')->default(5)
                ->comment('Prioridade de processamento (1 = mais alta)');
            
            $table->timestamps();
            $table->softDeletes();
            
            // ✅ ÍNDICES PARA PERFORMANCE
            $table->index(['instance_name', 'event_type'], 'idx_instance_event');
            $table->index(['status', 'priority', 'next_attempt_at'], 'idx_processing_queue');
            $table->index(['received_at', 'status'], 'idx_received_status');
            $table->index(['whatsapp_chat_id', 'whatsapp_message_id'], 'idx_whatsapp_ids');
            $table->index(['duplicate_hash'], 'idx_duplicate_hash');
            $table->index(['conversation_id', 'message_id'], 'idx_related_entities');
            $table->index(['event_type', 'processed_at'], 'idx_event_processed');
            $table->index(['instance_name', 'whatsapp_chat_id'], 'idx_instance_chat');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('evolution_webhooks');
    }
};