// modules/Chat/resources/js/composables/useChatAttachments.js
import { ref, computed } from 'vue'

export function useChatAttachments(options = {}) {
  const {
    maxAnexos = 5,
    maxFileSize = 16 * 1024 * 1024, // 16MB
    allowedFileTypes = [
      'image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/bmp', 'image/svg+xml',
      'application/pdf',
      'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      'text/plain',
      'application/zip', 'application/x-rar-compressed',
      'audio/mpeg', 'audio/wav', 'audio/ogg', 'audio/m4a', 'audio/aac', 'audio/flac', 'audio/webm', 
      'video/mp4', 'video/webm', 'video/ogg'
    ],
    onStatusUpdate = null
  } = options

  // ===== ESTADO =====
  const anexos = ref([])
  const inputAnexo = ref(null)
  const modalImagem = ref({ mostrar: false, anexo: null })

  // ===== COMPUTED =====
  const hasAttachments = computed(() => anexos.value.length > 0)
  const canAddMore = computed(() => anexos.value.length < maxAnexos)
  const attachmentCount = computed(() => anexos.value.length)

  const acceptedFileTypes = computed(() => {
    return 'image/*,application/pdf,.doc,.docx,.txt,.zip,.rar,.mp3,.mp4,.wav,.ogg,.webm'
  })

  const attachButtonTitle = computed(() => {
    if (anexos.value.length >= maxAnexos) return `Máximo de ${maxAnexos} anexos atingido`
    return `Anexar arquivo (${anexos.value.length}/${maxAnexos})`
  })

  const attachButtonLabel = computed(() => 'Anexar arquivo')

  // ===== VALIDAÇÃO =====
  function validarArquivo(file) {
    const errors = []
    
    // Validar tamanho
    if (file.size > maxFileSize) {
      errors.push(`Arquivo muito grande. Máximo: ${formatFileSize(maxFileSize)}`)
    }
    
    // Validar tipo MIME
    const mimeType = (file.type || '').split(';')[0]
    if (!allowedFileTypes.includes(mimeType)) {
      errors.push(`Tipo de arquivo não permitido: ${mimeType}`)
    }
    
    // Validações específicas para áudio
    if (mimeType.startsWith('audio/') || mimeType === 'video/webm') {
      const audioErrors = validarArquivoAudio(file)
      errors.push(...audioErrors)
    }
    
    return errors
  }

  function validarArquivoAudio(file) {
    const errors = []
    
    // Validar se é realmente áudio
    if (!file.type.startsWith('audio/') && !file.type.startsWith('video/webm')) {
      errors.push('Arquivo não é um áudio válido')
      return errors
    }
    
    // Validar tamanho para áudio (máximo 16MB para WhatsApp)
    const maxAudioSize = 16 * 1024 * 1024 // 16MB
    if (file.size > maxAudioSize) {
      errors.push(`Áudio muito grande. Máximo: ${formatFileSize(maxAudioSize)}`)
    }
    
    return errors
  }

  // ===== MÉTODOS PRINCIPAIS =====
  function abrirSeletorArquivo() {
    if (!canAddMore.value) return
    inputAnexo.value?.click()
  }

  function handleFileSelect(event) {
    const files = Array.from(event.target.files || [])
    files.forEach(file => adicionarAnexo(file))
    if (inputAnexo.value) inputAnexo.value.value = ''
  }

  function adicionarAnexo(file) {
    try {
      if (!canAddMore.value) {
        showStatus('warning', `Máximo de ${maxAnexos} anexos permitidos`, 'ExclamationTriangle')
        return
      }
      
      // Validar arquivo
      const erros = validarArquivo(file)
      if (erros.length > 0) {
        erros.forEach(erro => showStatus('error', erro, 'XCircle'))
        return
      }
      
      // Verificar duplicatas
      if (anexos.value.some(anexo => anexo.name === file.name && anexo.size === file.size)) {
        showStatus('warning', 'Arquivo já adicionado', 'ExclamationTriangle')
        return
      }

      const mimeType = (file.type || '').split(';')[0]

      const anexo = {
        file,
        name: file.name,
        type: mimeType,
        size: file.size,
        preview: null,
        id: Date.now() + Math.random(),
        isAudio: mimeType.startsWith('audio/') || mimeType === 'video/webm'
      }

      // Gerar preview para imagens
      if (isImageFile(anexo)) {
        const reader = new FileReader()
        reader.onload = (e) => { anexo.preview = e.target.result }
        reader.onerror = () => { console.warn('Erro ao gerar preview da imagem:', anexo.name) }
        reader.readAsDataURL(file)
      }

      anexos.value.push(anexo)
      console.log('Anexo adicionado:', { name: anexo.name, type: anexo.type, size: formatFileSize(anexo.size) })
      showStatus('success', `Anexo adicionado: ${truncateFileName(anexo.name, 25)}`, 'CheckCircle')

    } catch (error) {
      console.error('Erro ao adicionar anexo:', error)
      showStatus('error', `Erro ao adicionar anexo: ${error.message}`, 'XCircle')
    }
  }

  function removerAnexo(index) {
    if (index >= 0 && index < anexos.value.length) {
      const anexo = anexos.value[index]
      anexos.value.splice(index, 1)
      console.log('Anexo removido:', anexo.name)
      showStatus('info', `Anexo removido: ${truncateFileName(anexo.name, 25)}`, 'InformationCircle')
    }
  }

  function limparAnexos() {
    anexos.value = []
    modalImagem.value = { mostrar: false, anexo: null }
  }

  // ===== MODAL DE IMAGEM =====
  function abrirModalImagem(anexo) {
    if (!anexo || !anexo.preview) {
      console.error('Anexo inválido para modal de imagem:', anexo)
      return
    }
    if (!isImageFile(anexo)) {
      console.error('Arquivo não é uma imagem para preview:', anexo.type)
      return
    }
    modalImagem.value = { mostrar: true, anexo }
    console.log('Modal de imagem aberto para anexo', anexo.name)
  }

  function fecharModalImagem() {
    modalImagem.value = { mostrar: false, anexo: null }
  }

  // ===== HELPERS =====
  function isImageFile(anexo) {
    return !!anexo?.type && anexo.type.startsWith('image/')
  }

  function getFileIcon(type) {
    const icons = {
      'application/pdf': 'DocumentText',
      'application/msword': 'Document',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document': 'Document',
      'text/plain': 'DocumentText',
      'application/zip': 'ArchiveBox',
      'application/x-rar-compressed': 'ArchiveBox',
      'audio/mpeg': 'VolumeUp', 'audio/wav': 'VolumeUp', 'audio/ogg': 'VolumeUp', 'audio/m4a': 'VolumeUp', 
      'audio/aac': 'VolumeUp', 'audio/flac': 'VolumeUp', 'audio/webm': 'VolumeUp', 
      'video/mp4': 'Camera', 'video/webm': 'Camera', 'video/ogg': 'Camera',
    }
    if (icons[type]) return icons[type]
    if (type.startsWith('image/')) return 'Photograph'
    if (type.startsWith('video/')) return 'Camera'
    if (type.startsWith('audio/')) return 'VolumeUp'
    return 'Document'
  }

  function formatFileSize(bytes) {
    if (!bytes || bytes <= 0) return '0 B'
    const k = 1024
    const sizes = ['B', 'KB', 'MB', 'GB']
    const i = Math.floor(Math.log(bytes) / Math.log(k))
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i]
  }

  function truncateFileName(name, maxLength) {
    if (name.length <= maxLength) return name
    const extension = name.split('.').pop()
    const nameWithoutExt = name.substring(0, name.lastIndexOf('.'))
    const truncated = nameWithoutExt.substring(0, maxLength - extension.length - 4) + '...'
    return truncated + '.' + extension
  }

  function showStatus(tipo, mensagem, icone) {
    if (onStatusUpdate) {
      onStatusUpdate(tipo, mensagem, icone)
    }
  }

  // ===== CLEANUP =====
  function cleanup() {
    anexos.value = []
    modalImagem.value = { mostrar: false, anexo: null }
  }

  return {
    // Estado
    anexos,
    inputAnexo,
    modalImagem,
    
    // Computed
    hasAttachments,
    canAddMore,
    attachmentCount,
    acceptedFileTypes,
    attachButtonTitle,
    attachButtonLabel,
    
    // Métodos
    abrirSeletorArquivo,
    handleFileSelect,
    adicionarAnexo,
    removerAnexo,
    limparAnexos,
    abrirModalImagem,
    fecharModalImagem,
    
    // Helpers
    isImageFile,
    getFileIcon,
    formatFileSize,
    truncateFileName,
    
    // Cleanup
    cleanup
  }
}