// modules/Chat/resources/js/composables/useChatAudioRecording.js
import { ref, computed } from 'vue'

export function useChatAudioRecording(options = {}) {
  const {
    maxTempoGravacao = 180, // 3 minutos
    onAudioRecorded = null,
    onStatusUpdate = null
  } = options

  // ===== ESTADO =====
  const gravandoAudio = ref(false)
  const mediaRecorderRef = ref(null)
  const audioChunks = ref([])
  const tempoGravacao = ref(0)
  let timerGravacaoInterval = null

  // ===== COMPUTED =====
  const suportaGravacaoAudio = computed(() => {
    return typeof window !== 'undefined' &&
      !!navigator.mediaDevices &&
      typeof MediaRecorder !== 'undefined'
  })

  const tempoGravacaoFormatado = computed(() => {
    const total = tempoGravacao.value || 0
    const m = String(Math.floor(total / 60)).padStart(2, '0')
    const s = String(total % 60).padStart(2, '0')
    return `${m}:${s}`
  })

  const titleBotaoGravar = computed(() => {
    if (!suportaGravacaoAudio.value) return 'Gravação de áudio não suportada neste navegador'
    if (gravandoAudio.value) return 'Parar gravação de áudio'
    return 'Gravar áudio e enviar como anexo'
  })

  // ===== MÉTODOS =====
  function iniciarTimerGravacao() {
    pararTimerGravacao()
    tempoGravacao.value = 0

    timerGravacaoInterval = setInterval(() => {
      tempoGravacao.value += 1
      if (tempoGravacao.value >= maxTempoGravacao) {
        console.log('Tempo máximo de gravação atingido, parando...')
        pararGravacaoAudio()
      }
    }, 1000)
  }

  function pararTimerGravacao() {
    if (timerGravacaoInterval) {
      clearInterval(timerGravacaoInterval)
      timerGravacaoInterval = null
    }
  }

  async function iniciarGravacaoAudio() {
    if (!suportaGravacaoAudio.value) return

    try {
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true })
      const options = MediaRecorder.isTypeSupported?.('audio/webm') ? { mimeType: 'audio/webm' } : {}

      const mediaRecorder = new MediaRecorder(stream, options)
      mediaRecorderRef.value = mediaRecorder
      audioChunks.value = []

      mediaRecorder.ondataavailable = (event) => {
        if (event.data && event.data.size > 0) {
          audioChunks.value.push(event.data)
        }
      }

      mediaRecorder.onstop = () => {
        pararTimerGravacao()
        gravandoAudio.value = false
        stream.getTracks().forEach(track => track.stop())

        const mimeType = mediaRecorder.mimeType || 'audio/webm'
        const blob = new Blob(audioChunks.value, { type: mimeType })

        // Validações
        if (!blob || blob.size === 0) {
          console.warn('Gravação vazia')
          showStatus('warning', 'Gravação vazia', 'ExclamationTriangle')
          return
        }
        
        if (blob.size < 1000) {
          console.warn('Gravação muito pequena')
          showStatus('warning', 'Gravação muito curta', 'ExclamationTriangle')
          return
        }
        
        if (tempoGravacao.value < 1) {
          console.warn('Gravação muito curta em tempo')
          showStatus('warning', 'Gravação muito curta (mínimo 1 segundo)', 'ExclamationTriangle')
          return
        }

        const ext = mimeType.includes('ogg') ? 'ogg' : 'webm'
        const fileName = `audio-${new Date().toISOString().replace(/[:.]/g, '-')}.${ext}`
        const file = new File([blob], fileName, { type: mimeType })

        console.log('Áudio gravado:', {
          duration: `${tempoGravacao.value}s`,
          size: formatFileSize(file.size),
          type: mimeType
        })
        
        if (onAudioRecorded) {
          onAudioRecorded(file)
        }
        
        showStatus('success', `Áudio gravado: ${tempoGravacao.value}s`, 'CheckCircle')
      }

      mediaRecorder.start()
      gravandoAudio.value = true
      iniciarTimerGravacao()
      console.log('Gravação de áudio iniciada')

    } catch (error) {
      console.error('Erro ao iniciar gravação de áudio:', error)
      showStatus('error', 'Não foi possível acessar o microfone', 'XCircle')
    }
  }

  async function pararGravacaoAudio() {
    if (!gravandoAudio.value || !mediaRecorderRef.value) return
    
    try {
      const recorder = mediaRecorderRef.value
      
      // Verificar se gravação é muito curta
      if (tempoGravacao.value < 1) {
        console.warn('Gravação muito curta, cancelando...')
        pararTimerGravacao()
        gravandoAudio.value = false
        
        // Parar tracks do microfone
        if (recorder.stream) {
          recorder.stream.getTracks().forEach(track => track.stop())
        }
        
        showStatus('warning', 'Gravação muito curta (mínimo 1 segundo)', 'ExclamationTriangle')
        return
      }
      
      if (recorder.state !== 'inactive') {
        recorder.stop()
      } else {
        pararTimerGravacao()
        gravandoAudio.value = false
      }
    } catch (error) {
      console.error('Erro ao parar gravação de áudio:', error)
      pararTimerGravacao()
      gravandoAudio.value = false
      showStatus('error', 'Erro ao parar gravação', 'XCircle')
    }
  }

  async function toggleGravacaoAudio() {
    if (gravandoAudio.value) {
      await pararGravacaoAudio()
    } else {
      await iniciarGravacaoAudio()
    }
  }

  // ===== HELPERS =====
  function formatFileSize(bytes) {
    if (!bytes || bytes <= 0) return '0 B'
    const k = 1024
    const sizes = ['B', 'KB', 'MB', 'GB']
    const i = Math.floor(Math.log(bytes) / Math.log(k))
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i]
  }

  function showStatus(tipo, mensagem, icone) {
    if (onStatusUpdate) {
      onStatusUpdate(tipo, mensagem, icone)
    }
  }

  // ===== CLEANUP =====
  function cleanup() {
    pararGravacaoAudio()
    pararTimerGravacao()
  }

  return {
    // Estado
    gravandoAudio,
    tempoGravacao,
    suportaGravacaoAudio,
    tempoGravacaoFormatado,
    titleBotaoGravar,
    
    // Métodos
    iniciarGravacaoAudio,
    pararGravacaoAudio,
    toggleGravacaoAudio,
    cleanup
  }
}