// modules/Chat/resources/js/composables/useChatInput.js
import { ref, computed, nextTick } from 'vue'

export function useChatInput(options = {}) {
  const {
    maxCaracteres = 4000,
    isConversationClosed = ref(false),
    websocketStatus = ref({ isConnected: false })
  } = options

  // ===== ESTADO =====
  const mensagemTexto = ref('')
  const textareaRef = ref(null)
  const isFocused = ref(false)

  // ===== COMPUTED =====
  const showCharacterCount = computed(() => {
    return mensagemTexto.value.length > maxCaracteres * 0.8
  })

  const characterCountClass = computed(() => {
    return {
      'limite-proximo': mensagemTexto.value.length > maxCaracteres * 0.9,
      'limite-critico': mensagemTexto.value.length > maxCaracteres * 0.95
    }
  })

  const placeholder = computed(() => {
    if (isConversationClosed.value) return 'Esta conversa foi finalizada'
    if (!websocketStatus.value.isConnected) return 'Conectando...'
    return 'Digite sua mensagem... (Ctrl+Enter para enviar)'
  })

  const textareaLabel = computed(() => {
    if (isConversationClosed.value) return 'Conversa finalizada'
    return 'Digite sua mensagem'
  })

  const hasContent = computed(() => {
    return mensagemTexto.value.trim().length > 0
  })

  // ===== MÉTODOS =====
  function ajustarAlturaTextarea() {
    if (!textareaRef.value) return
    
    const textarea = textareaRef.value
    textarea.style.height = 'auto'
    const scrollHeight = textarea.scrollHeight
    const maxHeight = 120
    textarea.style.height = Math.min(scrollHeight, maxHeight) + 'px'
    textarea.style.overflowY = scrollHeight > maxHeight ? 'auto' : 'hidden'
  }

  function focusTextarea() {
    if (!isConversationClosed.value && textareaRef.value) {
      textareaRef.value.focus()
    }
  }

  function inserirTexto(texto, posicao = null) {
    if (isConversationClosed.value || !textareaRef.value) return

    const textarea = textareaRef.value
    const start = posicao?.start ?? textarea.selectionStart
    const end = posicao?.end ?? textarea.selectionEnd
    const currentText = mensagemTexto.value

    mensagemTexto.value = currentText.substring(0, start) + texto + currentText.substring(end)

    nextTick(() => {
      textarea.focus()
      const newPosition = start + texto.length
      textarea.setSelectionRange(newPosition, newPosition)
      ajustarAlturaTextarea()
    })
  }

  function limparTexto() {
    mensagemTexto.value = ''
    nextTick(() => {
      ajustarAlturaTextarea()
      focusTextarea()
    })
  }

  function handleFocus() {
    isFocused.value = true
  }

  function handleBlur() {
    isFocused.value = false
  }

  function handleInput() {
    ajustarAlturaTextarea()
  }

  function handlePaste(event, onFileDetected) {
    if (isConversationClosed.value) return

    const items = event.clipboardData?.items
    if (!items) return

    for (let i = 0; i < items.length; i++) {
      const item = items[i]
      if (item.type.indexOf('image') !== -1) {
        event.preventDefault()
        const file = item.getAsFile()
        if (file && onFileDetected) {
          onFileDetected(file)
        }
      }
    }
  }

  // ===== CLEANUP =====
  function cleanup() {
    mensagemTexto.value = ''
    isFocused.value = false
  }

  return {
    // Estado
    mensagemTexto,
    textareaRef,
    isFocused,
    
    // Computed
    showCharacterCount,
    characterCountClass,
    placeholder,
    textareaLabel,
    hasContent,
    
    // Métodos
    ajustarAlturaTextarea,
    focusTextarea,
    inserirTexto,
    limparTexto,
    handleFocus,
    handleBlur,
    handleInput,
    handlePaste,
    cleanup
  }
}