// modules/Chat/resources/js/composables/useChatTyping.js
import { ref, computed } from 'vue'
import chatStore from '../../servicos/chatStore.js'

export function useChatTyping(options = {}) {
  const {
    conversationId = ref(null),
    isConversationClosed = ref(false)
  } = options

  // ===== ESTADO =====
  const digitando = ref(false)
  let typingTimeout = null

  // ===== COMPUTED =====
  const usuariosDigitandoTexto = computed(() => {
    if (!conversationId.value) return ''
    
    const digitacao = chatStore.usuariosDigitando.value.get(conversationId.value)
    if (!digitacao || !Array.isArray(digitacao.users) || digitacao.users.length === 0) return ''
    
    const ws = getWebSocketService()
    const currentUserId = parseInt(ws?.getCurrentUser?.()?.id)
    
    const activeUsers = digitacao.users.filter(u => parseInt(u.id) !== currentUserId)
    if (activeUsers.length === 0) return ''
    
    const agora = Date.now()
    const recentActiveUsers = activeUsers.filter(u => (agora - (digitacao.timestamp || 0)) < 5000)
    const count = recentActiveUsers.length
    
    if (count === 0) return ''
    if (count === 1) return `${recentActiveUsers[0].name} está digitando...`
    if (count === 2) return `${recentActiveUsers[0].name} e ${recentActiveUsers[1].name} estão digitando...`
    return `${count} pessoas estão digitando...`
  })

  // ===== HELPERS =====
  function getWebSocketService() {
    return window.websocketService
  }

  // ===== MÉTODOS =====
  async function iniciarDigitacao() {
    if (digitando.value || isConversationClosed.value || !conversationId.value) return

    try {
      digitando.value = true
      
      const ws = getWebSocketService()
      if (!ws || !ws.isValid()) {
        console.warn('WebSocket não disponível para digitação')
        return
      }

      const success = ws.sendTypingIndicator(conversationId.value, true)
      if (success) {
        console.log('Indicador de digitação enviado via WebSocket')
      } else {
        console.warn('Falha ao enviar indicador de digitação')
      }

      // Auto-parar após 3 segundos
      if (typingTimeout) {
        clearTimeout(typingTimeout)
      }
      
      typingTimeout = setTimeout(() => {
        pararDigitacao()
      }, 3000)

    } catch (error) {
      console.error('Erro ao iniciar digitação:', error)
      digitando.value = false
    }
  }

  async function pararDigitacao() {
    if (!digitando.value) return

    try {
      digitando.value = false
      
      if (typingTimeout) {
        clearTimeout(typingTimeout)
        typingTimeout = null
      }

      const ws = getWebSocketService()
      if (!ws || !ws.isValid()) {
        return
      }

      if (conversationId.value) {
        ws.sendTypingIndicator(conversationId.value, false)
        console.log('Indicador de digitação parado via WebSocket')
      }

    } catch (error) {
      console.error('Erro ao parar digitação:', error)
    }
  }

  function handleInput(hasContent) {
    if (typingTimeout) {
      clearTimeout(typingTimeout)
    }
    
    if (hasContent) {
      if (!digitando.value) {
        iniciarDigitacao()
      } else {
        // Renovar timeout
        typingTimeout = setTimeout(() => {
          pararDigitacao()
        }, 3000)
      }
    } else {
      pararDigitacao()
    }
  }

  // ===== CLEANUP =====
  function cleanup() {
    pararDigitacao()
    if (typingTimeout) {
      clearTimeout(typingTimeout)
      typingTimeout = null
    }
  }

  return {
    // Estado
    digitando,
    usuariosDigitandoTexto,
    
    // Métodos
    iniciarDigitacao,
    pararDigitacao,
    handleInput,
    cleanup
  }
}