import { ref } from 'vue'

let notificationInstance = null

export function useChatNotifications() {
  if (notificationInstance) {
    return notificationInstance
  }

  const isInitialized = ref(false)
  const audioContext = ref(null)
  const permissions = ref({
    sound: true,
    desktop: false,
    permission: 'default'
  })

  // Sistema de throttling simplificado
  const notificationThrottle = new Map()
  const THROTTLE_TIME = 3000
  let throttleCleanupInterval = null

  const generateThrottleKey = (conversation, message) => {
    const conversationId = conversation?.id || 'unknown'
    const senderName = message?.sender_name || conversation?.contact_name || 'Contato'
    return `${conversationId}_${senderName}`.replace(/\s/g, '_')
  }

  const shouldThrottleNotification = (conversation, message) => {
    try {
      const key = generateThrottleKey(conversation, message)
      const now = Date.now()
      const lastTime = notificationThrottle.get(key)
      
      if (lastTime && (now - lastTime) < THROTTLE_TIME) {
        return true
      }
      
      notificationThrottle.set(key, now)
      return false
    } catch (error) {
      return false // Fail-safe: permitir notificação
    }
  }

  const startThrottleCleanup = () => {
    if (throttleCleanupInterval) return
    
    throttleCleanupInterval = setInterval(() => {
      const now = Date.now()
      for (const [key, timestamp] of notificationThrottle.entries()) {
        if (now - timestamp > THROTTLE_TIME * 2) {
          notificationThrottle.delete(key)
        }
      }
    }, THROTTLE_TIME)
  }

  const stopThrottleCleanup = () => {
    if (throttleCleanupInterval) {
      clearInterval(throttleCleanupInterval)
      throttleCleanupInterval = null
    }
  }

  const initializeChatNotifications = async () => {
    if (isInitialized.value) return

    try {
      if ('Notification' in window) {
        permissions.value.permission = Notification.permission
        permissions.value.desktop = Notification.permission === 'granted'
      }

      isInitialized.value = true
      startThrottleCleanup()
    } catch (error) {
      console.error('Erro ao inicializar notificações:', error)
    }
  }

  const isInnoclappsReady = () => {
    return window.Innoclapps && 
           window.Innoclapps.app && 
           typeof window.Innoclapps.success === 'function'
  }

  const waitForInnoclapps = () => {
    return new Promise((resolve) => {
      if (isInnoclappsReady()) {
        resolve(true)
        return
      }

      let attempts = 0
      const maxAttempts = 50

      const check = () => {
        attempts++
        if (isInnoclappsReady()) {
          resolve(true)
        } else if (attempts >= maxAttempts) {
          resolve(false)
        } else {
          setTimeout(check, 100)
        }
      }

      setTimeout(check, 100)
    })
  }

  const initializeAudio = async () => {
    if (audioContext.value) return

    try {
      audioContext.value = new (window.AudioContext || window.webkitAudioContext)()
      
      if (audioContext.value.state === 'suspended') {
        await audioContext.value.resume()
      }
    } catch (error) {
      // Silencioso - áudio é opcional
    }
  }

  const playNotificationSound = () => {
    if (!permissions.value.sound || !audioContext.value) return

    try {
      const oscillator = audioContext.value.createOscillator()
      const gainNode = audioContext.value.createGain()
      
      oscillator.connect(gainNode)
      gainNode.connect(audioContext.value.destination)
      
      oscillator.frequency.setValueAtTime(800, audioContext.value.currentTime)
      oscillator.frequency.setValueAtTime(600, audioContext.value.currentTime + 0.1)
      
      gainNode.gain.setValueAtTime(0, audioContext.value.currentTime)
      gainNode.gain.linearRampToValueAtTime(0.3, audioContext.value.currentTime + 0.01)
      gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.value.currentTime + 0.3)
      
      oscillator.start(audioContext.value.currentTime)
      oscillator.stop(audioContext.value.currentTime + 0.3)
    } catch (error) {
      // Silencioso - som é opcional
    }
  }

  const showChatNotification = async (options) => {
    try {
      if (!options || typeof options !== 'object') return

      const {
        type = 'info',
        title = '',
        message = '',
        duration = 4000
      } = options

      const safeTitle = String(title || '').trim()
      const safeMessage = String(message || '').trim()
      const safeType = String(type || 'info').toLowerCase()

      if (!safeMessage) return

      const innoclappsReady = await waitForInnoclapps()
      
      if (innoclappsReady) {
        try {
          const displayMessage = safeTitle ? `${safeTitle}: ${safeMessage}` : safeMessage
          
          switch (safeType) {
            case 'success':
              window.Innoclapps.success(displayMessage, duration)
              break
            case 'error':
            case 'warning':
              window.Innoclapps.error(displayMessage, duration)
              break
            case 'info':
            default:
              window.Innoclapps.info(displayMessage, duration)
              break
          }
          return
        } catch (innoclappsError) {
          // Fallback silencioso
        }
      }

    } catch (error) {
      console.error('Erro ao mostrar notificação:', error)
    }
  }

  // ✅ FUNÇÃO PRINCIPAL CORRIGIDA E SIMPLIFICADA
  const showNewMessageNotification = (conversation, message) => {
    try {
      if (!conversation || !message) return

      // ✅ CORREÇÃO: Obter currentUser de forma segura
      const getCurrentUser = () => {
        try {
          const chatConfig = window.Innoclapps?.scriptConfig('chat')
          if (chatConfig?.user_id) {
            return { id: parseInt(chatConfig.user_id) }
          }

          if (window.currentUser?.id) {
            return { id: parseInt(window.currentUser.id) }
          }

          return null
        } catch (error) {
          return null
        }
      }

      const currentUser = getCurrentUser()
      const currentUserId = currentUser?.id

      const messageIsFromMe = message.is_from_me === true || message.whatsapp_key_from_me === true
      const senderId = message.sender_id || message.sender?.id

      if (messageIsFromMe || (currentUserId && senderId && parseInt(currentUserId) === parseInt(senderId))) {
        return // Não mostrar notificação se for mensagem própria
      }

      if (shouldThrottleNotification(conversation, message)) {
        return // Throttled
      }

      const senderName = String(message.sender_name || conversation.contact_name || 'Contato').trim()
      const messageText = String(message.content || message.text || '').trim()
      
      if (!messageText) return

      const truncatedMessage = messageText.length > 50 
        ? messageText.substring(0, 50) + '...' 
        : messageText

      showChatNotification({
        type: 'info',
        title: `Nova mensagem de ${senderName}`,
        message: truncatedMessage,
        duration: 5000
      })

      if (permissions.value.sound) {
        playNotificationSound()
      }

      if (permissions.value.desktop && document.hidden) {
        showDesktopNotification(senderName, truncatedMessage)
      }
    } catch (error) {
      console.error('Erro ao mostrar notificação de mensagem:', error)
    }
  }

  const showDesktopNotification = (title, message) => {
    try {
      if (!('Notification' in window) || Notification.permission !== 'granted') return

      const notification = new Notification(title, {
        body: message,
        icon: '/favicon.ico',
        tag: 'chat-message',
        requireInteraction: false,
        silent: false
      })

      setTimeout(() => notification.close(), 5000)

      notification.onclick = () => {
        window.focus()
        notification.close()
      }
    } catch (error) {
      // Silencioso - notificação desktop é opcional
    }
  }

  const showConnectionNotification = (status) => {
    // Apenas log interno - sem spam de notificações
    if (window.console && typeof window.console.log === 'function') {
      const messages = {
        connected: 'Chat conectado',
        disconnected: 'Chat desconectado',
        reconnecting: 'Reconectando chat...'
      }
      console.log(`[CHAT] ${messages[status] || 'Status alterado'}`)
    }
  }

  const showSuccessNotification = (title, message) => {
    showChatNotification({ type: 'success', title, message, duration: 4000 })
  }

  const showErrorNotification = (title, message, details = '') => {
    const fullMessage = details ? `${message} (${details})` : message
    showChatNotification({ type: 'error', title, message: fullMessage, duration: 6000 })
  }

  const showWarningNotification = (title, message) => {
    showChatNotification({ type: 'warning', title, message, duration: 4000 })
  }

  const showInfoNotification = (title, message) => {
    showChatNotification({ type: 'info', title, message, duration: 4000 })
  }

  // Setup de interação do usuário
  const setupUserInteractionListener = () => {
    const handleFirstInteraction = () => {
      initializeAudio()
      document.removeEventListener('click', handleFirstInteraction)
      document.removeEventListener('keydown', handleFirstInteraction)
    }

    document.addEventListener('click', handleFirstInteraction, { once: true })
    document.addEventListener('keydown', handleFirstInteraction, { once: true })
  }

  if (typeof window !== 'undefined') {
    setupUserInteractionListener()
  }

  const clearNotificationThrottle = () => {
    notificationThrottle.clear()
  }

  // ✅ CLEANUP ADEQUADO
  const destroy = () => {
    stopThrottleCleanup()
    clearNotificationThrottle()
    if (audioContext.value) {
      try {
        audioContext.value.close()
      } catch (e) {
        // Silencioso
      }
      audioContext.value = null
    }
  }

  const instance = {
    isInitialized,
    permissions,
    initializeChatNotifications,
    showChatNotification,
    showNewMessageNotification,
    showConnectionNotification,
    showSuccessNotification,
    showErrorNotification,
    showWarningNotification,
    showInfoNotification,
    playNotificationSound,
    clearNotificationThrottle,
    destroy,
    getThrottleStats: () => ({
      throttledNotifications: notificationThrottle.size,
      throttleTime: THROTTLE_TIME
    })
  }

  notificationInstance = instance
  return instance
}