// modules/Chat/resources/js/composables/useConversationActions.js
import { ref } from 'vue'
import servicoApi from '../servicos/api.js'
import chatStore from '../servicos/chatStore.js'

export function useConversationActions() {
  // ===== ESTADO =====
  const isProcessingAction = ref(false)
  const actionQueue = new Set()

  // ===== HELPERS =====
  function isActionInProgress(conversationId, action) {
    return actionQueue.has(`${conversationId}-${action}`)
  }

  function addToQueue(conversationId, action) {
    actionQueue.add(`${conversationId}-${action}`)
  }

  function removeFromQueue(conversationId, action) {
    actionQueue.delete(`${conversationId}-${action}`)
  }

  async function executeAction(conversationId, action, apiCall, localUpdate = null) {
    if (isActionInProgress(conversationId, action)) {
      throw new Error('Ação já em andamento')
    }

    try {
      isProcessingAction.value = true
      addToQueue(conversationId, action)

      // Aplicar mudança local primeiro (otimistic update)
      if (localUpdate) {
        chatStore.updateConversationMeta({
          id: conversationId,
          ...localUpdate
        })
      }

      // Executar chamada da API
      const result = await apiCall()

      // Atualizar com dados reais da API se disponível
      if (result && typeof result === 'object') {
        chatStore.updateConversationMeta({
          id: conversationId,
          ...result
        })
      }

      return result
    } catch (error) {
      // Reverter mudança local em caso de erro
      if (localUpdate) {
        const conversation = chatStore.getConversaById(conversationId)
        if (conversation) {
          // Reverter para estado anterior
          const revertData = {}
          Object.keys(localUpdate).forEach(key => {
            if (key === 'unread_count') {
              revertData[key] = localUpdate[key] > 0 ? 0 : 1
            } else if (typeof localUpdate[key] === 'boolean') {
              revertData[key] = !localUpdate[key]
            }
          })
          chatStore.updateConversationMeta({
            id: conversationId,
            ...revertData
          })
        }
      }
      throw error
    } finally {
      removeFromQueue(conversationId, action)
      isProcessingAction.value = actionQueue.size > 0
    }
  }

  // ===== AÇÕES =====
  async function markAsRead(conversationId) {
    return executeAction(
      conversationId,
      'mark-read',
      () => servicoApi.markAsRead(conversationId),
      { unread_count: 0, read_at: new Date().toISOString() }
    )
  }

  async function markAsUnread(conversationId) {
    return executeAction(
      conversationId,
      'mark-unread',
      () => servicoApi.markAsUnread(conversationId),
      { unread_count: 1 }
    )
  }

  async function togglePin(conversationId, isPinned) {
    return executeAction(
      conversationId,
      'toggle-pin',
      () => servicoApi.togglePinConversation(conversationId, isPinned),
      { is_pinned: isPinned }
    )
  }

  async function toggleMute(conversationId, isMuted) {
    return executeAction(
      conversationId,
      'toggle-mute',
      () => servicoApi.toggleMuteConversation(conversationId, isMuted),
      { is_muted: isMuted }
    )
  }

  async function archiveConversation(conversationId) {
    return executeAction(
      conversationId,
      'archive',
      () => servicoApi.archiveConversation(conversationId),
      { status: 'archived', archived_at: new Date().toISOString() }
    )
  }

  async function unarchiveConversation(conversationId) {
    return executeAction(
      conversationId,
      'unarchive',
      () => servicoApi.toggleConversationStatus(conversationId, { status: 'active' }),
      { status: 'active', archived_at: null }
    )
  }

  async function reopenConversation(conversationId, reason = 'Reabertura solicitada pelo usuário') {
    return executeAction(
      conversationId,
      'reopen',
      () => servicoApi.reopenConversation(conversationId, reason),
      { status: 'active', closed_at: null, reopened_at: new Date().toISOString() }
    )
  }

  async function closeConversation(conversationId, reason = 'Conversa finalizada') {
    return executeAction(
      conversationId,
      'close',
      () => servicoApi.closeConversation(conversationId, reason),
      { status: 'closed', closed_at: new Date().toISOString() }
    )
  }

  async function exportConversation(conversationId, format = 'pdf') {
    return executeAction(
      conversationId,
      'export',
      async () => {
        const response = await servicoApi.exportConversation(conversationId, format)
        
        if (response.download_url) {
          // Abrir download em nova aba
          const link = document.createElement('a')
          link.href = response.download_url
          link.target = '_blank'
          link.download = response.filename || `conversa-${conversationId}.${format}`
          document.body.appendChild(link)
          link.click()
          document.body.removeChild(link)
        }
        
        return response
      }
    )
  }

  async function deleteConversation(conversationId) {
    return executeAction(
      conversationId,
      'delete',
      () => servicoApi.deleteConversation(conversationId)
    )
  }

  async function assignUser(conversationId, userId, isMultiple = false) {
    return executeAction(
      conversationId,
      'assign-user',
      () => servicoApi.assignConversation(conversationId, { user_id: userId, multiple: isMultiple })
    )
  }

  async function unassignUser(conversationId, userId) {
    return executeAction(
      conversationId,
      'unassign-user',
      () => servicoApi.unassignConversation(conversationId, userId)
    )
  }

  async function transferConversation(conversationId, targetUserId, reason = '') {
    return executeAction(
      conversationId,
      'transfer',
      () => servicoApi.transferConversation(conversationId, targetUserId, reason)
    )
  }

  // ===== AÇÕES EM LOTE =====
  async function bulkAction(conversationIds, action, params = {}) {
    if (!Array.isArray(conversationIds) || conversationIds.length === 0) {
      throw new Error('IDs de conversas inválidos')
    }

    try {
      isProcessingAction.value = true

      const response = await servicoApi.bulkConversationAction({
        conversation_ids: conversationIds,
        action,
        ...params
      })

      // Atualizar conversas afetadas
      if (response.updated_conversations) {
        response.updated_conversations.forEach(conversation => {
          chatStore.updateConversationMeta(conversation)
        })
      }

      return response
    } catch (error) {
      console.error('Erro na ação em lote:', error)
      throw error
    } finally {
      isProcessingAction.value = false
    }
  }

  // ===== CLEANUP =====
  function cleanup() {
    actionQueue.clear()
    isProcessingAction.value = false
  }

  return {
    // Estado
    isProcessingAction,
    
    // Ações individuais
    markAsRead,
    markAsUnread,
    togglePin,
    toggleMute,
    archiveConversation,
    unarchiveConversation,
    reopenConversation,
    closeConversation,
    exportConversation,
    deleteConversation,
    assignUser,
    unassignUser,
    transferConversation,
    
    // Ações em lote
    bulkAction,
    
    // Helpers
    isActionInProgress,
    cleanup
  }
}