// modules/Chat/resources/js/composables/useConversationData.js
import { ref, computed, shallowRef } from 'vue'
import { ensureAbsoluteUrl, isValidImageUrl } from '../utils/urlHelpers.js'

// Cache global para otimização
const conversationCache = new Map()
const avatarCache = new Map()
const displayNameCache = new Map()

export function useConversationData() {
  
  // ===== HELPERS DE CACHE =====
  function getCachedValue(cacheMap, key, generator) {
    if (cacheMap.has(key)) {
      return cacheMap.get(key)
    }
    
    const result = generator()
    cacheMap.set(key, result)
    
    // Limitar cache (manter apenas 200 entradas)
    if (cacheMap.size > 200) {
      const firstKey = cacheMap.keys().next().value
      cacheMap.delete(firstKey)
    }
    
    return result
  }

  // ===== USUÁRIO ATUAL =====
  function getCurrentChatUser() {
    // Dados completos do chat config
    const chatConfig = window.Innoclapps?.scriptConfig?.('chat') || {}
    if (chatConfig.user && chatConfig.user.id) {
      return chatConfig.user
    }

    // WebSocket
    const wsUser = window.websocketService?.getCurrentUser?.()
    if (wsUser && wsUser.id) {
      return wsUser
    }

    // Dados dos usuários do sistema
    const users = window.Innoclapps?.scriptConfig?.('users') || []
    const currentUserId = window.Innoclapps?.scriptConfig?.('user_id')
    if (currentUserId && users.length > 0) {
      const currentUser = users.find(u => u.id === currentUserId)
      if (currentUser) {
        return currentUser
      }
    }

    // Fallback com Gravatar
    if (chatConfig.user_id) {
      const email = chatConfig.user_email || ''
      const gravatarUrl = `https://www.gravatar.com/avatar/${md5(email.toLowerCase())}?s=128&d=identicon`
      
      return {
        id: chatConfig.user_id,
        name: chatConfig.user_name || 'Você',
        email: email,
        avatar: null,
        avatar_url: gravatarUrl,
        gravatar_url: gravatarUrl
      }
    }

    return {}
  }

  function getCurrentUserId() {
    const user = getCurrentChatUser()
    const id = parseInt(user.id)
    return Number.isNaN(id) ? null : id
  }

  // ===== DISPLAY NAME =====
  function getDisplayName(conversa) {
    if (!conversa) return 'Conversa'
    
    const cacheKey = `display_${conversa.id}_${conversa.updated_at || ''}`
    return getCachedValue(displayNameCache, cacheKey, () => {
      return (
        conversa.processed_display_name ||
        conversa.contact_name ||
        conversa.subject ||
        conversa.contact?.display_name ||
        conversa.phone ||
        (conversa.id ? `Conversa #${conversa.id}` : 'Conversa')
      )
    })
  }

  // ===== AVATAR =====
  function md5(string) {
    return string.split('').reduce((a, b) => {
      a = ((a << 5) - a) + b.charCodeAt(0)
      return a & a
    }, 0).toString(16)
  }

  function getConversationAvatar(conversa) {
    if (!conversa) return null
    
    const cacheKey = `avatar_${conversa.id}_${conversa.updated_at || ''}`
    return getCachedValue(avatarCache, cacheKey, () => {
      const candidates = [
        conversa.processed_avatar,
        conversa.avatar_url,
        conversa.contact?.best_profile_photo,
        conversa.whatsapp_profile_picture,
        conversa.contact?.whatsapp_profile_photo_url,
        conversa.contact?.whatsapp_profile_photo_path
      ]

      for (const raw of candidates) {
        if (!raw) continue
        const full = ensureAbsoluteUrl(raw)
        if (full && isValidImageUrl(full)) {
          return full
        }
      }

      return null
    })
  }

  function getCurrentUserAvatar() {
    const user = getCurrentChatUser()
    const candidates = [
      user.avatar_url,       
      user.avatar,  
      user.photo_url,
      user.profile_photo_url,
      user.profile_photo_path
    ]

    for (const raw of candidates) {
      if (!raw) continue
      const full = ensureAbsoluteUrl(raw)
      if (full && isValidImageUrl(full)) {
        return full
      }
    }

    // Fallback para Gravatar
    if (user.email) {
      return `https://www.gravatar.com/avatar/${md5(user.email.toLowerCase())}?s=128&d=identicon`
    }

    return null
  }

  function getMessageAvatar(mensagem, conversa) {
    const currentUserId = getCurrentUserId()
    const isMine = currentUserId !== null &&
      parseInt(mensagem.sender_id) === currentUserId &&
      !mensagem.is_from_contact

    const cacheKey = `msg_avatar_${mensagem.id}_${isMine ? 'mine' : 'other'}`
    
    return getCachedValue(avatarCache, cacheKey, () => {
      const candidates = []

      if (isMine) {
        const currentUser = getCurrentChatUser()
        candidates.push(
          currentUser.avatar_url,
          currentUser.uploaded_avatar_url,
          currentUser.gravatar_url,
          currentUser.avatar
        )
      }

      // Avatar da mensagem
      candidates.push(
        mensagem.sender_avatar,
        mensagem.metadata?.sender_avatar
      )

      // Avatar da conversa (para contatos)
      if (mensagem.is_from_contact && conversa) {
        candidates.push(
          conversa.processed_avatar,
          conversa.avatar_url,
          conversa.contact?.best_profile_photo,
          conversa.whatsapp_profile_picture
        )
      }

      for (const raw of candidates) {
        if (!raw) continue
        const full = ensureAbsoluteUrl(raw)
        if (full && isValidImageUrl(full)) {
          return full
        }
      }

      // Fallback Gravatar para mensagens próprias
      if (isMine) {
        const currentUser = getCurrentChatUser()
        if (currentUser.email) {
          return `https://www.gravatar.com/avatar/${md5(currentUser.email.toLowerCase())}?s=128&d=identicon`
        }
      }

      return null
    })
  }

  // ===== INFORMAÇÕES DA EMPRESA =====
  function getCompanyInfo(conversa) {
    if (!conversa) return { name: null, id: null }
    
    return {
      name: conversa.company_name || conversa.contact?.company?.name || null,
      id: conversa.company_id || conversa.contact?.company?.id || null
    }
  }

  // ===== CANAL E TIPO =====
  function getChannelInfo(conversa) {
    if (!conversa) return { icon: 'ChatBubbleLeftRight', label: null, description: 'Conversa' }
    
    const channelData = {
      'evolution_api': { 
        icon: 'DevicePhoneMobile', 
        label: 'WhatsApp (Evolution API)',
        description: 'Conversa via WhatsApp (Evolution API)'
      },
      'whatsapp': { 
        icon: 'DevicePhoneMobile', 
        label: 'WhatsApp',
        description: 'Conversa via WhatsApp'
      },
      'internal_chat': { 
        icon: 'Users', 
        label: 'Interno',
        description: 'Chat interno entre usuários'
      },
      'email': { 
        icon: 'EnvelopeIcon', 
        label: 'Email',
        description: 'Conversa via Email'
      },
      'telegram': { 
        icon: 'ChatBubbleLeftRight', 
        label: 'Telegram',
        description: 'Conversa via Telegram'
      },
      'webchat': { 
        icon: 'GlobeAltIcon', 
        label: 'Web Chat',
        description: 'Conversa via Web Chat'
      }
    }
    
    if (conversa.type === 'internal') {
      return {
        icon: 'Users',
        label: 'Interno', 
        description: 'Chat interno entre usuários do sistema'
      }
    }
    
    return channelData[conversa.channel_type] || {
      icon: 'ChatBubbleLeftRight',
      label: 'Externo',
      description: 'Conversa externa'
    }
  }

  // ===== STATUS =====
  function getStatusInfo(conversa) {
    if (!conversa) return { icon: 'QuestionMarkCircle', text: 'Desconhecido', dot: 'ativa' }
    
    const statusData = {
      'open': { icon: 'ChatBubbleLeftRight', text: 'Aberta', dot: 'ativa' },
      'active': { icon: 'ChatBubbleLeftRight', text: 'Ativa', dot: 'ativa' },
      'closed': { icon: 'CheckCircle', text: 'Resolvida', dot: 'resolvida' },
      'archived': { icon: 'ArchiveBox', text: 'Arquivada', dot: 'arquivada' },
      'pending': { icon: 'Clock', text: 'Pendente', dot: 'ativa' }
    }
    
    let result = statusData[conversa.status] || statusData['active']
    
    // Adicionar modificadores
    const modifiers = []
    if (conversa.is_pinned) modifiers.push('Fixada')
    if (conversa.is_muted) modifiers.push('Silenciada')
    if (conversa.type === 'internal') modifiers.push('Interna')
    
    if (modifiers.length > 0) {
      result.text += ' • ' + modifiers.join(' • ')
    }
    
    // Dot especial para conversas internas
    if (conversa.type === 'internal') {
      result.dot = 'interna'
    }
    
    return result
  }

  // ===== CONTATO =====
  function getContactInfo(conversa) {
    if (!conversa) return { info: null, icon: 'UserIcon' }
    
    if (conversa.phone && conversa.channel_type === 'whatsapp') {
      return {
        info: formatPhoneNumber(conversa.phone),
        icon: 'DevicePhoneMobile'
      }
    }
    
    if (conversa.contact?.email) {
      return {
        info: conversa.contact.email,
        icon: 'EnvelopeIcon'
      }
    }
    
    return { info: null, icon: 'UserIcon' }
  }

  function formatPhoneNumber(phone) {
    if (!phone) return ''
    const clean = phone.replace(/\D/g, '')
    if (clean.length === 11) return `(${clean.slice(0, 2)}) ${clean.slice(2, 7)}-${clean.slice(7)}`
    if (clean.length === 10) return `(${clean.slice(0, 2)}) ${clean.slice(2, 6)}-${clean.slice(6)}`
    return phone
  }

  // ===== ATRIBUIÇÃO =====
  function getAssignedUser(conversa) {
    if (!conversa) return null
    if (conversa.assignedUser) return conversa.assignedUser
    if (conversa.assigned_users && conversa.assigned_users.length > 0) {
      const firstAssignment = conversa.assigned_users[0]
      return firstAssignment.user || firstAssignment
    }
    return null
  }

  function getAssignmentInfo(conversa) {
    if (!conversa) return { 
      user: null, 
      avatar: null, 
      name: 'Não atribuída', 
      tooltip: 'Não atribuída',
      hasMultiple: false,
      count: 0
    }
    
    const user = getAssignedUser(conversa)
    if (!user) return { 
      user: null, 
      avatar: null, 
      name: 'Não atribuída', 
      tooltip: 'Não atribuída',
      hasMultiple: false,
      count: 0
    }
    
    const userName = user.name || user.display_name || 'Usuário'
    const hasMultiple = hasMultipleAssignments(conversa)
    const count = getAssignmentCount(conversa)
    
    let displayName = userName
    if (hasMultiple && count > 1) {
      displayName = `${userName} +${count - 1}`
    }
    
    let tooltip = `Atribuída a: ${userName}`
    if (user.email) {
      tooltip += ` (${user.email})`
    }
    
    if (hasMultiple && conversa.assigned_users) {
      const otherUsers = conversa.assigned_users
        .filter(a => a.user && a.user.id !== user.id)
        .map(a => a.user.name || a.user.display_name)
        .filter(Boolean)
        .slice(0, 2)
        .join(', ')
      
      if (otherUsers) {
        const remaining = count - 1 - otherUsers.split(',').length
        tooltip += ` e ${otherUsers}`
        if (remaining > 0) {
          tooltip += ` +${remaining} outros`
        }
      }
    }
    
    return {
      user,
      avatar: getAssignedUserAvatar(conversa),
      name: displayName,
      tooltip,
      hasMultiple,
      count
    }
  }

  function getAssignedUserAvatar(conversa) {
    const user = getAssignedUser(conversa)
    if (!user) return null

    const candidates = [
      user.avatar_url,
      user.avatar,
      user.profile_photo_url,
      user.profile_photo_path
    ]

    for (const raw of candidates) {
      if (!raw) continue
      const full = ensureAbsoluteUrl(raw)
      if (full && isValidImageUrl(full)) {
        return full
      }
    }

    return null
  }

  function hasMultipleAssignments(conversa) {
    if (!conversa) return false

    const hasSingleAssignedUser = conversa.assignedUser && (!conversa.assigned_users || conversa.assigned_users.length === 0)
    const hasMultipleInList = conversa.assigned_users && conversa.assigned_users.length > 1
    const hasOneAssignedAndList = conversa.assignedUser && conversa.assigned_users && conversa.assigned_users.length > 0

    return hasMultipleInList || hasOneAssignedAndList || !hasSingleAssignedUser && conversa.assigned_users && conversa.assigned_users.length > 1
  }

  function getAssignmentCount(conversa) {
    if (!conversa) return 0

    let count = 0

    if (conversa.assignedUser) {
      count++
    }

    if (conversa.assigned_users && conversa.assigned_users.length > 0) {
      const assignedUserId = conversa.assignedUser?.id
      conversa.assigned_users.forEach(assignment => {
        if (assignment.user?.id !== assignedUserId) {
          count++
        }
      })
      if (!conversa.assignedUser && conversa.assigned_users.length > 0) {
        count++
      }
    } else if (!conversa.assignedUser && conversa.assigned_users && conversa.assigned_users.length === 1) {
      count = 1
    }

    if (count === 0 && (conversa.assignedUser || (conversa.assigned_users && conversa.assigned_users.length > 0))) {
      return 1
    }

    return count
  }

  // ===== CORES E INICIAIS =====
  function getAvatarColor(conversa) {
    if (!conversa) return '#3b82f6'
    
    const colors = [
      '#3b82f6', '#ef4444', '#10b981', '#f59e0b',
      '#8b5cf6', '#06b6d4', '#ec4899', '#84cc16',
      '#f97316', '#14b8a6', '#a855f7', '#eab308'
    ]
    
    const id = conversa.id || 0
    return colors[id % colors.length]
  }

  function getInitials(conversa) {
    if (!conversa) return 'C'
    
    const name = getDisplayName(conversa)
    
    // Para números de telefone
    if (name.includes('+') || /^\(?\d+/.test(name)) {
      const digits = name.replace(/\D/g, '')
      return digits.slice(-2) || 'C'
    }
    
    // Para nomes normais
    const parts = name.split(' ')
    const first = parts[0]?.charAt(0) || ''
    const last = parts[1]?.charAt(0) || ''
    return (first + last).toUpperCase() || 'C'
  }

  function getAssignedUserInitials(conversa) {
    const user = getAssignedUser(conversa)
    if (!user) return 'U'
    
    const name = user.name || user.display_name || 'Usuário'
    
    if (name.includes('+') || /^\(?\d+/.test(name)) {
      const digits = name.replace(/\D/g, '')
      return digits.slice(-2) || 'U'
    }

    const parts = name.split(' ')
    const first = parts[0]?.charAt(0) || ''
    const last = parts[1]?.charAt(0) || ''
    return (first + last).toUpperCase() || 'U'
  }

  function getAssignedUserColor(conversa) {
    const user = getAssignedUser(conversa)
    const userId = user?.id || 0

    const colors = [
      '#3b82f6', '#ef4444', '#10b981', '#f59e0b',
      '#8b5cf6', '#06b6d4', '#ec4899', '#84cc16',
      '#f97316', '#14b8a6', '#a855f7', '#eab308'
    ]
    return colors[userId % colors.length]
  }

  // ===== LIMPEZA DE CACHE =====
  function clearCache() {
    conversationCache.clear()
    avatarCache.clear()
    displayNameCache.clear()
  }

  // ===== RETURN =====
  return {
    // Usuário atual
    getCurrentChatUser,
    getCurrentUserId,
    getCurrentUserAvatar,
    
    // Display e Avatar
    getDisplayName,
    getConversationAvatar,
    getMessageAvatar,
    getInitials,
    getAvatarColor,
    
    // Informações da conversa
    getCompanyInfo,
    getChannelInfo,
    getStatusInfo,
    getContactInfo,
    
    // Atribuição
    getAssignedUser,
    getAssignmentInfo,
    getAssignedUserAvatar,
    getAssignedUserInitials,
    getAssignedUserColor,
    hasMultipleAssignments,
    getAssignmentCount,
    
    // Utilitários
    formatPhoneNumber,
    clearCache
  }
}