// modules/Chat/resources/js/composables/useConversationFilters.js
import { ref, computed, shallowRef } from 'vue'
import chatStore from '../servicos/chatStore.js'

export function useConversationFilters() {
  // ===== ESTADO =====
  const activeFilter = shallowRef('fila') // ✅ Inicia na fila
  const searchTerm = ref('')
  const searchTimeout = ref(null)

  // ===== COMPUTED =====
  const conversas = computed(() => chatStore.conversas.value || [])
  
  const currentUser = computed(() => {
    try {
      const ws = window.websocketService
      return ws?.getCurrentUser?.() || null
    } catch (error) {
      return null
    }
  })

  const isAdmin = computed(() => currentUser.value?.super_admin || false)
  const currentUserId = computed(() => currentUser.value?.id || null)

  const conversationStats = computed(() => {
    if (!Array.isArray(conversas.value) || conversas.value.length === 0) {
      return {
        total: 0,
        assignedToMe: 0,
        unassigned: 0
      }
    }

    const assignedToMe = conversas.value.filter(c => {
      if (['archived', 'closed'].includes(c.status)) return false
      if (c.assigned_user_id === currentUserId.value) return true
      if (c.assigned_users?.some(a => a.user?.id === currentUserId.value)) return true
      return false
    }).length

    const unassigned = conversas.value.filter(c =>
      !c.assigned_user_id && 
      (!c.assigned_users || c.assigned_users.length === 0) &&
      !['archived', 'closed'].includes(c.status) &&
      c.type === 'external'
    ).length

    return {
      total: conversas.value.filter(c => !['archived', 'closed'].includes(c.status)).length,
      assignedToMe,
      unassigned
    }
  })

  // ✅ FILTROS CORRIGIDOS
  const availableFilters = computed(() => {
    const stats = conversationStats.value
    const filters = []
    
    // ✅ FILA - SEMPRE VISÍVEL
    filters.push({
      id: 'fila',
      label: 'Fila',
      count: stats.unassigned,
      description: 'Conversas aguardando atendimento',
      icon: 'Clock',
      priority: 'high'
    })
    
    // ✅ MINHAS - SEMPRE VISÍVEL
    filters.push({
      id: 'minhas',
      label: 'Minhas',
      count: stats.assignedToMe,
      description: 'Suas conversas ativas',
      icon: 'UserCheck',
      priority: 'high'
    })
    
    // ✅ TODAS - APENAS PARA ADMINS
    if (isAdmin.value) {
      filters.push({
        id: 'todas',
        label: 'Todas',
        count: stats.total,
        description: 'Todas as conversas do sistema',
        icon: 'ChatBubbleLeftRight',
        priority: 'admin'
      })
    }
    
    return filters
  })

  // ===== MÉTODOS =====
  function setActiveFilter(filterId) {
    if (!filterId || activeFilter.value === filterId) return false
    
    const filterExists = availableFilters.value.some(f => f.id === filterId)
    if (!filterExists) {
      console.warn('Filtro inválido:', filterId)
      return false
    }

    console.log('🔄 Mudando filtro de', activeFilter.value, 'para', filterId)
    activeFilter.value = filterId
    return true
  }

  // ✅ MAPEAMENTO CORRETO DE PARÂMETROS
  function getFilterParams() {
    const filter = activeFilter.value
    const params = {}

    console.log('📋 Gerando parâmetros para filtro:', filter)

    switch (filter) {
      case 'fila':
        // ✅ APENAS conversas não atribuídas
        params.show_unassigned_only = true
        params.show_my_only = false
        params.show_all_conversations = false
        params.show_mine_and_unassigned = false
        break
        
      case 'minhas':
        // ✅ APENAS conversas atribuídas ao usuário
        params.show_my_only = true
        params.show_unassigned_only = false
        params.show_all_conversations = false
        params.show_mine_and_unassigned = false
        break
        
      case 'todas':
        // ✅ TODAS as conversas (admin)
        params.show_all_conversations = true
        params.show_my_only = false
        params.show_unassigned_only = false
        params.show_mine_and_unassigned = false
        break
        
      default:
        // ✅ FALLBACK: suas + não atribuídas
        params.show_mine_and_unassigned = true
        params.show_my_only = false
        params.show_unassigned_only = false
        params.show_all_conversations = false
        break
    }

    console.log('📤 Parâmetros gerados:', params)
    return params
  }

  function debouncedSearch(callback, delay = 300) {
    if (searchTimeout.value) {
      clearTimeout(searchTimeout.value)
    }

    if (searchTerm.value.trim().length > 0 && searchTerm.value.trim().length < 2) {
      return
    }

    searchTimeout.value = setTimeout(() => {
      callback(searchTerm.value.trim())
    }, delay)
  }

  function clearSearch() {
    searchTerm.value = ''
    if (searchTimeout.value) {
      clearTimeout(searchTimeout.value)
    }
  }

  // ===== CLEANUP =====
  function cleanup() {
    clearSearch()
  }

  return {
    // Estado
    activeFilter,
    searchTerm,
    availableFilters,
    conversationStats,
    
    // Métodos
    setActiveFilter,
    getFilterParams,
    debouncedSearch,
    clearSearch,
    cleanup
  }
}