// modules/Chat/resources/js/composables/useConversationHelpers.js
import { ref } from 'vue'

const avatarCache = new Map()
const brokenImages = ref(new Set())

export function useConversationHelpers() {
  
  // =============== FUNÇÕES AUXILIARES INTERNAS ===============
  
  const ensureAbsoluteUrl = (url) => {
    try {
      if (!url) return null
      if (url.startsWith('http://') || url.startsWith('https://')) {
        return url
      }
      if (url.startsWith('//')) {
        return `https:${url}`
      }
      if (url.startsWith('/')) {
        return `${window.location.origin}${url}`
      }
      return `${window.location.origin}/storage/${url}`
    } catch (error) {
      console.warn('Erro em ensureAbsoluteUrl:', error)
      return null
    }
  }

  const isValidImageUrl = (url) => {
    try {
      if (!url || typeof url !== 'string') return false
      if (url.includes('/chat?') || url.includes('/conversation?') || url.includes('/chat#')) return false
      return /^https?:\/\/.+\.(jpg|jpeg|png|gif|webp|svg)(\?.*)?$/i.test(url)
    } catch (error) {
      console.warn('Erro em isValidImageUrl:', error)
      return false
    }
  }

  const formatPhoneNumber = (phone) => {
    try {
      if (!phone) return ''
      const clean = phone.replace(/\D/g, '')
      if (clean.length === 11) return `(${clean.slice(0, 2)}) ${clean.slice(2, 7)}-${clean.slice(7)}`
      if (clean.length === 10) return `(${clean.slice(0, 2)}) ${clean.slice(2, 6)}-${clean.slice(6)}`
      return phone
    } catch (error) {
      console.warn('Erro em formatPhoneNumber:', error)
      return phone || ''
    }
  }

  // =============== FUNÇÕES PRINCIPAIS EXPORTADAS ===============
  
  const getAvatarColor = (conversa) => {
    try {
      const colors = [
        '#3b82f6', '#ef4444', '#10b981', '#f59e0b',
        '#8b5cf6', '#06b6d4', '#ec4899', '#84cc16',
        '#f97316', '#14b8a6', '#a855f7', '#eab308'
      ]
      return colors[(conversa?.id || 0) % colors.length]
    } catch (error) {
      console.warn('Erro em getAvatarColor:', error)
      return '#3b82f6'
    }
  }

  const getInitials = (conversa) => {
    try {
      const name = conversa?.processed_display_name || conversa?.contact_name || 'C'
      if (name.includes('+') || /^\(?\d+/.test(name)) {
        const digits = name.replace(/\D/g, '')
        return digits.slice(-2) || 'C'
      }
      const parts = name.split(' ')
      const first = parts[0]?.charAt(0) || ''
      const last = parts[1]?.charAt(0) || ''
      return (first + last).toUpperCase() || 'C'
    } catch (error) {
      console.warn('Erro em getInitials:', error)
      return 'C'
    }
  }

  const formatTime = (timestamp) => {
    try {
      if (!timestamp) return ''
      
      const date = new Date(timestamp)
      const now = new Date()
      const diff = now - date
      
      if (diff < 60000) return 'agora'
      if (diff < 3600000) return `${Math.floor(diff / 60000)}m`
      if (diff < 86400000) return `${Math.floor(diff / 3600000)}h`
      if (diff < 604800000) return `${Math.floor(diff / 86400000)}d`
      return date.toLocaleDateString('pt-BR', { day: '2-digit', month: '2-digit' })
    } catch (error) {
      console.warn('Erro em formatTime:', error)
      return ''
    }
  }

  const getLastMessagePreview = (conversa) => {
    try {
      if (!conversa || typeof conversa !== 'object') {
        return 'Nenhuma mensagem'
      }

      // Obter usuário atual de forma segura
      let currentUserId = null
      try {
        currentUserId = window.Innoclapps?.scriptConfig?.('chat')?.user?.id || 
                       window.Innoclapps?.user?.id ||
                       window.websocketService?.getCurrentUser?.()?.id ||
                       null
      } catch (e) {
        // Silencioso
      }

      // Extrair dados da mensagem
      let message = ''
      let senderName = ''
      let senderId = null
      
      if (conversa.last_message && typeof conversa.last_message === 'string') {
        message = String(conversa.last_message).trim()
        senderName = String(conversa.last_message_sender || '').trim()
        senderId = conversa.last_message_sender_id
      } else if (conversa.lastMessage && typeof conversa.lastMessage === 'object') {
        message = String(conversa.lastMessage.content || conversa.lastMessage.text || '').trim()
        senderName = String(conversa.lastMessage.sender_name || conversa.lastMessage.sender || '').trim()
        senderId = conversa.lastMessage.sender_id
      } else if (conversa.messages && Array.isArray(conversa.messages) && conversa.messages.length > 0) {
        const lastMsg = conversa.messages[conversa.messages.length - 1]
        if (lastMsg && typeof lastMsg === 'object') {
          message = String(lastMsg.content || lastMsg.text || '').trim()
          senderName = String(lastMsg.sender_name || lastMsg.sender || '').trim()
          senderId = lastMsg.sender_id
        }
      }
      
      if (!message) {
        if (conversa.status === 'closed') return 'Conversa finalizada'
        if (conversa.status === 'archived') return 'Conversa arquivada'
        return 'Nenhuma mensagem'
      }
      
      // Limpar formatação markdown
      try {
        message = message.replace(/\*\*(.+?)\*\*/g, '$1')
        message = message.replace(/\*(.+?)\*/g, '$1')
        message = message.replace(/```(.+?)```/g, '$1')
        message = message.replace(/`(.+?)`/g, '$1')
        message = message.replace(/\s+/g, ' ').trim()
      } catch (e) {
        // Se falhar, usar mensagem original
      }
      
      // Verificar se é mensagem própria
      let isOwnMessage = false
      if (currentUserId && senderId) {
        try {
          isOwnMessage = parseInt(senderId) === parseInt(currentUserId) ||
                        String(senderId) === String(currentUserId) ||
                        senderId === currentUserId
        } catch (e) {
          isOwnMessage = false
        }
      }
      
      // Adicionar prefixo de remetente
      if (isOwnMessage) {
        message = `Você: ${message}`
      } else if (senderName && senderName !== 'Sistema' && senderName.length > 0) {
        message = `${senderName}: ${message}`
      }
      
      // Truncar se muito longa
      const maxLength = 35
      return message.length > maxLength ? message.substring(0, maxLength) + '...' : message
      
    } catch (error) {
      console.warn('Erro em getLastMessagePreview:', error)
      return 'Erro ao carregar mensagem'
    }
  }

  // ✅ FUNÇÃO PRINCIPAL QUE ESTAVA FALTANDO
  const processarAvatarConversa = (conversa) => {
    try {
      if (!conversa || typeof conversa !== 'object') {
        return {
          avatar_url: null,
          display_name: 'Conversa',
          participant: null
        }
      }

      // Processar avatar
      let avatar_url = null
      const candidates = [
        conversa.processed_avatar,
        conversa.avatar_url,
        conversa.contact?.best_profile_photo,
        conversa.whatsapp_profile_picture
      ]

      for (const raw of candidates) {
        if (!raw) continue
        const full = ensureAbsoluteUrl(raw)
        if (isValidImageUrl(full)) {
          avatar_url = full
          break
        }
      }

      // Processar display name
      let display_name = 'Conversa'
      if (conversa.processed_display_name) {
        display_name = conversa.processed_display_name
      } else if (conversa.contact_name && conversa.contact_name !== 'Contato sem nome') {
        display_name = conversa.contact_name
      } else if (conversa.subject) {
        display_name = conversa.subject
      } else if (conversa.contact?.display_name) {
        display_name = conversa.contact.display_name
      } else if (conversa.phone) {
        display_name = formatPhoneNumber(conversa.phone)
      } else {
        display_name = `Conversa #${conversa.id}`
      }

      // Processar participant (para conversas internas)
      let participant = null
      if (conversa.type === 'internal' && conversa.participants && conversa.participants.length > 0) {
        participant = conversa.participants[0]
      }

      return {
        avatar_url,
        display_name,
        participant
      }
    } catch (error) {
      console.warn('Erro em processarAvatarConversa:', error)
      return {
        avatar_url: null,
        display_name: 'Erro ao processar',
        participant: null
      }
    }
  }

  const handleImageError = (conversa, event) => {
    try {
      if (conversa?.id) {
        brokenImages.value.add(conversa.id)
      }
      if (event?.target) {
        event.target.style.display = 'none'
      }
    } catch (error) {
      console.warn('Erro em handleImageError:', error)
    }
  }

  const clearCache = () => {
    try {
      avatarCache.clear()
      brokenImages.value.clear()
    } catch (error) {
      console.warn('Erro em clearCache:', error)
    }
  }

  // =============== FUNÇÕES AUXILIARES ADICIONAIS ===============

  const getDisplayName = (conversa) => {
    try {
      if (!conversa) return 'Conversa'

      if (conversa.processed_display_name) return conversa.processed_display_name
      if (conversa.contact_name && conversa.contact_name !== 'Contato sem nome') return conversa.contact_name
      if (conversa.subject) return conversa.subject
      if (conversa.contact?.display_name) return conversa.contact.display_name
      if (conversa.phone) return formatPhoneNumber(conversa.phone)
      return `Conversa #${conversa.id}`
    } catch (error) {
      console.warn('Erro em getDisplayName:', error)
      return 'Conversa'
    }
  }

  const getProfilePhoto = (conversa) => {
    try {
      if (!conversa) return null

      const candidates = [
        conversa.processed_avatar,
        conversa.avatar_url,
        conversa.contact?.best_profile_photo,
        conversa.whatsapp_profile_picture
      ]

      for (const raw of candidates) {
        if (!raw) continue
        const full = ensureAbsoluteUrl(raw)
        if (isValidImageUrl(full)) return full
      }

      return null
    } catch (error) {
      console.warn('Erro em getProfilePhoto:', error)
      return null
    }
  }

  // ✅ RETORNAR TODAS AS FUNÇÕES
  const helpers = {
    // Funções principais
    getAvatarColor,
    getInitials,
    formatTime,
    getLastMessagePreview,
    processarAvatarConversa, // ✅ FUNÇÃO QUE ESTAVA FALTANDO
    handleImageError,
    brokenImages,
    clearCache,
    
    // Funções auxiliares
    getDisplayName,
    getProfilePhoto,
    formatPhoneNumber,
    ensureAbsoluteUrl,
    isValidImageUrl
  }

  return helpers
}