// modules/Chat/resources/js/composables/useConversationList.js
import { ref, computed, shallowRef, nextTick } from 'vue'
import chatStore from '../servicos/chatStore.js'
import servicoApi from '../servicos/api.js'

export function useConversationList() {
  // ===== ESTADO =====
  const isLoading = ref(false)
  const isLoadingMore = ref(false)
  const currentPage = ref(1)
  const hasMorePages = ref(true)
  const itemsPerPage = ref(20)
  const lastQueryTime = ref(null)
  const loadingController = ref(null)

  // ===== COMPUTED =====
  const conversas = computed(() => chatStore.conversas.value || []) // ✅ GARANTIR ARRAY
  const totalConversas = computed(() => conversas.value?.length || 0)
  const isInitialLoading = computed(() => isLoading.value && totalConversas.value === 0)

  // ===== CARREGAMENTO =====
  async function loadConversations(options = {}) {
    const {
      reset = false,
      search = '',
      filters = {},
      showAll = false
    } = options

    // Cancelar requisição anterior se existir
    if (loadingController.value) {
      loadingController.value.abort()
    }

    if (reset) {
      currentPage.value = 1
      hasMorePages.value = true
      isLoading.value = true
    } else {
      if (!hasMorePages.value || isLoadingMore.value || isLoading.value) {
        return { success: false, reason: 'conditions_not_met' }
      }
      isLoadingMore.value = true
    }

    try {
      loadingController.value = new AbortController()
      const timeoutId = setTimeout(() => loadingController.value.abort(), 30000)

      const params = {
        page: currentPage.value,
        per_page: itemsPerPage.value,
        search: search?.trim() || undefined,
        show_all: showAll,
        ...filters
      }

      // ✅ LOG PARA DEBUG
      console.log('🔄 Carregando conversas com parâmetros:', params)

      const startTime = performance.now()
      const response = await servicoApi.getConversations(params, {
        signal: loadingController.value.signal
      })
      
      clearTimeout(timeoutId)
      lastQueryTime.value = Math.round(performance.now() - startTime)

      // ✅ VALIDAÇÃO MAIS ROBUSTA
      if (!response || !response.data) {
        throw new Error('Resposta inválida da API')
      }

      // ✅ GARANTIR QUE É ARRAY
      const rawData = Array.isArray(response.data) ? response.data : []
      const validConversations = rawData.filter(conv => 
        conv && typeof conv === 'object' && conv.id
      )

      console.log('✅ Conversas carregadas:', {
        total: validConversations.length,
        permitidas: validConversations.length,
        processadas: validConversations.length
      })

      if (reset) {
        chatStore.setConversas(validConversations)
      } else {
        const existingIds = new Set(conversas.value.map(c => c.id))
        const newConversations = validConversations.filter(c => !existingIds.has(c.id))
        
        if (newConversations.length > 0) {
          chatStore.setConversas([...conversas.value, ...newConversations])
        }
      }

      const meta = response.meta || {}
      hasMorePages.value = Boolean(
        meta.has_more || 
        validConversations.length === itemsPerPage.value ||
        (meta.current_page && meta.last_page && meta.current_page < meta.last_page)
      )

      currentPage.value = meta.current_page ? meta.current_page + 1 : currentPage.value + 1

      return { 
        success: true, 
        loaded: validConversations.length, 
        hasMore: hasMorePages.value,
        queryTime: lastQueryTime.value
      }

    } catch (error) {
      if (error.name === 'AbortError') {
        return { success: false, reason: 'aborted' }
      }
      
      console.error('Erro ao carregar conversas:', error)
      return { success: false, error: error.message }
    } finally {
      isLoading.value = false
      isLoadingMore.value = false
      loadingController.value = null
    }
  }

  async function reloadConversations(options = {}) {
    return await loadConversations({ ...options, reset: true })
  }

  function cancelLoading() {
    if (loadingController.value) {
      loadingController.value.abort()
    }
  }

  function resetPagination() {
    currentPage.value = 1
    hasMorePages.value = true
  }

  // ===== CLEANUP =====
  function cleanup() {
    cancelLoading()
    resetPagination()
  }

  return {
    // Estado
    conversas,
    totalConversas,
    isLoading,
    isLoadingMore,
    isInitialLoading,
    hasMorePages,
    lastQueryTime,
    
    // Métodos
    loadConversations,
    reloadConversations,
    cancelLoading,
    resetPagination,
    cleanup
  }
}