// modules/Chat/resources/js/composables/useConversationManager.js
import { ref, computed, nextTick } from 'vue'
import { useRouter } from 'vue-router'
import { useApp } from '@/Core/composables/useApp'
import servicoApi from '@Chat/servicos/api.js'
import chatStore from '@Chat/servicos/chatStore.js'
import websocketService from '@Chat/servicos/websocket.js'
import { useConversationHelpers } from './useConversationHelpers.js'
import { useChatNotifications } from './useChatNotifications.js'

export function useConversationManager() {
  const router = useRouter()
  const { currentUser } = useApp()
  const { processarAvatarConversa } = useConversationHelpers()
  const chatNotifications = useChatNotifications()

  const carregandoConversas = ref(false)
  const showAllConversations = ref(false)
  const requestCache = new Map()

  // =============== PERMISSÕES ===============

  const chatPermissions = computed(() => {
    try {
      const chatConfig = window.Innoclapps?.scriptConfig?.('chat')
      return chatConfig?.permissions || {}
    } catch (error) {
      console.warn('Erro ao obter permissões do chat:', error)
      return {}
    }
  })

  const isAdmin = computed(() => currentUser.value?.super_admin || false)

  // ✅ VALIDAÇÕES DE PERMISSÃO
  function canCreateConversation() {
    if (isAdmin.value) return true
    return chatPermissions.value.create_conversations || false
  }

  function canViewAllConversations() {
    if (isAdmin.value) return true
    return chatPermissions.value.view_all_conversations || false
  }

  function canViewConversation(conversation) {
    if (isAdmin.value) return true
    
    const permissions = chatPermissions.value
    const userId = currentUser.value?.id

    // Verificar permissão base
    if (!permissions.access_chat_module) return false

    // Pode ver todas
    if (permissions.view_all_conversations) return true

    // É dono da conversa
    if (conversation && (
      parseInt(conversation.assigned_user_id) === parseInt(userId) ||
      conversation.assigned_users?.some(a => parseInt(a.user?.id || a.user_id) === parseInt(userId))
    )) {
      return permissions.view_own_conversations || true
    }

    // Pode ver não atribuídas
    if (conversation && !conversation.assigned_user_id && !conversation.assigned_users?.length) {
      return permissions.view_unassigned_conversations || false
    }

    // Pode ver da equipe (implementar lógica de equipe se necessário)
    if (permissions.view_team_conversations) {
      // Implementar verificação de equipe
      return true
    }

    return false
  }

  function canSendMessage(conversation) {
    if (isAdmin.value) return true
    
    const permissions = chatPermissions.value
    
    // Verificar permissão base
    if (!permissions.access_chat_module) return false
    
    // Verificar se pode ver a conversa primeiro
    if (!canViewConversation(conversation)) return false
    
    // Verificar permissões de envio
    if (conversation?.type === 'internal') {
      return permissions.send_internal_messages || permissions.send_messages
    } else {
      return permissions.send_external_messages || permissions.send_messages
    }
  }

  // =============== NORMALIZAÇÃO E PROCESSAMENTO ===============

  const normalizeResponse = (response) => {
    if (!response) return []
    if (Array.isArray(response)) return response
    if (response.data) {
      if (Array.isArray(response.data)) return response.data
      if (response.data.data && Array.isArray(response.data.data)) {
        return response.data.data
      }
    }
    return []
  }

  const processConversationsInBatches = async (conversas, batchSize = 20) => {
    const results = []

    for (let i = 0; i < conversas.length; i += batchSize) {
      const batch = conversas.slice(i, i + batchSize)
      const processed = batch.map(conversa => {
        try {
          const avatarInfo = processarAvatarConversa(conversa)

          return {
            ...conversa,
            type: conversa.type || (conversa.participants ? 'internal' : 'external'),
            unread_count: conversa.unread_count || 0,
            contact_name: conversa.contact_name || conversa.subject || `Conversa #${conversa.id}`,
            last_message: conversa.last_message || '',
            last_message_at: conversa.last_message_at || conversa.updated_at || conversa.created_at,
            processed_avatar: avatarInfo.avatar_url,
            processed_display_name: avatarInfo.display_name,
            processed_participant: avatarInfo.participant,
          }
        } catch (error) {
          console.warn('Erro ao processar conversa:', conversa.id, error)
          return {
            ...conversa,
            type: conversa.type || 'external',
            unread_count: conversa.unread_count || 0,
            contact_name: conversa.contact_name || `Conversa #${conversa.id}`,
            last_message: conversa.last_message || '',
            last_message_at: conversa.last_message_at || conversa.created_at,
            processed_avatar: null,
            processed_display_name: conversa.contact_name || `Conversa #${conversa.id}`,
            processed_participant: null,
          }
        }
      })

      results.push(...processed)

      // Yield para não bloquear UI
      if (i + batchSize < conversas.length) {
        await new Promise(resolve => setTimeout(resolve, 0))
      }
    }

    return results
  }

  // =============== CARREGAMENTO DE CONVERSAS ===============

  const carregarConversas = async (filterId = 'todas') => {
    // ✅ VERIFICAR PERMISSÃO BASE
    if (!isAdmin.value && !chatPermissions.value.access_chat_module) {
      chatNotifications.showErrorNotification(
        'Acesso Negado',
        'Você não tem permissão para acessar conversas'
      )
      return
    }

    if (carregandoConversas.value) return

    try {
      carregandoConversas.value = true
      chatStore.isLoading.value = true

      const cacheKey = `${filterId}-${showAllConversations.value}`
      const now = Date.now()

      // Verificar cache
      if (requestCache.has(cacheKey)) {
        const cached = requestCache.get(cacheKey)
        if (now - cached.timestamp < 30000) {
          chatStore.setConversas(cached.data)
          return
        }
      }

      // ✅ LÓGICA CORRIGIDA COM PARÂMETROS ESPECÍFICOS
      let apiParams = {
        include_archived: filterId === 'arquivadas'
      }

      if (isAdmin.value) {
        if (showAllConversations.value) {
          // ✅ ADMIN QUER VER TODAS: usar parâmetro específico
          apiParams.show_all_conversations = true
          console.log('🔍 Admin vendo TODAS as conversas do sistema')
        } else {
          // ✅ ADMIN PADRÃO: suas conversas + não atribuídas
          console.log('🔍 Admin vendo suas conversas + não atribuídas')
        }
      } else {
        // Para usuário normal: comportamento normal
        if (showAllConversations.value) {
          apiParams.show_my_only = true
          console.log('🔍 Usuário normal vendo apenas suas conversas')
        } else {
          console.log('🔍 Usuário normal vendo suas conversas + não atribuídas')
        }
      }

      console.log('🔄 Carregando conversas com parâmetros:', {
        isAdmin: isAdmin.value,
        showAllConversations: showAllConversations.value,
        apiParams
      })

      const response = filterId === 'arquivadas'
        ? await servicoApi.getArchivedConversations(apiParams)
        : await servicoApi.getConversations(apiParams)

      const todasConversas = normalizeResponse(response)

      if (todasConversas.length > 0) {
        // ✅ FILTRAR CONVERSAS BASEADO EM PERMISSÕES
        const conversasPermitidas = todasConversas.filter(conversa => 
          canViewConversation(conversa)
        )

        const processedConversations = await processConversationsInBatches(conversasPermitidas)
        chatStore.setConversas(processedConversations)

        requestCache.set(cacheKey, {
          data: processedConversations,
          timestamp: now,
        })

        await carregarConversaDaUrl()

        console.log('✅ Conversas carregadas:', {
          total: todasConversas.length,
          permitidas: conversasPermitidas.length,
          processadas: processedConversations.length
        })
      } else {
        chatStore.setConversas([])
        console.log('✅ Nenhuma conversa encontrada')
      }

    } catch (error) {
      console.error('Erro ao carregar conversas:', error)
      chatStore.setConversas([])
      chatNotifications.showErrorNotification(
        'Erro',
        'Falha ao carregar conversas',
        error.message || ''
      )
    } finally {
      carregandoConversas.value = false
      chatStore.isLoading.value = false
    }
  }

  // =============== PERMISSÕES E VALIDAÇÕES ===============

  const verificarSeDeveMarcarComoLida = (conversa) => {
    try {
      if (!conversa) return false

      // ✅ VERIFICAR PERMISSÃO PRIMEIRO
      if (!canViewConversation(conversa)) return false

      const currentUserId = websocketService?.getCurrentUser()?.id
      if (!currentUserId) return false

      // Admin vendo todas as conversas
      if (showAllConversations.value && websocketService?.getCurrentUser()?.super_admin) {
        return true
      }

      // Verificar atribuição única
      const assignedUserId = conversa.assigned_user_id
      if (assignedUserId) {
        return parseInt(assignedUserId) === parseInt(currentUserId)
      }

      // Verificar atribuições múltiplas
      const assignedUsers = conversa.assigned_users
      if (assignedUsers && Array.isArray(assignedUsers) && assignedUsers.length > 0) {
        return assignedUsers.some(assignment => {
          const userId = assignment.user?.id || assignment.user_id || assignment.id
          return userId && parseInt(userId) === parseInt(currentUserId)
        })
      }

      // Verificar participantes (conversas internas)
      if (conversa.type === 'internal' && conversa.participants) {
        return conversa.participants.some(participant => 
          parseInt(participant.id) === parseInt(currentUserId)
        )
      }

      // Sem atribuição - qualquer um pode marcar (se tiver permissão)
      return chatPermissions.value.view_unassigned_conversations || isAdmin.value
    } catch (error) {
      console.error('Erro ao verificar atribuição:', error)
      return false
    }
  }

  const marcarComoLidaViaApi = async (conversationId) => {
    try {
      // ✅ VERIFICAR PERMISSÃO ANTES DE MARCAR
      const conversa = chatStore.conversas.value.find(c => c.id === conversationId)
      if (!verificarSeDeveMarcarComoLida(conversa)) {
        console.warn('Sem permissão para marcar conversa como lida:', conversationId)
        return
      }

      // Marcar localmente primeiro
      chatStore.markConversaAsRead(conversationId)
      
      // Enviar para backend
      const resultado = await servicoApi.markAsRead(conversationId)

      if (resultado.success !== false) {
        if (websocketService?.isValid()) {
          websocketService.sendEvent('mark_as_read', {
            conversation_id: conversationId,
            user_id: websocketService.getCurrentUser()?.id,
            read_at: new Date().toISOString(),
          })
        }
      }
    } catch (error) {
      console.error('Erro ao marcar como lida:', error)
      
      // Reverter marcação local em caso de erro
      const conversa = chatStore.conversas.value.find(c => c.id === conversationId)
      if (conversa && conversa.unread_count === 0) {
        chatStore.updateConversa({
          ...conversa,
          unread_count: 1
        })
      }
    }
  }

  // =============== SELEÇÃO E NAVEGAÇÃO ===============

  const selecionarConversa = async (conversa) => {
    try {
      if (!conversa || chatStore.conversaSelecionada.value?.id === conversa.id) return

      // ✅ VERIFICAR PERMISSÃO ANTES DE SELECIONAR
      if (!canViewConversation(conversa)) {
        chatNotifications.showErrorNotification(
          'Acesso Negado',
          'Você não tem permissão para visualizar esta conversa'
        )
        return
      }

      // Sair da conversa anterior
      if (chatStore.conversaSelecionada.value?.id && websocketService?.isValid()) {
        await websocketService.leaveConversation(chatStore.conversaSelecionada.value.id)
      }

      // Selecionar nova conversa
      chatStore.selectConversa(conversa)

      // Entrar na nova conversa
      if (websocketService?.isValid()) {
        await websocketService.joinConversation(conversa.id, conversa.type)
      }

      const promises = [atualizarUrlConversa(conversa.id)]
      
      // Marcar como lida se permitido
      if (verificarSeDeveMarcarComoLida(conversa)) {
        promises.push(marcarComoLidaViaApi(conversa.id))
      }

      await Promise.allSettled(promises)
    } catch (error) {
      console.error('Erro ao selecionar conversa:', error)
      chatNotifications.showErrorNotification(
        'Erro',
        'Falha ao selecionar conversa',
        error.message || ''
      )
    }
  }

  const atualizarUrlConversa = async (conversationId) => {
    try {
      await nextTick()
      const currentQuery = router.currentRoute.value.query
      if (currentQuery.conversation !== conversationId.toString()) {
        await router.replace({
          query: { ...currentQuery, conversation: conversationId },
        })
      }
    } catch (error) {
      console.warn('Erro ao atualizar URL:', error)
    }
  }

  const carregarConversaDaUrl = async () => {
    try {
      const conversationId = router.currentRoute.value.query.conversation
      if (conversationId && chatStore.conversas.value.length > 0) {
        const conversa = chatStore.conversas.value.find(
          c => c.id === parseInt(conversationId)
        )
        if (conversa && canViewConversation(conversa)) {
          await selecionarConversa(conversa)
        }
      }
    } catch (error) {
      console.warn('Erro ao carregar conversa da URL:', error)
    }
  }

  // =============== TRATAMENTO DE EVENTOS ===============

  const tratarMensagemEnviada = async (mensagemData) => {
    try {
      if (mensagemData && chatStore.conversaSelecionada.value?.id) {
        // ✅ VERIFICAR PERMISSÃO PARA ENVIAR MENSAGEM
        if (!canSendMessage(chatStore.conversaSelecionada.value)) {
          chatNotifications.showErrorNotification(
            'Acesso Negado',
            'Você não tem permissão para enviar mensagens nesta conversa'
          )
          return
        }

        const currentUserId = websocketService?.getCurrentUser()?.id
        if (currentUserId) {
          chatStore.updateConversationList(
            chatStore.conversaSelecionada.value.id,
            mensagemData,
            true
          )
        }
      }
    } catch (error) {
      console.error('Erro ao processar mensagem enviada:', error)
    }
  }

  const tratarNovaConversa = async (dadosConversa) => {
    try {
      // ✅ VERIFICAR PERMISSÃO PARA CRIAR
      if (!canCreateConversation()) {
        chatNotifications.showErrorNotification(
          'Acesso Negado',
          'Você não tem permissão para criar conversas'
        )
        return { success: false, error: 'Sem permissão para criar conversas' }
      }

      const response = dadosConversa.type === 'internal'
        ? await servicoApi.createInternalConversation(dadosConversa)
        : await servicoApi.createConversation(dadosConversa)

      if (response?.conversation || response?.data?.conversation) {
        const novaConversa = response.conversation || response.data.conversation
        const avatarInfo = processarAvatarConversa({
          ...novaConversa,
          type: dadosConversa.type,
        })

        const conversaProcessada = {
          ...novaConversa,
          type: dadosConversa.type,
          unread_count: 0,
          contact_name: novaConversa.contact_name || dadosConversa.contact_name || 'Nova Conversa',
          last_message: '',
          last_message_at: novaConversa.created_at,
          processed_avatar: avatarInfo.avatar_url,
          processed_display_name: avatarInfo.display_name,
          processed_participant: avatarInfo.participant,
        }

        chatStore.addConversation(conversaProcessada)
        await selecionarConversa(conversaProcessada)

        const notificacaoResposta = response.notification || {
          type: 'success',
          title: 'Sucesso',
          message: `Conversa ${dadosConversa.type === 'internal' ? 'interna' : 'externa'} criada!`,
        }
        
        chatNotifications.showSuccessNotification(
          notificacaoResposta.title,
          notificacaoResposta.message
        )

        return { success: true, conversation: conversaProcessada }
      }
    } catch (error) {
      console.error('Erro ao criar conversa:', error)
      
      // Tratar conversa existente
      if (error.response?.status === 409) {
        const existingConversation = error.response?.data?.conversation
        if (existingConversation) {
          const avatarInfo = processarAvatarConversa(existingConversation)
          await selecionarConversa({
            ...existingConversation,
            processed_avatar: avatarInfo.avatar_url,
            processed_display_name: avatarInfo.display_name,
            processed_participant: avatarInfo.participant,
          })
          chatNotifications.showInfoNotification('Informação', 'Conversa já existe')
          return { success: true, conversation: existingConversation, existed: true }
        }
      }
      
      const mensagem = error.response?.data?.message || error.message || 'Erro ao criar conversa'
      chatNotifications.showErrorNotification('Erro', mensagem)
      return { success: false, error: mensagem }
    }
  }

  const tratarConversaAtualizada = (updatedConversation) => {
    try {
      // ✅ VERIFICAR SE AINDA TEM PERMISSÃO PARA VER A CONVERSA
      if (!canViewConversation(updatedConversation)) {
        chatStore.removeConversation(updatedConversation.id)
        
        // Se era a conversa selecionada, limpar seleção
        if (chatStore.conversaSelecionada.value?.id === updatedConversation.id) {
          chatStore.selectConversa(null)
        }
        
        chatNotifications.showInfoNotification(
          'Conversa Removida',
          'Esta conversa não está mais disponível para você'
        )
        return
      }

      if (!updatedConversation.processed_avatar || !updatedConversation.processed_display_name) {
        const avatarInfo = processarAvatarConversa(updatedConversation)
        updatedConversation.processed_avatar = avatarInfo.avatar_url
        updatedConversation.processed_display_name = avatarInfo.display_name
        updatedConversation.processed_participant = avatarInfo.participant
      }

      chatStore.updateConversationMeta(updatedConversation)

      if (chatStore.conversaSelecionada.value?.id === updatedConversation.id) {
        Object.assign(chatStore.conversaSelecionada.value, updatedConversation)
      }

      // Recarregar lista se conversa foi removida do escopo atual
      const shouldReload = updatedConversation.status === 'archived' ||
                          updatedConversation.status === 'deleted' ||
                          (updatedConversation.assigned_user_id !== currentUser.value?.id &&
                           !websocketService?.getCurrentUser()?.super_admin &&
                           !showAllConversations.value)

      if (shouldReload) {
        carregarConversas()
      }
    } catch (error) {
      console.error('Erro ao tratar conversa atualizada:', error)
    }
  }

  // =============== UTILITÁRIOS ===============

  const toggleShowAllConversations = async () => {
    // ✅ VERIFICAR PERMISSÃO ANTES DE ALTERNAR
    if (!canViewAllConversations()) {
      chatNotifications.showErrorNotification(
        'Acesso Negado',
        'Você não tem permissão para ver todas as conversas'
      )
      return
    }

    showAllConversations.value = !showAllConversations.value
    requestCache.clear()
    await carregarConversas()
  }

  const clearCache = () => {
    requestCache.clear()
  }

  return {
    // Estado
    carregandoConversas,
    showAllConversations,
    
    // Métodos principais
    carregarConversas,
    selecionarConversa,
    tratarMensagemEnviada,
    tratarNovaConversa,
    tratarConversaAtualizada,
    toggleShowAllConversations,
    
    // Métodos auxiliares
    carregarConversaDaUrl,
    marcarComoLidaViaApi,
    verificarSeDeveMarcarComoLida,
    clearCache,
    
    // ✅ VALIDAÇÕES DE PERMISSÃO
    canCreateConversation,
    canViewAllConversations,
    canViewConversation,
    canSendMessage,
    
    // Utilitários
    normalizeResponse,
    processConversationsInBatches
  }
}