// modules/Chat/resources/js/composables/useInfiniteScroll.js
import { ref, onUnmounted } from 'vue'

export function useInfiniteScroll(options = {}) {
  const {
    onLoadMore,
    hasMore,
    isLoading,
    threshold = 100,
    debounceTime = 150,
    enableVirtualScrolling = false,
    enableMetrics = false
  } = options

  // ===== ESTADO =====
  const containerRef = ref(null)
  const scrollTimeout = ref(null)
  const isNearBottom = ref(false)
  const lastScrollTop = ref(0)
  const scrollDirection = ref('down')

  // ===== HELPERS =====
  function debounce(func, delay) {
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(scrollTimeout.value)
        func(...args)
      }
      clearTimeout(scrollTimeout.value)
      scrollTimeout.value = setTimeout(later, delay)
    }
  }

  function checkScrollPosition(container) {
    if (!container) return

    const { scrollTop, scrollHeight, clientHeight } = container
    
    // Detectar direção do scroll
    if (scrollTop > lastScrollTop.value) {
      scrollDirection.value = 'down'
    } else if (scrollTop < lastScrollTop.value) {
      scrollDirection.value = 'up'
    }
    lastScrollTop.value = scrollTop

    // Verificar se está próximo ao final
    const distanceFromBottom = scrollHeight - (scrollTop + clientHeight)
    isNearBottom.value = distanceFromBottom <= threshold

    // Carregar mais se necessário
    if (isNearBottom.value && 
        scrollDirection.value === 'down' &&
        hasMore?.value && 
        !isLoading?.value &&
        onLoadMore) {
      onLoadMore()
    }
  }

  const debouncedScrollCheck = debounce(checkScrollPosition, debounceTime)

  function handleScroll(event) {
    const container = event.target
    debouncedScrollCheck(container)
  }

  function scrollToTop(smooth = true) {
    if (containerRef.value) {
      containerRef.value.scrollTo({
        top: 0,
        behavior: smooth ? 'smooth' : 'auto'
      })
    }
  }

  function scrollToBottom(smooth = true) {
    if (containerRef.value) {
      containerRef.value.scrollTo({
        top: containerRef.value.scrollHeight,
        behavior: smooth ? 'smooth' : 'auto'
      })
    }
  }

  function scrollToElement(elementId, smooth = true) {
    if (containerRef.value) {
      const element = containerRef.value.querySelector(`#${elementId}`)
      if (element) {
        element.scrollIntoView({
          behavior: smooth ? 'smooth' : 'auto',
          block: 'center'
        })
      }
    }
  }

  // ===== VIRTUAL SCROLLING =====
  const visibleRange = ref({ start: 0, end: 50 })
  const itemHeight = ref(80)
  const bufferSize = ref(10)

  function updateVisibleRange() {
    if (!containerRef.value) return

    const { scrollTop, clientHeight } = containerRef.value
    const start = Math.floor(scrollTop / itemHeight.value)
    const end = Math.ceil((scrollTop + clientHeight) / itemHeight.value)

    visibleRange.value = {
      start: Math.max(0, start - bufferSize.value),
      end: end + bufferSize.value
    }
  }

  const debouncedRangeUpdate = debounce(updateVisibleRange, 50)

  function handleVirtualScroll(event) {
    debouncedRangeUpdate()
    handleScroll(event)
  }

  // ===== PERFORMANCE MONITORING =====
  const scrollMetrics = ref({
    totalScrollEvents: 0,
    averageScrollTime: 0,
    lastScrollTime: 0
  })

  function trackScrollPerformance() {
    const now = performance.now()
    scrollMetrics.value.totalScrollEvents++
    
    if (scrollMetrics.value.lastScrollTime > 0) {
      const timeDiff = now - scrollMetrics.value.lastScrollTime
      scrollMetrics.value.averageScrollTime = 
        (scrollMetrics.value.averageScrollTime + timeDiff) / 2
    }
    
    scrollMetrics.value.lastScrollTime = now
  }

  function handleScrollWithMetrics(event) {
    if (enableMetrics) {
      trackScrollPerformance()
    }
    handleScroll(event)
  }

  // ===== INTERSECTION OBSERVER =====
  let intersectionObserver = null

  function setupIntersectionObserver() {
    if (!('IntersectionObserver' in window) || !containerRef.value) return

    const sentinel = document.createElement('div')
    sentinel.className = 'scroll-sentinel'
    sentinel.style.height = '1px'
    sentinel.style.position = 'absolute'
    sentinel.style.bottom = `${threshold}px`
    sentinel.style.width = '100%'
    sentinel.style.pointerEvents = 'none'
    
    containerRef.value.appendChild(sentinel)

    intersectionObserver = new IntersectionObserver(
      (entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting && 
              hasMore?.value && 
              !isLoading?.value &&
              onLoadMore) {
            onLoadMore()
          }
        })
      },
      {
        root: containerRef.value,
        rootMargin: '0px',
        threshold: 0.1
      }
    )

    intersectionObserver.observe(sentinel)
  }

  function cleanupIntersectionObserver() {
    if (intersectionObserver) {
      intersectionObserver.disconnect()
      intersectionObserver = null
    }

    if (containerRef.value) {
      const sentinel = containerRef.value.querySelector('.scroll-sentinel')
      if (sentinel) {
        sentinel.remove()
      }
    }
  }

  // ===== ESCOLHER HANDLER BASEADO NAS OPÇÕES =====
  function getScrollHandler() {
    if (enableVirtualScrolling) {
      return handleVirtualScroll
    } else if (enableMetrics) {
      return handleScrollWithMetrics
    } else {
      return handleScroll
    }
  }

  // ===== LIFECYCLE =====
  function setup(useIntersectionObserver = false) {
    if (useIntersectionObserver) {
      setupIntersectionObserver()
    }
  }

  function cleanup() {
    if (scrollTimeout.value) {
      clearTimeout(scrollTimeout.value)
    }
    cleanupIntersectionObserver()
  }

  onUnmounted(() => {
    cleanup()
  })

  return {
    // Refs
    containerRef,
    isNearBottom,
    scrollDirection,
    visibleRange,
    scrollMetrics,

    // Métodos de scroll
    handleScroll: getScrollHandler(),
    scrollToTop,
    scrollToBottom,
    scrollToElement,

    // Configuração
    setup,
    cleanup,

    // Virtual scrolling
    updateVisibleRange,
    itemHeight,
    bufferSize
  }
}