// modules/Chat/resources/js/composables/useMessageHandling.js
import { ref, shallowRef, computed, nextTick } from 'vue'
import servicoApi from '../servicos/api.js'
import chatStore from '../servicos/chatStore.js'
import { useConversationData } from './useConversationData.js'
import { ensureAbsoluteUrl } from '../utils/urlHelpers.js'

export function useMessageHandling() {
  const conversationData = useConversationData()
  
  // ===== ESTADO =====
  const mensagens = shallowRef([])
  const carregandoMensagens = ref(false)
  const carregandoMaisMensagens = ref(false)
  const enviandoMensagem = ref(false)
  const paginaAtual = ref(1)
  const temMaisMensagens = ref(false)
  const mensagemDestacada = ref(null)
  
  // ===== CACHES =====
  const processedMessageIds = new Set()
  const messageCache = new Map()
  const statusCache = new Map()
  
  // ===== TIMERS =====
  let highlightTimeout = null
  const currentUserId = computed(() => conversationData.getCurrentUserId())
  
  // ===== HELPERS =====
  function toInt(value, fallback = 0) {
    const n = parseInt(value)
    return Number.isNaN(n) ? fallback : n
  }
  
  function isCurrentUserMessage(mensagem) {
    const userId = currentUserId.value
    return userId !== null &&
      toInt(mensagem.sender_id, -1) === userId &&
      !mensagem.is_from_contact
  }
  
  function getSenderDisplayName(mensagem) {
    const userId = currentUserId.value
    const isMine = userId !== null &&
      toInt(mensagem.sender_id, -1) === userId &&
      !mensagem.is_from_contact

    if (isMine) {
      return 'Você'
    }

    // Para mensagens de contato
    if (mensagem.is_from_contact) {
      return mensagem.sender_name ||
             mensagem.from_name ||
             'Contato'
    }

    // Para mensagens internas
    return mensagem.sender_name ||
           mensagem.sender?.name ||
           mensagem.from_name ||
           'Usuário'
  }
  
  // ✅ FUNÇÃO PARA ORDENAR MENSAGENS (MAIS NOVAS EMBAIXO)
function sortMessagesByDate(messages) {
  if (!Array.isArray(messages)) return []
  
  return [...messages].sort((a, b) => {
    // ✅ USAR sent_at como prioridade, depois created_at
    const dateA = new Date(a.sent_at || a.created_at || a.timestamp || 0)
    const dateB = new Date(b.sent_at || b.created_at || b.timestamp || 0)
    
    // ✅ ORDEM CRESCENTE: mais antigas primeiro, mais novas embaixo
    return dateA.getTime() - dateB.getTime()
  })
}
  
  // ===== PROCESSAMENTO DE MENSAGENS =====
function processarMensagem(msg, conversa = null) {
  const m = { ...msg }
  const userId = currentUserId.value

  // ✅ CORREÇÃO: Usar sent_at como prioridade para WhatsApp
  m.created_at = m.sent_at ||        // ← PRIORIDADE para WhatsApp
              m.created_at || 
              m.createdAt || 
              m.timestamp ||
              new Date().toISOString()

  // Determinar se é de contato
  let isFromContact
  if (conversa?.type === 'internal') {
    isFromContact = toInt(m.sender_id, -1) !== toInt(userId, -1)
  } else {
    if (m.hasOwnProperty('is_from_contact')) {
      isFromContact = Boolean(m.is_from_contact)
    } else if (m.hasOwnProperty('is_from_me')) {
      isFromContact = !Boolean(m.is_from_me)
    } else {
      isFromContact = Boolean(m.is_from_contact)
    }
  }
  m.is_from_contact = isFromContact

  // Nome do remetente
  if (!isFromContact && !m.sender_id && m.sender_name) {
    m.sender_name = m.sender_name
  } else {
    m.sender_name = m.sender_name || m.sender?.name || m.from_name || 'Usuário'
  }

  // Avatar usando o composable
  m.sender_avatar = conversationData.getMessageAvatar(m, conversa)

  // Processar metadata
  let metadata = m.metadata || {}
  if (typeof metadata === 'string') {
    try {
      metadata = JSON.parse(metadata)
    } catch (e) {
      metadata = {}
    }
  }

  // Processar anexos
  const processedAttachments = []
  const rawAttachments = Array.isArray(m.attachments)
    ? m.attachments
    : Array.isArray(metadata.attachments)
      ? metadata.attachments
      : []

  rawAttachments.forEach(att => {
    const a = { ...att }
    const url = a.file_url || a.external_url || a.url || null
    a.file_url = url ? ensureAbsoluteUrl(url) : null
    if (a.thumbnail_url) a.thumbnail_url = ensureAbsoluteUrl(a.thumbnail_url)
    a.file_type = a.file_type || a.mime_type || a.type || ''
    a.file_name = a.file_name || a.name || ''
    a.download_status = a.download_status || 'completed'
    processedAttachments.push(a)
  })

  m.attachments = processedAttachments

  // Normalizar status
  m.status = normalizarStatusMensagem(
    m.status,
    m.delivery_status || m.metadata?.whatsapp_status
  )

  m.id = String(m.id)

  return {
    id: m.id,
    conversation_id: m.conversation_id,
    content: m.content || m.message || '',
    is_from_contact: m.is_from_contact,
    sender_id: m.sender_id,
    sender_name: getSenderDisplayName(m),
    sender_avatar: m.sender_avatar,
    created_at: m.created_at, // ← Agora usa sent_at como prioridade
    sent_at: m.sent_at,       // ← Manter original também
    status: m.status,
    attachments: m.attachments,
    metadata,
    _isTemporary: false,
    _isOutbound: true
  }
}
  
  function normalizarStatusMensagem(rawStatus, deliveryStatus = null) {
    const s = (rawStatus || deliveryStatus || '')
      .toString()
      .toLowerCase()

    if (!s) return 'sent'

    if (['pending', 'sending', 'queue', 'queued', 'waiting'].includes(s)) {
      return 'sending'
    }

    if (['sent', 'server_sent', 'client_sent', 'submitted'].includes(s)) {
      return 'sent'
    }

    if (['delivered', 'server_ack', 'device_delivered'].includes(s)) {
      return 'delivered'
    }

    if (['read', 'seen', 'viewed'].includes(s)) {
      return 'read'
    }

    if (['failed', 'error', 'canceled', 'cancelled'].includes(s)) {
      return 'failed'
    }

    if (['received'].includes(s)) {
      return 'received'
    }

    return s
  }
  
  // ===== CARREGAMENTO =====
  async function carregarMensagens(conversaId, reset = false) {
  if (!conversaId) return { success: false, error: 'ID da conversa é obrigatório' }
  
  if (reset) {
    carregandoMensagens.value = true
    paginaAtual.value = 1
    temMaisMensagens.value = false
  } else {
    if (!temMaisMensagens.value || carregandoMaisMensagens.value) {
      return { success: false, reason: 'Não há mais mensagens ou já carregando' }
    }
    carregandoMaisMensagens.value = true
  }

  try {
    const beforeId = !reset && mensagens.value.length > 0 ? mensagens.value[0].id : null
    
    const response = await servicoApi.getMessages(conversaId, {
      page: reset ? 1 : paginaAtual.value + 1,
      per_page: 50,
      sort: 'sent_at',  // ✅ USAR sent_at para ordenação
      order: 'asc',     // ✅ ORDEM CRESCENTE
      before_id: beforeId
    })

    console.log('📥 Resposta da API:', response)
    
    // ✅ VERIFICAR ESTRUTURA: response.data ao invés de response.messages
    const mensagensCarregadas = Array.isArray(response.data) ? response.data : 
                                Array.isArray(response.messages) ? response.messages : []
    const paginationInfo = response.pagination || {}

    console.log('📋 Mensagens da API (ordem original):', 
      mensagensCarregadas.map(m => ({
        id: m.id,
        sent_at: m.sent_at,
        content: m.content?.substring(0, 30)
      }))
    )

    if (reset) {
      processedMessageIds.clear()
      messageCache.clear()
    }

    const userId = currentUserId.value
    const conversa = chatStore.getConversaById(conversaId)
    
    const processed = mensagensCarregadas
      .filter(msg => {
        if (!msg || typeof msg !== 'object' || !msg.id) {
          console.warn('⚠️ Mensagem inválida ignorada:', msg)
          return false
        }
        return true
      })
      .filter(msg => !processedMessageIds.has(String(msg.id)))
      .map(msg => {
        try {
          processedMessageIds.add(String(msg.id))
          return processarMensagem(msg, conversa)
        } catch (error) {
          console.error('❌ Erro ao processar mensagem:', msg.id, error)
          return null
        }
      })
      .filter(Boolean) // Remove nulls

    console.log('✅ Mensagens processadas (antes da ordenação):', 
      processed.map(m => ({
        id: m.id,
        created_at: m.created_at,
        sent_at: m.sent_at,
        content: m.content?.substring(0, 30)
      }))
    )

    if (reset) {
      // ✅ PRIMEIRA CARGA: ordenar e definir
      mensagens.value = sortMessagesByDate(processed)
    } else {
      // ✅ PAGINAÇÃO: adicionar mensagens antigas no início e reordenar
      const todasMensagens = [...processed, ...mensagens.value]
      mensagens.value = sortMessagesByDate(todasMensagens)
      paginaAtual.value = paginationInfo.current_page || (paginaAtual.value + 1)
    }

    temMaisMensagens.value = paginationInfo.has_more || false

    console.log('🔄 Mensagens finais ordenadas:', 
      mensagens.value.map(m => ({
        id: m.id,
        created_at: m.created_at,
        content: m.content?.substring(0, 30)
      }))
    )

    return { 
      success: true, 
      loaded: processed.length,
      hasMore: temMaisMensagens.value,
      total: mensagens.value.length
    }
    
  } catch (error) {
    console.error('❌ Erro ao carregar mensagens:', error)
    return { success: false, error: error.message }
  } finally {
    carregandoMensagens.value = false
    carregandoMaisMensagens.value = false
  }
}

  // ===== ENVIO =====
  async function enviarMensagem(conversaId, dadosMensagem, conversa = null) {
    if (!conversaId || enviandoMensagem.value) {
      return { success: false, error: 'Conversa inválida ou já enviando' }
    }

    let mensagemTemporaria = null

    try {
      enviandoMensagem.value = true
      const currentUser = conversationData.getCurrentChatUser()

      let content = ''
      if (dadosMensagem instanceof FormData) {
        content = dadosMensagem.get('content') || ''
      } else {
        content = dadosMensagem.content || dadosMensagem.message || ''
      }

      // Criar mensagem temporária
      mensagemTemporaria = {
        id: `temp-${Date.now()}`,
        conversation_id: conversaId,
        content: content,
        is_from_contact: false,
        sender_id: currentUser.id || null,
        sender_name: 'Você',
        sender_avatar: conversationData.getCurrentUserAvatar(),
        created_at: new Date().toISOString(),
        status: 'sending',
        attachments: [],
        metadata: {},
        _isTemporary: true
      }

      // ✅ ADICIONAR MENSAGEM TEMPORÁRIA NO FINAL (MAIS NOVA)
      adicionarMensagem(mensagemTemporaria)

      // Enviar para API
      let resultado
      if (dadosMensagem instanceof FormData) {
        resultado = await servicoApi.sendMessageWithAttachments(conversaId, dadosMensagem)
      } else {
        resultado = await servicoApi.sendMessage(conversaId, dadosMensagem)
      }

      if (resultado) {
        // Remover mensagem temporária
        const tempIndex = mensagens.value.findIndex(msg => msg.id === mensagemTemporaria.id)
        if (tempIndex !== -1) {
          const newMensagens = [...mensagens.value]
          newMensagens.splice(tempIndex, 1)
          mensagens.value = newMensagens
        }

        // ✅ ADICIONAR MENSAGEM REAL NO FINAL
        const nova = resultado.message || resultado.data || resultado
        if (nova) {
          adicionarMensagem(nova)
          
          // Atualizar chatStore
          if (chatStore.updateConversationList) {
            chatStore.updateConversationList(conversaId, nova, true, false)
          }
        }

        return { success: true, message: nova }
      }

      return { success: false, error: 'Resposta inválida da API' }

    } catch (error) {
      console.error('❌ Erro ao enviar mensagem:', error)

      // Marcar mensagem temporária como falha
      if (mensagemTemporaria) {
        const tempIndex = mensagens.value.findIndex(msg => msg.id === mensagemTemporaria.id)
        if (tempIndex !== -1) {
          const newMensagens = [...mensagens.value]
          newMensagens[tempIndex] = {
            ...newMensagens[tempIndex],
            status: 'failed',
            content: newMensagens[tempIndex].content.replace('Enviando', 'Falha ao enviar')
          }
          mensagens.value = newMensagens
        }
      }

      return { success: false, error: error.message }
    } finally {
      enviandoMensagem.value = false
    }
  }
  
  // ===== ADICIONAR/ATUALIZAR MENSAGEM =====
  function adicionarMensagem(novaMensagem, conversa = null) {
    try {
      if (!novaMensagem?.id) {
        console.warn('⚠️ Tentativa de adicionar mensagem sem ID:', novaMensagem)
        return false
      }

      const novaId = String(novaMensagem.id)
      const existeIndex = mensagens.value.findIndex(msg =>
        String(msg.id) === novaId || (msg._isTemporary && msg.content === novaMensagem.content)
      )

      if (existeIndex !== -1 && !novaMensagem._isTemporary) {
        // Atualizar mensagem existente
        const mensagemProcessada = processarMensagem(novaMensagem, conversa)
        const newMensagens = [...mensagens.value]
        newMensagens[existeIndex] = mensagemProcessada
        mensagens.value = newMensagens
        destacarMensagem(mensagemProcessada.id)
        return true
      }

      if (!novaMensagem._isTemporary && processedMessageIds.has(novaId)) {
        return false // Mensagem já processada
      }

      if (!novaMensagem._isTemporary) {
        processedMessageIds.add(novaId)
      }

      const mensagemProcessada = processarMensagem(novaMensagem, conversa)

      // ✅ ADICIONAR E REORDENAR PARA GARANTIR ORDEM CORRETA
      const todasMensagens = [...mensagens.value, mensagemProcessada]
      mensagens.value = sortMessagesByDate(todasMensagens)

      if (!mensagemProcessada._isTemporary) {
        destacarMensagem(mensagemProcessada.id)
      }

      return true
    } catch (error) {
      console.error('❌ Erro ao adicionar mensagem:', error)
      return false
    }
  }
  
  function atualizarStatusMensagem(messageId, novoStatus, metadata = {}) {
    try {
      const messageIndex = mensagens.value.findIndex(msg =>
        String(msg.id) === String(messageId) ||
        msg.whatsapp_message_id === metadata.whatsapp_message_id
      )

      if (messageIndex !== -1) {
        const updatedMessages = [...mensagens.value]
        const currentMessage = updatedMessages[messageIndex]

        const statusNormalizado = normalizarStatusMensagem(novoStatus, currentMessage.delivery_status)

        updatedMessages[messageIndex] = {
          ...currentMessage,
          status: statusNormalizado,
          delivery_status: statusNormalizado,
          delivered_at: statusNormalizado === 'delivered'
            ? metadata.updated_at || currentMessage.delivered_at
            : currentMessage.delivered_at,
          read_at: statusNormalizado === 'read'
            ? metadata.updated_at || currentMessage.read_at
            : currentMessage.read_at
        }

        mensagens.value = updatedMessages
        destacarMensagem(String(messageId))
        return true
      }

      return false
    } catch (error) {
      console.error('❌ Erro ao atualizar status da mensagem:', error)
      return false
    }
  }
  
  function destacarMensagem(messageId) {
    mensagemDestacada.value = messageId
    if (highlightTimeout) clearTimeout(highlightTimeout)
    highlightTimeout = setTimeout(() => {
      mensagemDestacada.value = null
    }, 3000)
  }
  
  // ===== AGRUPAMENTO =====
  function isFirstInGroup(index) {
    if (index === 0) return true
    const current = mensagens.value[index]
    const previous = mensagens.value[index - 1]
    if (!current || !previous) return true

    if (current.sender_id !== previous.sender_id || current.is_from_contact !== previous.is_from_contact) {
      return true
    }

    const timeDiff = new Date(current.created_at) - new Date(previous.created_at)
    return timeDiff > 5 * 60 * 1000 // 5 minutos
  }

  function isLastInGroup(index) {
    if (index === mensagens.value.length - 1) return true
    const current = mensagens.value[index]
    const next = mensagens.value[index + 1]
    if (!current || !next) return true

    if (current.sender_id !== next.sender_id || current.is_from_contact !== next.is_from_contact) {
      return true
    }

    const timeDiff = new Date(next.created_at) - new Date(current.created_at)
    return timeDiff > 5 * 60 * 1000 // 5 minutos
  }

  function isFirstOfDay(index) {
    if (index === 0) return true
    const current = mensagens.value[index]
    const previous = mensagens.value[index - 1]
    if (!current || !previous) return true
    
    const currentDate = new Date(current.created_at).toDateString()
    const previousDate = new Date(previous.created_at).toDateString()
    return currentDate !== previousDate
  }
  
  // ===== LIMPEZA =====
  function limparMensagens() {
    mensagens.value = []
    processedMessageIds.clear()
    messageCache.clear()
    statusCache.clear()
    paginaAtual.value = 1
    temMaisMensagens.value = false
    mensagemDestacada.value = null
    
    if (highlightTimeout) {
      clearTimeout(highlightTimeout)
      highlightTimeout = null
    }
  }
  
  function clearCache() {
    messageCache.clear()
    statusCache.clear()
  }
  
  // ===== CLEANUP =====
  function cleanup() {
    limparMensagens()
    if (highlightTimeout) {
      clearTimeout(highlightTimeout)
      highlightTimeout = null
    }
  }
  
  // ===== RETURN =====
  return {
    // Estado
    mensagens,
    carregandoMensagens,
    carregandoMaisMensagens,
    enviandoMensagem,
    temMaisMensagens,
    mensagemDestacada,
    
    // Computed
    currentUserId,
    
    // Métodos principais
    carregarMensagens,
    enviarMensagem,
    adicionarMensagem,
    atualizarStatusMensagem,
    
    // ✅ ADICIONAR FUNÇÃO DE ORDENAÇÃO
    sortMessagesByDate,
    
    // Helpers
    isCurrentUserMessage,
    getSenderDisplayName,
    processarMensagem,
    normalizarStatusMensagem,
    
    // Agrupamento
    isFirstInGroup,
    isLastInGroup,
    isFirstOfDay,
    
    // Limpeza
    limparMensagens,
    clearCache,
    cleanup
  }
}