// modules/Chat/resources/js/composables/useWebSocketManager.js - VERSÃO CORRIGIDA
import { ref, onUnmounted, onMounted } from 'vue'
import { useApp } from '@/Core/composables/useApp'
import websocketService from '@Chat/servicos/websocket.js'
import chatStore from '@Chat/servicos/chatStore.js'
import { useChatNotifications } from './useChatNotifications.js'

export function useWebSocketManager() {
  const { currentUser } = useApp()
  const chatNotifications = useChatNotifications()
  
  const isInitialized = ref(false)
  const statusInterval = ref(null)
  const eventListeners = ref([])
  const cleanupTimers = ref([])

  // ✅ NOVO: Estado reativo para o status do WebSocket que será exposto
  const websocketStatus = ref({
    isConnected: false,
    connectionState: 'disconnected',
    reconnectAttempts: 0,
    lastError: null,
    socketId: null,
    deduplication: { blocked: 0, allowed: 0 } // Incluir estatísticas de deduplicação
  })

  // ✅ NOVO: Função para atualizar o ref reativo do status a partir do serviço
  const updateWebsocketStatusRef = () => {
    try {
      if (!websocketService) return // Garantir que o serviço exista
      const statusFromService = websocketService.getStatus()
      websocketStatus.value = {
        isConnected: statusFromService.isConnected || false,
        connectionState: statusFromService.connectionState || 'disconnected',
        reconnectAttempts: statusFromService.reconnectAttempts || 0,
        lastError: statusFromService.lastError || null,
        socketId: statusFromService.socketId || null,
        // Garante que a estrutura de deduplicação exista, mesmo que vazia
        deduplication: statusFromService.deduplication || { blocked: 0, allowed: 0, blockingRate: 0 }
      }
    } catch (error) {
      console.warn('Erro ao atualizar status do WebSocket (ref):', error)
      // Fallback para estado desconectado em caso de erro
      websocketStatus.value = { 
        isConnected: false, 
        connectionState: 'disconnected', 
        reconnectAttempts: 0, 
        lastError: error.message || 'Erro desconhecido',
        socketId: null,
        deduplication: { blocked: 0, allowed: 0, blockingRate: 0 }
      }
    }
  }

  // Iniciar WebSocket
  const iniciarWebSocket = async () => {
    if (isInitialized.value) return true

    try {
      // Inicia a atualização do status antes de conectar para ter um estado inicial
      updateWebsocketStatusRef() 
      
      if (!websocketService.isConnected) {
        await websocketService.init()
      }
      
      // Identificar usuário
      if (currentUser.value) {
        websocketService.identifyUser(currentUser.value)
      }

      // Configurar listeners de eventos
      configurarEventListeners()
      
      // Iniciar monitoramento de status
      iniciarMonitoramentoStatus()
      
      isInitialized.value = true
      updateWebsocketStatusRef() // Atualiza o status após a inicialização bem-sucedida
      return true
    } catch (error) {
      console.error('Erro ao iniciar WebSocket:', error)
      chatNotifications.showErrorNotification('Erro', 'Falha na conexão WebSocket')
      updateWebsocketStatusRef() // Atualiza o status em caso de erro de inicialização
      return false
    }
  }

  // Parar WebSocket
  const pararWebSocket = () => {
    try {
      // Sair da conversa atual se houver
      if (chatStore.conversaSelecionada.value?.id && websocketService.isConnected) {
        websocketService.leaveConversation(chatStore.conversaSelecionada.value.id)
      }

      // Parar monitoramento
      pararMonitoramentoStatus()
      
      // Remover listeners
      removerEventListeners()
      
      // Limpar timers
      limparTimers()
      
      isInitialized.value = false
      websocketService.destroy() // Garante que o serviço interno seja destruído
      updateWebsocketStatusRef() // Atualiza o status para desconectado
    } catch (error) {
      console.error('Erro ao parar WebSocket:', error)
    }
  }

  // Reconectar WebSocket
  const reconectarWebSocket = async () => {
    try {
      if (!websocketService) return false
      
      const sucesso = await websocketService.restart()
      
      if (sucesso) {
        // Reentrar na conversa atual se houver
        if (chatStore.conversaSelecionada.value?.id) {
          await websocketService.joinConversation(
            chatStore.conversaSelecionada.value.id,
            chatStore.conversaSelecionada.value.type
          )
        }
      }
      
      updateWebsocketStatusRef() // Atualiza o status após reconexão (sucesso ou falha)
      return sucesso
    } catch (error) {
      console.error('Erro ao reconectar WebSocket:', error)
      updateWebsocketStatusRef() // Atualiza o status em caso de erro de reconexão
      return false
    }
  }

  // Event Listeners
  const configurarEventListeners = () => {
    const listeners = [
      ['chat-message-received', handleMensagemRecebida],
      ['chat-user-typing', handleUsuarioDigitando],
      ['chat-conversation-updated', handleConversaAtualizada],
      ['chat-conversation-read', handleConversaLida],
      ['websocket-connected', handleWebSocketConnected],
      ['websocket-disconnected', handleWebSocketDisconnected],
      ['websocket-error', handleWebSocketError], // Este evento precisa ser emitido pelo websocketService se ocorrerem erros
    ]

    listeners.forEach(([event, handler]) => {
      window.addEventListener(event, handler)
      eventListeners.value.push({ event, handler })
    })
  }

  const removerEventListeners = () => {
    eventListeners.value.forEach(({ event, handler }) => {
      try {
        window.removeEventListener(event, handler)
      } catch (error) {
        console.warn('Erro ao remover listener:', error)
      }
    })
    eventListeners.value = []
  }

  // Event Handlers
  const handleMensagemRecebida = (event) => {
    try {
      const { conversation_id, message } = event.detail
      if (!conversation_id || !message) return

      const currentUserId = currentUser.value?.id
      const isOwnMessage = message.sender_id && currentUserId && 
                          parseInt(message.sender_id) === parseInt(currentUserId)

      // Atualizar ChatStore
      chatStore.updateConversationList(conversation_id, message, isOwnMessage)

      // Notificar se não for própria mensagem e não for conversa atual
      if (!isOwnMessage && chatStore.conversaSelecionada.value?.id !== parseInt(conversation_id)) {
        const conversa = chatStore.getConversaById(conversation_id) || {
          id: conversation_id,
          contact_name: message.sender_name || 'Nova mensagem',
          type: 'external'
        }
        
        chatNotifications.showNewMessageNotification(conversa, message)
      }
      updateWebsocketStatusRef() // Atualiza o status caso as estatísticas de deduplicação mudem
    } catch (error) {
      console.error('Erro ao processar mensagem recebida:', error)
    }
  }

  const handleUsuarioDigitando = (event) => {
    try {
      const { conversation_id, user_id, user_name, is_typing } = event.detail
      chatStore.setUsuarioDigitando(conversation_id, user_id, user_name, is_typing)
    } catch (error) {
      console.error('Erro ao processar usuário digitando:', error)
    }
  }

  const handleConversaAtualizada = (event) => {
    try {
      const { conversation } = event.detail
      if (conversation) {
        chatStore.updateConversa(conversation)
      }
      updateWebsocketStatusRef() // Atualiza o status caso as estatísticas de deduplicação mudem
    } catch (error) {
      console.error('Erro ao processar conversa atualizada:', error)
    }
  }

  const handleConversaLida = (event) => {
    try {
      const { conversation_id } = event.detail
      chatStore.markConversaAsRead(conversation_id)
      updateWebsocketStatusRef() // Atualiza o status caso as estatísticas de deduplicação mudem
    } catch (error) {
      console.error('Erro ao processar conversa lida:', error)
    }
  }

  const handleWebSocketConnected = () => {
    chatNotifications.showConnectionNotification('connected')
    updateWebsocketStatusRef() // Atualiza o ref reativo
  }

  const handleWebSocketDisconnected = () => {
    chatNotifications.showConnectionNotification('disconnected')
    updateWebsocketStatusRef() // Atualiza o ref reativo
  }

  const handleWebSocketError = (event) => {
    const error = event.detail?.error || 'Erro desconhecido'
    console.error('WebSocket Error:', error)
    updateWebsocketStatusRef() // Atualiza o ref reativo
  }

  // Monitoramento de Status
  const iniciarMonitoramentoStatus = () => {
    if (statusInterval.value) return

    const timer = setInterval(() => {
      try {
        updateWebsocketStatusRef() // Atualiza o status periodicamente (bom para status 'connecting' ou erros)
        // Health check básico
        if (!websocketService?.isConnected && isInitialized.value && websocketService.getStatus().connectionState !== 'connecting') {
          console.warn('WebSocketManager: Conexão perdida, tentando reiniciar...')
          websocketService?.restart()?.catch(error => {
            console.error('Restart automático falhou:', error)
            updateWebsocketStatusRef() // Atualiza o status após falha de restart
          })
        }
      } catch (error) {
        console.error('Erro no monitoramento de status:', error)
        updateWebsocketStatusRef() // Atualiza o status em caso de erro no monitoramento
      }
    }, 5000) // 5 segundos, mais frequente para feedback rápido

    statusInterval.value = timer
    adicionarTimer(timer)
  }

  const pararMonitoramentoStatus = () => {
    if (statusInterval.value) {
      clearInterval(statusInterval.value)
      statusInterval.value = null
    }
  }

  // Gerenciamento de Timers
  const adicionarTimer = (timerId) => {
    if (timerId) cleanupTimers.value.push(timerId)
    return timerId
  }

  const limparTimers = () => {
    cleanupTimers.value.forEach(timerId => {
      try {
        clearTimeout(timerId)
        clearInterval(timerId)
      } catch (error) {
        console.warn('Erro ao limpar timer:', error)
      }
    })
    cleanupTimers.value = []
  }

  // Health Check
  const performHealthCheck = () => {
    try {
      const wsOk = websocketService?.isConnected || false
      const storeOk = !!chatStore
      const initOk = isInitialized.value
      return wsOk && storeOk && initOk
    } catch (error) {
      console.error('Erro no health check:', error)
      return false
    }
  }

  // Cleanup automático
  onUnmounted(() => {
    pararWebSocket()
  })

  // ✅ Exposição do status reativo
  return {
    isInitialized,
    iniciarWebSocket,
    pararWebSocket,
    reconectarWebSocket,
    performHealthCheck,
    adicionarTimer,
    limparTimers,
    websocketStatus // ✅ AGORA EXPOSTO COMO REF REATIVO
  }
}