// modules/Chat/resources/js/global-chat.js - VERSÃO FINAL

import { markRaw } from 'vue'

class GlobalChatManager {
  constructor() {
    this.initialized = false
    this.chatNotifications = null
    this.services = { websocket: null, chatStore: null }
    this.config = this.getDefaultConfig()
    this.state = this.getInitialState()
    
    // ✅ DEDUPLICAÇÃO DE MENSAGENS PARA NOTIFICAÇÃO
    this.processedNotificationMessages = new Map() // message.id_timestamp -> timestamp
    this.messageNotificationTTL = 300000 // 5 minutos
    this.cleanupNotificationInterval = null
  }

  getDefaultConfig() {
    return {
      debug: false,
      enabled: true,
      websocket_url: null,
      notifications: {
        enabled: true,
        sound: true,
        desktop: true,
        maxVisible: 3,
        duration: 5000,
        throttleTime: 3000,
      },
    }
  }

  getInitialState() {
    return {
      lastNotification: { type: null, time: 0 },
      eventListeners: [],
      timers: [],
      healthCheckTimer: null,
    }
  }

  log(level, ...args) {
    if (!this.config.debug && level !== 'error' && level !== 'warn') return
    const prefix = `[Chat Global] - ${level.toUpperCase()}:`
    console[level] ? console[level](prefix, ...args) : console.log(prefix, ...args)
  }

  async init() {
    if (this.initialized) return true

    try {
      this.loadConfig()
      if (!this.config.enabled) {
        this.log('warn', 'Chat global desabilitado em configuração')
        return false
      }

      await this.requestNotificationPermission()
      await this.loadServices()

      const connected = await this.services.websocket.init({
        url: this.config.websocket_url,
        debug: this.config.debug,
      })

      if (!connected) {
        throw new Error('Falha ao conectar WebSocket')
      }

      this.setupEventListeners()
      await this.loadInitialData()
      this.integrateWithNotifications()
      this.startHealthMonitoring()
      this.startNotificationCleanupInterval() // ✅ INICIAR LIMPEZA DE CACHE DE NOTIFICAÇÕES

      this.initialized = true
      window.chatGlobalInitialized = true

      this.log('info', '✅ Chat Global inicializado com sucesso')
      return true
    } catch (error) {
      this.log('error', 'Falha na inicialização do Chat Global:', error)
      this.cleanup()
      return false
    }
  }

  loadConfig() {
    const chatConfig = window.Innoclapps?.scriptConfig?.('chat')
    if (!chatConfig) {
      this.log('warn', 'Nenhuma configuração de Chat encontrada')
      return
    }

    this.config = {
      ...this.config,
      debug: !!chatConfig.debug,
      enabled: chatConfig.enabled !== false,
      websocket_url: chatConfig.websocket_url,
      notifications: {
        ...this.config.notifications,
        enabled: chatConfig.notifications_enabled !== false,
        sound: chatConfig.sound_enabled !== false,
      }
    }
  }

  async loadServices() {
    const [{ default: websocketService }, { default: chatStore }] = await Promise.all([
      import('./servicos/websocket.js'),
      import('./servicos/chatStore.js'),
    ])

    this.services.websocket = websocketService
    this.services.chatStore = chatStore

    // Fallback para loadInitialConversations
    if (!chatStore.loadInitialConversations) {
      chatStore.loadInitialConversations = async () => {
        const { default: servicoApi } = await import('./servicos/api.js')
        const response = await servicoApi.getConversations({ include_archived: false })
        const conversas = Array.isArray(response) ? response : (response?.data || [])
        chatStore.setConversas(conversas)
      }
    }

    // Carregar sistema de notificações
    try {
      const { useChatNotifications } = await import('./composables/useChatNotifications.js')
      this.chatNotifications = useChatNotifications()
      this.chatNotifications.initializeChatNotifications()
    } catch (error) {
      this.log('warn', 'Sistema de notificações não disponível:', error)
    }

    // Expor globalmente
    window.websocketService = websocketService
    window.chatStore = chatStore
  }

  async loadInitialData() {
    try {
      if (window.Innoclapps?.request) {
        const response = await window.Innoclapps.request('/chat/conversations/summary')
        const unread = response?.data?.unread_count || 0
        this.updateNotificationBadge(unread)
      }
    } catch (error) {
      this.log('warn', 'Falha ao carregar dados iniciais:', error)
    }
  }

  setupEventListeners() {
    const listeners = [
      ['chat-message-received', this.handleNewMessage.bind(this)],
      ['chat-store-new-unread-message', this.handleNewMessage.bind(this)], // ✅ Manter para compatibilidade/eventos internos
      ['chat-store-updated', this.updateNotificationBadge.bind(this)],
      ['chat-outbound-message-received', this.handleOutboundMessage.bind(this)],
      ['websocket-connected', () => this.chatNotifications?.showConnectionNotification('connected')],
      ['websocket-disconnected', () => this.chatNotifications?.showConnectionNotification('disconnected')],
      ['beforeunload', this.cleanup.bind(this)],
      ['pagehide', this.cleanup.bind(this)],
    ]

    listeners.forEach(([event, handler]) => {
      window.addEventListener(event, handler)
      this.state.eventListeners.push({ element: window, event, handler })
    })
  }

  async requestNotificationPermission() {
    if (!('Notification' in window)) return

    if (Notification.permission === 'default') {
      try {
        await Notification.requestPermission()
      } catch (error) {
        this.log('warn', 'Falha ao solicitar permissão de notificação:', error)
      }
    }
  }

    handleNewMessage(event) {
    const detail = event?.detail || event || {}
    let conversation = detail.conversation || null
    let message = detail.message || null
    let conversationId = detail.conversation_id || detail.conversationId || 
                        (message && (message.conversation_id || message.conversationId))

    // Validações básicas
    if (!conversation && conversationId && this.services.chatStore?.conversas) {
      const convIdNum = parseInt(conversationId)
      conversation = this.services.chatStore.conversas.value?.find(c => parseInt(c.id) === convIdNum) || 
                    { id: convIdNum }
    }

    if (!message && detail.data?.message) {
      message = detail.data.message
    } else if (!message && detail.content) {
      message = detail
    }

    if (!conversation || !message || !message.id) {
      this.log('warn', 'Dados insuficientes para notificação de nova mensagem', detail)
      return
    }

    const convId = parseInt(conversation.id || conversationId)
    if (Number.isNaN(convId)) {
        this.log('warn', 'Conversation ID inválido para notificação de nova mensagem', detail)
        return
    }

    // --- CAMADA FINAL DE DEDUPLICAÇÃO NO GLOBAL CHAT MANAGER ---
    const messageUniqueId = `${message.id}_${message.created_at || message.sent_at || message.timestamp || ''}`
    const now = Date.now()
    
    if (this.processedNotificationMessages.has(messageUniqueId)) {
      this.log('debug', `🚫 Notificação de mensagem já processada pelo GlobalChatManager (ID: ${messageUniqueId})`)
      return; // Bloqueia a notificação duplicada
    }
    this.processedNotificationMessages.set(messageUniqueId, now);
    // --- FIM DA DEDUPLICAÇÃO ---

    // ✅ INÍCIO DA CORREÇÃO: VERIFICAR MENSAGENS PRÓPRIAS ROBUSTAMENTE
    try {
        const currentUserId = this.services.websocket?.getCurrentUser?.()?.id;
        const messageIsFromMe = message.is_from_me === true || message.whatsapp_key_from_me === true; // ✅ Checa as flags do webhook

        // Se a mensagem é marcada como "minha" pelo webhook, ignorar notificação
        if (messageIsFromMe) {
            this.log('debug', `🚫 Mensagem enviada por mim (${message.id}) ignorada para notificação (is_from_me: ${messageIsFromMe})`);
            return;
        }

        // Caso a flag acima seja falsa, mas o sender_id corresponda ao usuário do CRM, também ignorar
        // Isso cobre mensagens internas ou cenários específicos onde is_from_me não é setado, mas o sender_id é você
        const senderId = message.sender_id || message.sender?.id;
        if (currentUserId && senderId && parseInt(currentUserId) === parseInt(senderId)) {
            this.log('debug', `🚫 Mensagem própria (${message.id}) ignorada para notificação (sender_id: ${senderId} == currentUserId: ${currentUserId})`);
            return;
        }

    } catch (e) {
      this.log('warn', 'Erro ao comparar usuário em handleNewMessage:', e)
    }
    // ✅ FIM DA CORREÇÃO

    // Não notificar se já na conversa ativa
    if (this.isOnActiveConversation(convId)) {
      this.log('debug', `🚫 Mensagem (${message.id}) em conversa ativa, ignorando notificação`)
      return
    }

    // Mostrar notificação
    if (this.chatNotifications && this.shouldEmitNotification({ conversation, message })) {
      this.chatNotifications.showNewMessageNotification(conversation, message)
    }

    this.updateNotificationBadge()
  }


  handleOutboundMessage(event) {
    // Apenas atualizar badge para mensagens externas
    this.updateNotificationBadge()
  }

  shouldEmitNotification(data) {
    if (!this.config.notifications.enabled) return false

    const currentUserId = this.services.websocket?.getCurrentUser?.()?.id
    const message = data.message || data

    // Não notificar mensagens próprias
    if (message?.sender_id && currentUserId && message.sender_id === currentUserId) {
      return false
    }

    const conversationId = data.conversation_id || message?.conversation_id
    const conversation = this.services.chatStore?.conversas?.value?.find(
      c => c.id === parseInt(conversationId)
    )

    if (!conversation || conversation.is_muted) return false

    // Notificar se documento oculto ou não é conversa atual
    return document.hidden || 
           this.services.chatStore?.conversaSelecionada?.value?.id !== parseInt(conversationId)
  }

  updateNotificationBadge(countOrEvent = null) {
    let count = 0

    if (typeof countOrEvent === 'number') {
      count = countOrEvent
    } else if (countOrEvent?.detail?.unreadTotal != null) {
      count = countOrEvent.detail.unreadTotal
    } else if (this.services.chatStore?.unreadTotal?.value != null) {
      count = this.services.chatStore.unreadTotal.value
    }

    // Atualizar badges
    const badges = document.querySelectorAll('.chat-notification-badge')
    badges.forEach(badge => {
      badge.textContent = count > 99 ? '99+' : String(count)
      badge.classList.toggle('show', count > 0)
    })

    // Atualizar título
    const originalTitle = document.title.replace(/^\((\d+|\d+\+)\)\s*/, '') // Regex mais robusta para (99+)
    document.title = count > 0 ? `(${count}) ${originalTitle}` : originalTitle

    // App badge (PWA)
    if ('setAppBadge' in navigator) {
      if (count > 0) {
        navigator.setAppBadge(count).catch(() => {})
      } else {
        navigator.clearAppBadge().catch(() => {})
      }
    }
  }

  integrateWithNotifications() {
    const selectors = [
      '[data-notifications-button]',
      '.notifications-button',
      '[data-testid="notifications"]',
      '.notification-bell',
      '.nav-notifications',
    ]

    for (const selector of selectors) {
      const button = document.querySelector(selector)
      if (button && !button.querySelector('.chat-notification-badge')) {
        const badge = document.createElement('span')
        badge.className = 'chat-notification-badge'

        if (!['relative', 'absolute', 'fixed', 'sticky'].includes(getComputedStyle(button).position)) { // Usar getComputedStyle
          button.style.position = 'relative'
        }

        button.appendChild(badge)
        break
      }
    }
  }

  startHealthMonitoring() {
    if (this.state.healthCheckTimer) return

    const timer = setInterval(() => {
      const ok = this.performHealthCheck()
      if (!ok && this.services.websocket && !this.services.websocket.isConnected) {
        this.services.websocket.restart().catch(err => 
          this.log('error', 'Restart WebSocket falhou:', err)
        )
      }
      this.updateNotificationBadge()
    }, 30000)

    this.state.healthCheckTimer = timer
    this.addTimer(timer)
  }

  // ✅ NOVO MÉTODO: Limpar cache de deduplicação de notificações
  startNotificationCleanupInterval() {
    if (this.cleanupNotificationInterval) return;

    this.cleanupNotificationInterval = setInterval(() => {
      const now = Date.now();
      let cleanedCount = 0;
      for (const [key, timestamp] of this.processedNotificationMessages.entries()) {
        if (now - timestamp > this.messageNotificationTTL) {
          this.processedNotificationMessages.delete(key);
          cleanedCount++;
        }
      }
      if (cleanedCount > 0) {
          this.log('debug', `🧹 Limpeza de cache de notificações: ${cleanedCount} entradas removidas.`);
      }
    }, 60000); // Limpa a cada minuto
    this.addTimer(this.cleanupNotificationInterval); // Adicionar ao gerenciamento de timers
  }

  performHealthCheck() {
    return !!(this.services.websocket?.isConnected && this.services.chatStore && this.initialized)
  }

  isOnActiveConversation(conversationId) {
    const isOnChatPage = window.location.pathname.includes('/chat')
    const selected = this.services.chatStore?.conversaSelecionada?.value
    return isOnChatPage && selected && parseInt(selected.id) === parseInt(conversationId)
  }

  addTimer(timerId) {
    if (timerId) this.state.timers.push(timerId)
    return timerId
  }

  cleanup() {
    if (!this.initialized) return

    // Remover listeners
    this.state.eventListeners.forEach(({ element, event, handler }) => {
      try {
        element.removeEventListener(event, handler)
      } catch (e) {
        this.log('warn', 'Erro ao remover listener:', e)
      }
    })
    this.state.eventListeners = []

    // Limpar timers
    this.state.timers.forEach(timerId => {
      try {
        clearTimeout(timerId)
        clearInterval(timerId)
      } catch (e) {
        this.log('warn', 'Erro ao limpar timer:', e)
      }
    })
    this.state.timers = []
    this.state.healthCheckTimer = null
    this.cleanupNotificationInterval = null; // ✅ Limpar também este intervalo

    // Limpar cache de deduplicação de notificações
    this.processedNotificationMessages.clear();


    // Destruir WebSocket
    if (this.services.websocket?.destroy) {
      try {
        this.services.websocket.destroy()
      } catch (e) {
        this.log('warn', 'Erro ao destruir WebSocket:', e)
      }
    }

    this.initialized = false
    window.chatGlobalInitialized = false
  }

  getStatusSnapshot() {
    return {
      initialized: this.initialized,
      config: this.config,
      websocket: this.services.websocket?.getStatus?.() ?? null,
      chatStore: this.services.chatStore?.getStats?.() ?? null,
      globalChatManager: {
        processedNotificationMessages: this.processedNotificationMessages.size,
        messageNotificationTTL: this.messageNotificationTTL
      }
    }
  }
}

const globalChatManager = new GlobalChatManager()

// API global
window.chatRuntime = {
  init: () => globalChatManager.init(),
  cleanup: () => globalChatManager.cleanup(),
  status: () => globalChatManager.getStatusSnapshot(),
  debug: () => console.log('=== CHAT RUNTIME STATUS ===', globalChatManager.getStatusSnapshot()),
  // ✅ ADICIONAR FERRAMENTAS DE DEBUG DE DEDUPLICAÇÃO
  clearGlobalChatDeduplicationCache: () => {
    globalChatManager.processedNotificationMessages.clear();
    console.log('🧹 Cache de deduplicação do GlobalChatManager limpo.');
  }
}

// Helper global para notificações
window.showGlobalToast = function (message, type = 'info', duration = 4000) {
  if (globalChatManager.chatNotifications) {
    globalChatManager.chatNotifications.showChatNotification({
      type, message, duration
    })
  }
}

// Compatibilidade
window.cleanupGlobalChat = () => globalChatManager.cleanup()

// Boot automático
if (typeof window !== 'undefined') {
  const initWhenReady = () => {
    if (!globalChatManager.initialized) {
      globalChatManager.init()
    }
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      setTimeout(initWhenReady, 500)
    })
  } else {
    setTimeout(initWhenReady, 500)
  }
}

export default globalChatManager