/**
 * SERVIÇO DE API PARA CHAT - VERSÃO COMPLETA E OTIMIZADA
 * Compatível com Concord CRM (Innoclapps) - fallback para fetch
 * Foco em performance, robustez, clareza e manutenibilidade.
 */

let API_CONFIG = {
  baseURL: '/api',
  timeout: 30000,
  retries: 3,
  retryDelay: 1000,
  cacheTTL: 300000,
  debug: false
}

const cache = new Map()

function debug(...args) {
  if (API_CONFIG.debug) console.log('🌐 API:', ...args)
}

function normalizeParams(params = {}) {
  const normalized = {}
  if (!params || typeof params !== 'object') return normalized
  Object.keys(params).sort().forEach(k => {
    const v = params[k]
    if (v !== null && v !== undefined && v !== '') {
      normalized[k] = String(v).trim()
    }
  })
  return normalized
}

function getCacheKey(method, endpoint, params = {}) {
  const normalized = normalizeParams(params)
  const paramsString = Object.keys(normalized).length ? JSON.stringify(normalized) : ''
  return `${method.toLowerCase()}:${endpoint.toLowerCase()}:${paramsString}`
}

function isValidCache(timestamp) {
  return Date.now() - timestamp < API_CONFIG.cacheTTL
}

function getFromCache(key) {
  const item = cache.get(key)
  if (!item) return null
  if (isValidCache(item.timestamp)) {
    debug('📦 Cache hit:', key)
    return item.data
  }
  cache.delete(key)
  debug('🗑️ Cache expirado removido:', key)
  return null
}

function setCache(key, data) {
  cache.set(key, { data, timestamp: Date.now() })
  debug('💾 Dados cacheados:', key)
}

function clearCache(pattern = null) {
  if (!pattern) {
    cache.clear()
    debug('🧹 Cache totalmente limpo')
    return
  }
  const lowerPattern = pattern.toLowerCase()
  const keysToDelete = []
  for (const k of cache.keys()) {
    if (k.includes(lowerPattern)) {
      keysToDelete.push(k)
    }
  }
  keysToDelete.forEach(k => cache.delete(k))
  debug(`🧹 Cache limpo para padrão "${pattern}": ${keysToDelete.length} entradas removidas`)
}

function getCSRFToken() {
  try {
    if (window.Innoclapps?.scriptConfig) {
      const token = window.Innoclapps.scriptConfig('chat')?.csrf_token
      if (token) return token
    }
    if (window.chatConfig?.csrf_token) return window.chatConfig.csrf_token
    const meta = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content')
    if (meta) return meta
  } catch (e) {
    debug('❌ Erro ao obter CSRF token:', e)
  }
  return ''
}

function getDefaultHeaders() {
  const headers = {
    'Accept': 'application/json',
    'X-Requested-With': 'XMLHttpRequest'
  }
  const token = getCSRFToken()
  if (token) headers['X-CSRF-TOKEN'] = token
  return headers
}

function buildUrl(endpoint, params = {}) {
  try {
    let base = API_CONFIG.baseURL
    base = base.endsWith('/') ? base.slice(0, -1) : base
    let path = endpoint.startsWith('/') ? endpoint : `/${endpoint}`
    let fullPath = base + path
    const url = new URL(fullPath, window.location.origin)

    if (params && typeof params === 'object') {
      Object.keys(params).forEach(key => {
        const v = params[key]
        if (v !== null && v !== undefined && v !== '') url.searchParams.append(key, String(v))
      })
    }
    return url.toString()
  } catch (e) {
    debug('❌ Erro ao construir URL:', e, { endpoint, params, baseURL: API_CONFIG.baseURL })
    return `${API_CONFIG.baseURL}${endpoint.startsWith('/') ? '' : '/'}${endpoint}`
  }
}

function getHttpClient() {
  if (window.Innoclapps?.request) {
    return { type: 'axios', instance: window.Innoclapps.request() }
  }
  return { type: 'fetch', instance: null }
}

async function makeRequest(method, endpoint, payload = null, options = {}) {
  const start = Date.now()
  let lastError = null
  const client = getHttpClient()

  const queryParams = (method === 'GET' && payload && typeof payload === 'object') ? payload : {}
  const requestBody = (method !== 'GET') ? payload : undefined
  const requestUrl = buildUrl(endpoint, queryParams)

  for (let attempt = 1; attempt <= API_CONFIG.retries; attempt++) {
    try {
      debug(`📤 ${method} ${requestUrl} (tentativa ${attempt})`)

      if (method === 'GET' && options.cache !== false) {
        const ck = getCacheKey(method, endpoint, queryParams)
        const cached = getFromCache(ck)
        if (cached) return cached
      }

      if (client.type === 'axios') {
        const axiosOpts = {
          method,
          url: requestUrl,
          headers: { ...getDefaultHeaders(), ...(options.headers || {}) },
          timeout: API_CONFIG.timeout,
          data: requestBody,
          responseType: options.responseType || 'json'
        }
        const resp = await client.instance.request(axiosOpts)
        const duration = Date.now() - start
        debug(`📥 Axios ${requestUrl} (${resp.status}) - ${duration}ms`)

        if (resp.status < 200 || resp.status >= 300) {
          const err = new Error(resp.data?.message || `HTTP ${resp.status}`)
          err.status = resp.status
          err.data = resp.data
          throw err
        }

        const respData = resp.data
        if (method === 'GET' && options.cache !== false) {
          setCache(getCacheKey(method, endpoint, queryParams), respData)
        }
        if (['POST', 'PUT', 'PATCH', 'DELETE'].includes(method)) {
          postMutationCleanup(endpoint)
        }
        return respData
      }

      const fetchOpts = {
        method,
        headers: { ...getDefaultHeaders(), ...(options.headers || {}) },
        signal: AbortSignal.timeout(API_CONFIG.timeout)
      }

      if (method !== 'GET' && requestBody) {
        if (requestBody instanceof FormData) {
          fetchOpts.body = requestBody
        } else {
          fetchOpts.headers['Content-Type'] = 'application/json'
          fetchOpts.body = JSON.stringify(requestBody)
        }
      }

      const resp = await fetch(requestUrl, fetchOpts)
      const duration = Date.now() - start
      debug(`📥 Fetch ${requestUrl} (${resp.status}) - ${duration}ms`)

      const contentType = resp.headers.get('content-type') || ''
      let respData = null
      if (contentType.includes('application/json')) {
        try {
          respData = await resp.json()
        } catch (jsonError) {
          respData = await resp.text()
          debug('⚠️ Falha ao parsear JSON, fallback para texto:', respData, jsonError)
        }
      } else {
        respData = await resp.text()
      }

      if (!resp.ok) {
        let errorPayload = respData
        let errorMessage = `HTTP ${resp.status}`

        if (respData && typeof respData === 'object' && (respData.message || respData.error)) {
          errorMessage = respData.message || respData.error
        } else if (typeof respData === 'string' && respData) {
          errorMessage = respData
        }

        if (resp.status === 401) errorMessage = 'Não autorizado - faça login novamente.'
        else if (resp.status === 403) errorMessage = 'Acesso negado.'
        else if (resp.status === 404) errorMessage = 'Recurso não encontrado.'
        else if (resp.status === 422) errorMessage = 'Dados inválidos. Verifique as informações fornecidas.'
        else if (resp.status >= 500) errorMessage = 'Erro interno do servidor. Tente novamente mais tarde.'

        const err = new Error(errorMessage)
        err.status = resp.status
        err.data = errorPayload
        err.response = { status: resp.status, data: errorPayload }
        throw err
      }

      if (method === 'GET' && options.cache !== false) {
        setCache(getCacheKey(method, endpoint, queryParams), respData)
      }
      if (['POST', 'PUT', 'PATCH', 'DELETE'].includes(method)) {
        postMutationCleanup(endpoint)
      }
      return respData

    } catch (error) {
      lastError = error
      const errorStatus = error.status || error.response?.status
      debug(`❌ ${method} ${requestUrl} - tentativa ${attempt} falhou:`, error.message || error, errorStatus ? `(Status: ${errorStatus})` : '')
      
      if (errorStatus >= 400 && errorStatus < 500 && errorStatus !== 408 && errorStatus !== 429 && errorStatus !== 422) {
        throw error
      }
      if (attempt < API_CONFIG.retries) {
        const backoff = API_CONFIG.retryDelay * Math.pow(2, attempt - 1)
        await new Promise(r => setTimeout(r, backoff))
      }
    }
  }

  debug(`❌ ${method} ${requestUrl} - todas as tentativas falharam`)
  throw lastError || new Error('Falha na requisição após todas as tentativas.')
}

function postMutationCleanup(endpoint) {
  try {
    clearCache('/chat/conversations')
    clearCache('/chat/channels')
    const match = endpoint.match(/\/chat\/conversations\/(\d+)/)
    if (match?.[1]) {
      const id = match[1]
      clearCache(`/chat/conversations/${id}`)
      clearCache(`/chat/conversations/${id}/messages`)
    }
  } catch (e) {
    debug('❌ Erro no pós-limpeza de cache:', e)
  }
}

const apiService = {
  // === Métodos HTTP de baixo nível ===
  async get(url, params = null, options = {}) { return makeRequest('GET', url, params, options) },
  async post(url, data = null, options = {}) { return makeRequest('POST', url, data, options) },
  async put(url, data = null, options = {}) { return makeRequest('PUT', url, data, options) },
  async patch(url, data = null, options = {}) { return makeRequest('PATCH', url, data, options) },
  async delete(url, options = {}) { return makeRequest('DELETE', url, null, options) },

  // === Métodos de Configuração ===
  setBaseURL(url) { API_CONFIG.baseURL = url; debug('🔧 Base URL configurada:', url) },
  setTimeout(ms) { if (typeof ms === 'number' && ms > 0) API_CONFIG.timeout = ms },
  setCacheTTL(ms) { if (typeof ms === 'number' && ms > 0) API_CONFIG.cacheTTL = ms },
  setDebug(enabled) { API_CONFIG.debug = !!enabled; debug('🐛 Debug', API_CONFIG.debug) },
  getCacheStats() { return { size: cache.size, keys: Array.from(cache.keys()), config: {...API_CONFIG} } },
  clearCache(pattern = null) { clearCache(pattern) },

  // === MÉTODOS PARA CANAIS ===
  
  /**
   * Obtém lista de canais.
   * @param {object} params Parâmetros de filtro.
   * @returns {Promise<Array<object>>} Lista de canais.
   */
  async getChannels(params = {}) {
    return this.get('/chat/channels', params)
  },

  /**
   * Obtém um canal específico.
   * @param {number|string} channelId ID do canal.
   * @returns {Promise<object>} Dados do canal.
   */
  async getChannel(channelId) {
    return this.get(`/chat/channels/${channelId}`)
  },

  /**
   * Cria um novo canal.
   * @param {object} data Dados do canal.
   * @returns {Promise<object>} Canal criado.
   */
  async createChannel(data) {
    return this.post('/chat/channels', data)
  },

  /**
   * Atualiza um canal.
   * @param {number|string} channelId ID do canal.
   * @param {object} data Dados para atualizar.
   * @returns {Promise<object>} Canal atualizado.
   */
  async updateChannel(channelId, data) {
    return this.put(`/chat/channels/${channelId}`, data)
  },

  /**
   * Exclui um canal.
   * @param {number|string} channelId ID do canal.
   * @param {boolean} force Se deve forçar a exclusão.
   * @returns {Promise<object>} Resposta da API.
   */
  async deleteChannel(channelId, force = false) {
    const endpoint = force 
      ? `/chat/channels/${channelId}/force-delete`
      : `/chat/channels/${channelId}`
    return this.delete(endpoint)
  },

  /**
   * Alterna status de um canal (ativo/inativo).
   * @param {number|string} channelId ID do canal.
   * @returns {Promise<object>} Canal atualizado.
   */
  async toggleChannelStatus(channelId) {
    return this.patch(`/chat/channels/${channelId}/toggle-status`)
  },

  /**
   * Testa conexão de um canal.
   * @param {number|string} channelId ID do canal.
   * @returns {Promise<object>} Resultado do teste.
   */
  async testChannelConnection(channelId) {
    return this.post(`/chat/channels/${channelId}/test-connection`)
  },

  /**
   * Testa conexão Evolution específica.
   * @param {number|string} channelId ID do canal.
   * @returns {Promise<object>} Resultado do teste Evolution.
   */
  async testEvolutionChannel(channelId) {
    return this.post(`/chat/channels/${channelId}/test-evolution`)
  },

  /**
   * Sincroniza canal Evolution.
   * @param {number|string} channelId ID do canal.
   * @returns {Promise<object>} Resultado da sincronização.
   */
  async syncEvolutionChannel(channelId) {
    return this.post(`/chat/channels/${channelId}/sync-evolution`)
  },

  /**
   * Obtém estatísticas de um canal.
   * @param {number|string} channelId ID do canal.
   * @returns {Promise<object>} Estatísticas do canal.
   */
  async getChannelStats(channelId) {
    return this.get(`/chat/channels/${channelId}/stats`)
  },

  /**
   * ✅ OBTER INSTÂNCIAS EVOLUTION (CRM + API)
   */
  async getAvailableEvolutionInstances() {
    try {
      console.log('📋 Buscando instâncias Evolution...')
      
      const response = await this.get('/chat/evolution-instances')
      
      console.log('✅ Resposta da API:', response)
      
      if (!response || typeof response !== 'object') {
        throw new Error('Resposta inválida da API')
      }
      
      return {
        local_instances: response.local_instances || [],
        evolution_instances: response.evolution_instances || [],
        summary: response.summary || {}
      }
      
    } catch (error) {
      console.error('❌ Erro ao buscar instâncias:', error)
      throw error
    }
  },

/**
 * Cria instância local baseada na Evolution API.
 * @param {string} instanceName Nome da instância.
 * @param {object} apiData Dados da API.
 * @returns {Promise<object>} Instância criada.
 */
async createInstanceFromEvolutionApi(instanceName, apiData) {
  return this.post('/chat/evolution-instances/create-from-api', {
    instance_name: instanceName,
    api_data: apiData
  })
},

  // === Métodos Evolution API ===
  async getEvolutionInstances(params = {}) { return this.get('/evolution/instances', params) },
  async getEvolutionInstanceStatus(name) { return this.get(`/evolution/instances/${name}/status`) },
  async getEvolutionQrCode(name) { return this.get(`/evolution/instances/${name}/qrcode`) },
  async connectEvolutionInstance(name) { return this.post(`/evolution/instances/${name}/connect`) },
  async disconnectEvolutionInstance(name) { return this.post(`/evolution/instances/${name}/disconnect`) },
  async createEvolutionInstance(data) { return this.post('/evolution/instances', data) },
  async updateEvolutionInstance(name, data) { return this.put(`/evolution/instances/${name}`, data) },
  async deleteEvolutionInstance(name) { return this.delete(`/evolution/instances/${name}`) },

  // === Métodos para Conversas ===
  async getConversations(params = { include_archived: false }) {
    return this.get('/chat/conversations', params)
  },
  async getConversation(id) {
    return this.get(`/chat/conversations/${id}`)
  },
  async createConversation(payload) { return this.post('/chat/conversations', payload) },
  async createInternalConversation(payload) { 
    console.log('💬 API: Criando conversa interna:', payload)
    
    // ✅ GARANTIR QUE O TIPO SEJA INTERNAL E USAR A ROTA PADRÃO
    const data = { ...payload, type: 'internal' }
    
    // ✅ USAR A ROTA PADRÃO /chat/conversations AO INVÉS DE /internal
    const result = await this.post('/chat/conversations', data)
    
    // Limpar cache relacionado
    this.invalidateConversationsCache()
    
    return result
  },
  async updateConversation(id, payload) { return this.patch(`/chat/conversations/${id}`, payload) },
  async deleteConversation(id) { return this.delete(`/chat/conversations/${id}`) },
  async markAsUnread(conversationId) { 
  return this.post(`/chat/conversations/${conversationId}/mark-unread`) 
},

  // === Métodos para Mensagens ===
  async getMessages(conversationId, params = {}) {
    const query = { page: params.page || 1, per_page: params.per_page || 50 }
    if (params.before_id) query.before_id = params.before_id
    const resp = await this.get(`/chat/conversations/${conversationId}/messages`, query)
    
    if (!resp) return { messages: [], pagination: { current_page: 1, per_page: query.per_page, has_more: false, total: 0 } }
    if (Array.isArray(resp)) {
      return {
        messages: resp,
        pagination: { current_page: query.page, per_page: query.per_page, has_more: resp.length === query.per_page, total: resp.length }
      }
    }
    if (Array.isArray(resp.data)) {
      return {
        messages: resp.data,
        pagination: resp.pagination || { current_page: query.page, per_page: query.per_page, has_more: false, total: resp.data.length }
      }
    }
    return { messages: [], pagination: { current_page: 1, per_page: query.per_page, has_more: false, total: 0 } }
  },

  async sendMessage(conversationId, messageData) {
    let payload = {}
    if (typeof messageData === 'string') {
      payload = { content: messageData.trim(), message_type: 'text', timestamp: new Date().toISOString() }
    } else if (messageData && typeof messageData === 'object') {
      payload = {
        content: (messageData.content || messageData.message || messageData.text || '').trim(),
        message_type: messageData.type || messageData.message_type || 'text',
        timestamp: messageData.timestamp || new Date().toISOString(),
        attachments: messageData.attachments,
        reply_to: messageData.reply_to,
        metadata: messageData.metadata
      }
    }
    if (!payload.content) throw new Error('Conteúdo da mensagem é obrigatório.')
    return this.post(`/chat/conversations/${conversationId}/messages`, payload)
  },

  async sendMessageWithAttachments(conversationId, formData) {
    if (!(formData instanceof FormData)) {
      throw new Error('Dados devem ser FormData para envio de anexos.')
    }
    const endpoint = `/chat/conversations/${conversationId}/messages`
    return makeRequest('POST', endpoint, formData)
  },

  async markAsRead(conversationId) { return this.post(`/chat/conversations/${conversationId}/read`) },
  async sendTypingIndicator(conversationId, isTyping = true) { 
    return this.post(`/chat/conversations/${conversationId}/typing`, { is_typing: isTyping, timestamp: new Date().toISOString() }) 
  },

  // === Métodos para Atribuições ===
async assignConversation(conversationId, data) {
  return this.patch(`/chat/conversations/${conversationId}/assign`, data)
},

async assignMultipleUsers(conversationId, data) { 
  return this.post(`/chat/conversations/${conversationId}/assignments`, data) 
},

// ✅ CORRIGIDO: getConversationAssignments
async getConversationAssignments(conversationId) { 
  try {
    console.log('🔍 Buscando atribuições da conversa:', conversationId)
    
    const response = await this.get(`/chat/conversations/${conversationId}/assignments`)
    
    console.log('📋 Atribuições recebidas:', response)
    
    // ✅ VALIDAR ESTRUTURA DA RESPOSTA
    if (!response) {
      return {
        conversation_id: conversationId,
        assignments: [],
        total_assignments: 0,
        primary_user: null
      }
    }
    
    return {
      conversation_id: response.conversation_id || conversationId,
      assignments: response.assignments || [],
      total_assignments: response.total_assignments || 0,
      primary_user: response.primary_user || null
    }
    
  } catch (error) {
    console.error('❌ Erro ao buscar atribuições:', error)
    // ✅ RETORNAR ESTRUTURA PADRÃO EM CASO DE ERRO
    return {
      conversation_id: conversationId,
      assignments: [],
      total_assignments: 0,
      primary_user: null
    }
  }
},

async removeUserAssignment(conversationId, userId) { 
  return this.delete(`/chat/conversations/${conversationId}/assignments/${userId}`) 
},

async updateUserAssignment(conversationId, userId, data) { 
  return this.patch(`/chat/conversations/${conversationId}/assignments/${userId}`, data) 
},

 /**
 * Finaliza uma conversa com dados avançados
 * @param {number|string} conversationId ID da conversa
 * @param {object} data Dados da finalização
 * @returns {Promise<object>} Conversa finalizada
 */
async finalizeConversation(conversationId, data) {
  console.log('🏁 API: Finalizando conversa:', conversationId, data)
  
  const payload = {
    status: 'closed',
    reason: data.reason,
    resolution: data.resolution || null,
    send_farewell_message: data.sendFarewellMessage || false,
    farewell_message: data.farewellMessage || null,
    archive_conversation: data.archiveConversation || false,
    block_contact: data.blockContact || false,
    notify_team: data.notifyTeam || false,
    request_feedback: data.requestFeedback || false,
    closed_at: data.finalizedAt || new Date().toISOString(),
    finalized_by: window.Innoclapps?.scriptConfig?.('chat')?.user_id || null
  }
  
  const result = await this.put(`/chat/conversations/${conversationId}/finalize`, payload)
  
  this.invalidateConversationsCache()
  this.invalidateMessagesCache(conversationId)
  
  return result
},

/**
 * Reabre uma conversa fechada
 * @param {number|string} conversationId ID da conversa
 * @param {string} reason Motivo da reabertura
 * @returns {Promise<object>} Conversa reaberta
 */
async reopenConversation(conversationId, reason = 'Reabertura solicitada') {
  console.log('�� API: Reabrindo conversa:', conversationId, reason)
  
  const payload = {
    status: 'active',
    reopen_reason: reason,
    reopened_at: new Date().toISOString(),
    reopened_by: window.Innoclapps?.scriptConfig?.('chat')?.user_id || null
  }
  
  const result = await this.put(`/chat/conversations/${conversationId}/reopen`, payload)
  
  this.invalidateConversationsCache()
  
  return result
},

  // === Métodos para Status ===
  async toggleConversationStatus(conversationId, status, reason = null, extraData = {}) {
    return this.put(`/chat/conversations/${conversationId}/status-toggle`, {
      status,
      reason,
      ...extraData,
    })
  },
  async togglePinConversation(conversationId, isPinned = true) { 
    return this.patch(`/chat/conversations/${conversationId}`, { is_pinned: isPinned }) 
  },
  async toggleMuteConversation(conversationId, isMuted = true) { 
    return this.patch(`/chat/conversations/${conversationId}`, { is_muted: isMuted }) 
  },
  async archiveConversation(conversationId) { 
    return this.patch(`/chat/conversations/${conversationId}/status`, { status: 'archived' }) 
  },

  /**
   * 🔍 BUSCAR CONTATOS COM FILTROS
   */
  async getContacts(params = {}) {
    try {
      // ✅ USAR ENDPOINT DO CONCORD CRM
      const response = await this.get('/contacts', {
        per_page: params.limit || 50,
        order: params.order || 'created_at|desc',
        search: params.search || '',
        ...params
      })
      
      console.log('📋 Contatos recebidos:', response)
      return response
    } catch (error) {
      console.error('❌ Erro ao buscar contatos:', error)
      throw error
    }
  },

/**
 * 👥 BUSCAR USUÁRIOS - VERSÃO COM AVATAR FALLBACK
 */
async getUsers(params = {}) {
  try {
    console.log('🔍 Buscando usuários com parâmetros:', params)
    
    const response = await this.get('/users', {
      per_page: params.limit || 100,
      order: params.order || 'name|asc',
      search: params.search || '',
      ...params
    })
    
    console.log('📋 Resposta bruta da API:', response)
    
    // ✅ EXTRAIR DADOS DE FORMA ROBUSTA
    let users = []
    
    if (response) {
      if (Array.isArray(response)) {
        users = response
      } else if (response.data && Array.isArray(response.data)) {
        users = response.data
      } else if (response.users && Array.isArray(response.users)) {
        users = response.users
      } else if (typeof response === 'object' && response.id) {
        users = [response]
      }
    }
    
    if (!Array.isArray(users)) {
      console.warn('⚠️ Não foi possível extrair array de usuários:', response)
      users = []
    }
    
    // ✅ PROCESSAR AVATARS
    users = users.map(user => ({
      ...user,
      // ✅ PRIORIZAR AVATAR UPLOADED, DEPOIS AVATAR_URL, DEPOIS GRAVATAR
      avatar_url: user.uploaded_avatar_url || 
                  user.avatar_url || 
                  `https://www.gravatar.com/avatar/${btoa(user.email || 'default')}?s=40&d=identicon`
    }))
    
    console.log(`✅ ${users.length} usuários processados com avatars:`, users)
    
    return users
    
  } catch (error) {
    console.error('❌ Erro ao buscar usuários:', error)
    throw error
  }
},

  /**
   * 📺 BUSCAR CANAIS DE CHAT
   */
  async getChannels(params = {}) {
    try {
      const response = await this.get('/chat/channels', {
        paginate: false, // Não paginar para modal
        include_stats: false,
        is_active: true, // Apenas canais ativos
        ...params
      })
      
      console.log('📺 Canais recebidos:', response)
      return response
    } catch (error) {
      console.error('❌ Erro ao buscar canais:', error)
      throw error
    }
  },

  /**
   * 🔍 BUSCA ESPECÍFICA DE CONTATOS (para autocomplete)
   */
  async searchContacts(term, limit = 20) {
    try {
      if (!term || term.length < 2) {
        return this.getContacts({ limit })
      }
      
      const response = await this.get('/contacts/search', {
        q: term,
        limit: limit,
        fields: 'id,display_name,first_name,last_name,email,phones,best_profile_photo'
      })
      
      return response
    } catch (error) {
      console.error('❌ Erro na busca de contatos:', error)
      // Fallback para busca normal
      return this.getContacts({ search: term, limit })
    }
  },

  /**
   * 👥 BUSCA ESPECÍFICA DE USUÁRIOS (para autocomplete)
   */
  async searchUsers(term, limit = 50) {
    try {
      if (!term || term.length < 2) {
        return this.getUsers({ limit })
      }
      
      const response = await this.get('/users/search', {
        q: term,
        limit: limit,
        fields: 'id,name,email,avatar_url'
      })
      
      return response
    } catch (error) {
      console.error('❌ Erro na busca de usuários:', error)
      // Fallback para busca normal
      return this.getUsers({ search: term, limit })
    }
  },

/**
 * ✅ CRIAR CONTATO DIRETAMENTE - SIMPLIFICADO
 * @param {object} contactData Dados do contato
 * @returns {Promise<object>} Contato criado
 */
async createContact(contactData) {
  try {
    console.log('➕ Criando novo contato:', contactData)
    
    // ✅ Preparar dados básicos (sem campos que podem não existir)
    const payload = {
      first_name: contactData.contact_name || contactData.first_name,
      email: contactData.contact_email || contactData.email || null
    }
    
    // ✅ Adicionar last_name se fornecido
    if (contactData.last_name) {
      payload.last_name = contactData.last_name
    }
    
    // ✅ Adicionar telefone se fornecido
    if (contactData.phone) {
      payload.phones = [{
        number: contactData.phone.replace(/\D/g, ''), // Apenas números
        type: 'mobile'
      }]
    }
    
    const response = await window.Innoclapps.request().post('/contacts', payload)
    
    console.log('✅ Contato criado:', response.data)
    
    // ✅ Limpar cache de contatos
    this.clearCache('/contacts')
    
    return response.data
    
  } catch (error) {
    console.error('❌ Erro ao criar contato:', error)
    throw error
  }
},

/**
 * ✅ BUSCAR CONTATO POR TELEFONE - SIMPLIFICADO
 * @param {string} phone Número de telefone
 * @returns {Promise<object|null>} Contato encontrado ou null
 */
async findContactByPhone(phone) {
  try {
    const cleanPhone = phone.replace(/\D/g, '')
    console.log('🔍 Buscando contato por telefone:', cleanPhone)
    
    // ✅ Buscar usando apenas o termo de busca
    const response = await this.getContacts({
      search: cleanPhone,
      limit: 5 // Buscar alguns para verificar
    })
    
    // ✅ Procurar contato que tenha esse telefone
    const contacts = Array.isArray(response) ? response : []
    const contact = contacts.find(c => {
      // ✅ Verificar se tem telefones
      if (!c.phones || !Array.isArray(c.phones)) return false
      return c.phones.some(p => {
        if (!p.number) return false
        const contactPhone = p.number.replace(/\D/g, '')
        return contactPhone === cleanPhone || contactPhone.endsWith(cleanPhone) || cleanPhone.endsWith(contactPhone)
      })
    })
    
    console.log('🔍 Contato encontrado:', contact || 'Nenhum')
    return contact || null
    
  } catch (error) {
    console.error('❌ Erro ao buscar contato por telefone:', error)
    return null
  }
},

/**
 * ✅ OBTER USUÁRIO ATUAL - SIMPLIFICADO
 * @returns {Promise<object>} Dados do usuário atual
 */
async getCurrentUser() {
  try {
    // ✅ Primeiro tentar pegar do Innoclapps
    if (window.Innoclapps?.user) {
      return window.Innoclapps.user
    }
    
    // ✅ Senão buscar via API padrão
    const response = await window.Innoclapps.request().get('/user')
    return response.data
    
  } catch (error) {
    console.error('❌ Erro ao obter usuário atual:', error)
    
    // ✅ Fallback para dados básicos se disponível
    if (window.Innoclapps?.scriptConfig) {
      const config = window.Innoclapps.scriptConfig('chat')
      if (config?.user_id) {
        return {
          id: config.user_id,
          name: config.user_name || 'Usuário',
          email: config.user_email || ''
        }
      }
    }
    
    throw error
  }
},

  // === Métodos para Exportação/Estatísticas ===
  async exportConversation(conversationId, format = 'pdf') { 
    return this.get(`/chat/conversations/${conversationId}/export`, { format }) 
  },
  async getConversationStats() { return this.get('/chat/stats') },
  async getArchivedConversations(params = {}) {
    return this.get('/chat/conversations/archived', params)
  },
  async getConversationSummary(params = {}) {
    return this.get('/chat/conversations/summary', params)
  },

  // === Helpers ===
  invalidateMessagesCache(conversationId) {
    clearCache(`/chat/conversations/${conversationId}/messages`)
    debug(`🧹 Cache de mensagens invalidado para conversa: ${conversationId}`)
  },
  invalidateConversationsCache() {
    clearCache('/chat/conversations')
    clearCache('/chat/conversations/archived')
    clearCache('/chat/conversations/summary')
    debug('🧹 Cache de conversas invalidado')
  },
  buildUrl(endpoint, params = {}) { return buildUrl(endpoint, params) },
  getHeaders() { return getDefaultHeaders() },
  async healthCheck() { return this.get('/chat-test') }
}

debug('🚀 Serviço de API inicializado', { baseURL: API_CONFIG.baseURL, debug: API_CONFIG.debug, cacheTTL: API_CONFIG.cacheTTL })

export default apiService