// modules/Chat/resources/js/servicos/chatService.js
import { ref } from 'vue'
import websocketService from './websocket.js'
import chatStore from './chatStore.js'
import apiService from './api.js'
import { useChatNotifications } from '../composables/useChatNotifications.js'

class ChatService {
  constructor() {
    this.initialized = ref(false)
    this.sidebarMounted = ref(false)
    this.isOnChatPage = ref(false)
    this.unreadCount = ref(0)
    
    this.notifications = null
    this.eventListeners = []
    this.sidebarApp = null
    this.sidebarContainer = null
    this.routeObserver = null
    this.lastDetectedPath = null
    
    // Cache para otimização
    this.badgeElements = new Set()
    this.lastTitleUpdate = ''
    
    // Debounced functions
    this.debouncedUpdateBadges = this.debounce(this._updateBadges.bind(this), 100)
    this.debouncedUpdateTitle = this.debounce(this._updatePageTitle.bind(this), 200)
  }

  async init() {
    if (this.initialized.value) return true

    try {
      console.log('🚀 Iniciando Chat Service...')

      // Carregar notificações
      this.notifications = useChatNotifications()
      await this.notifications.initializeChatNotifications()

      // ✅ CARREGAR DADOS INICIAIS DO CHATSTORE
      await this.loadInitialChatData()

      // Conectar WebSocket se necessário
      if (!websocketService.isConnected) {
        await this.connectWebSocket()
      }

      // Configurar listeners
      this.setupGlobalListeners()

      // ✅ CORRIGIDO: Inicializar sidebar
      await this.initializeSidebar()

      this.initialized.value = true
      
      // Expor globalmente
      window.chatService = this
      
      console.log('✅ Chat Service inicializado com sucesso')
      return true
    } catch (error) {
      console.error('❌ Erro ao inicializar Chat Service:', error)
      return false
    }
  }

  // ✅ MÉTODO CORRIGIDO: Carregar dados iniciais do chat
  async loadInitialChatData() {
    try {
      console.log('🔄 Carregando dados iniciais do chat...')
      
      // ✅ 1. CARREGAR ESTATÍSTICAS BÁSICAS
      try {
        const summaryResponse = await apiService.getConversationSummary()
        
        if (summaryResponse) {
          const { unread_count = 0, total_conversations = 0 } = summaryResponse
          
          console.log(`📊 Estatísticas carregadas: ${total_conversations} conversas, ${unread_count} não lidas`)
          
          // ✅ 2. SEMPRE CARREGAR LISTA PARA SIDEBAR (mesmo sem não lidas)
          await this.loadConversationsForSidebar()
          
          return true
        }
      } catch (summaryError) {
        console.warn('⚠️ Erro ao carregar estatísticas, tentando método alternativo...', summaryError)
        
        // ✅ FALLBACK: Carregar conversas diretamente
        await this.loadConversationsForSidebar()
      }
      
    } catch (error) {
      console.warn('⚠️ Erro ao carregar dados iniciais do chat:', error)
    }
  }

  // ✅ NOVO MÉTODO: Carregar conversas para sidebar
  async loadConversationsForSidebar() {
    try {
      console.log('📋 Carregando conversas para sidebar...')
      
      const response = await apiService.getConversations({
        page: 1,
        per_page: 15,
        only_unread: false,
        include_archived: false
      })
      
      if (response?.data && Array.isArray(response.data)) {
        const conversations = response.data.filter(conv => 
          conv && typeof conv === 'object' && conv.id
        )
        
        // ✅ DEFINIR NO CHATSTORE
        chatStore.setConversas(conversations)
        
        // ✅ CALCULAR CONTADOR DE NÃO LIDAS
        const unreadCount = conversations.reduce((total, conv) => {
          return total + (conv.unread_count || 0)
        }, 0)
        
        console.log('📊 Conversas carregadas - detalhes unread:', {
          total: conversations.length,
          unreadCount,
          conversasComUnread: conversations.filter(c => (c.unread_count || 0) > 0).map(c => ({
            id: c.id,
            name: c.contact_name,
            unread: c.unread_count
          }))
        })
        
        // ✅ CORREÇÃO: Atualizar AMBOS os contadores
        chatStore.updateUnreadTotal(unreadCount)
        this.updateUnreadCount(unreadCount)
        
        console.log(`✅ Conversas carregadas para sidebar: ${conversations.length} conversas, ${unreadCount} não lidas`)
        return true
      }
      
    } catch (error) {
      console.warn('⚠️ Erro ao carregar conversas para sidebar:', error)
      
      // ✅ FALLBACK: Tentar apenas contador
      try {
        const response = await apiService.getConversationSummary()
        if (response?.unread_count) {
          const count = response.unread_count
          chatStore.updateUnreadTotal(count)
          this.updateUnreadCount(count)
          console.log(`📊 Apenas contador carregado: ${count} não lidas`)
        }
      } catch (fallbackError) {
        console.warn('⚠️ Todos os fallbacks falharam:', fallbackError)
      }
    }
  }

  async connectWebSocket() {
    try {
      await websocketService.init()
      
      const user = window.Innoclapps?.scriptConfig?.('chat')?.user
      if (user && websocketService.identifyUser) {
        websocketService.identifyUser(user)
      }
    } catch (error) {
      console.warn('⚠️ WebSocket não conectou:', error.message)
    }
  }

  setupGlobalListeners() {
    this.detectChatPage()
    
    const listeners = [
      ['popstate', () => this.detectChatPage()],
      ['chat-message-received', (event) => this.handleGlobalMessage(event.detail)],
      ['chat-unread-count-changed', (event) => this.updateUnreadCount(event.detail.count)],
      ['chat-store-updated', (event) => this.handleStoreUpdate(event.detail)],
      ['beforeunload', () => this.cleanup()],
      ['pagehide', () => this.cleanup()],
    ]

    listeners.forEach(([event, handler]) => {
      window.addEventListener(event, handler)
      this.eventListeners.push({ element: window, event, handler })
    })
    
    // ✅ NOVO: Detectar mudanças de rota em SPA
    this.setupRouteDetection()
  }

  // ✅ NOVO MÉTODO: Detectar mudanças de rota
  setupRouteDetection() {
    try {
      // Override history methods para detectar navegação programática
      const originalPushState = history.pushState
      const originalReplaceState = history.replaceState
      
      history.pushState = (...args) => {
        originalPushState.apply(history, args)
        setTimeout(() => this.detectChatPage(), 100)
      }
      
      history.replaceState = (...args) => {
        originalReplaceState.apply(history, args)
        setTimeout(() => this.detectChatPage(), 100)
      }
      
      // Observer para mudanças no DOM
      this.routeObserver = new MutationObserver(() => {
        const currentPath = window.location.pathname
        if (this.lastDetectedPath !== currentPath) {
          this.lastDetectedPath = currentPath
          this.detectChatPage()
        }
      })
      
      this.lastDetectedPath = window.location.pathname
      this.routeObserver.observe(document.body, { 
        childList: true, 
        subtree: true 
      })
    } catch (error) {
      console.warn('⚠️ Erro ao configurar detecção de rota:', error)
    }
  }

  // ✅ MÉTODO CORRIGIDO: initializeSidebar
  async initializeSidebar() {
    if (this.sidebarMounted.value) return

    try {
      console.log('🔄 Inicializando sidebar...')
      
      // Aguardar Vue estar disponível
      await this.waitForVue()
      
      // Aguardar DOM
      if (document.readyState === 'loading') {
        await new Promise(resolve => {
          document.addEventListener('DOMContentLoaded', resolve, { once: true })
        })
      }

      // Aguardar um pouco mais para garantir que tudo esteja carregado
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Criar container
      await this.createSidebarContainer()
      
      // Montar componente
      await this.mountSidebarComponent()
      
      this.sidebarMounted.value = true
      console.log('✅ Sidebar montada com sucesso')
      
    } catch (error) {
      console.error('❌ Erro ao inicializar sidebar:', error)
      this.sidebarMounted.value = false
    }
  }

  async createSidebarContainer() {
    let container = document.getElementById('chat-sidebar-container')
    if (!container) {
      container = document.createElement('div')
      container.id = 'chat-sidebar-container'
      container.setAttribute('data-chat-sidebar', 'true')
      document.body.appendChild(container)
    }
    this.sidebarContainer = container
  }

  async mountSidebarComponent() {
    try {
      const { createApp } = await import('vue')
      const ChatSidebar = await import('../componentes/ChatSidebar.vue')
      
      if (!ChatSidebar.default) {
        throw new Error('ChatSidebar component not found')
      }
      
      this.sidebarApp = createApp(ChatSidebar.default)
      
      // Configurar app se necessário
      if (window.Innoclapps?.configureVueApp) {
        window.Innoclapps.configureVueApp(this.sidebarApp)
      }
      
      this.sidebarApp.mount(this.sidebarContainer)
      
    } catch (error) {
      console.error('❌ Erro ao montar sidebar component:', error)
      throw error
    }
  }

  async waitForVue() {
    return new Promise((resolve) => {
      const checkVue = () => {
        if (window.Vue || (window.Innoclapps && window.Innoclapps.app)) {
          resolve()
        } else {
          setTimeout(checkVue, 50)
        }
      }
      checkVue()
    })
  }

  // ✅ MELHORAR: detectChatPage
  detectChatPage() {
    const wasOnChatPage = this.isOnChatPage.value
    this.isOnChatPage.value = window.location.pathname.includes('/chat')
    
    console.log('🔍 Detectando página de chat:', {
      wasOnChatPage,
      isOnChatPage: this.isOnChatPage.value,
      pathname: window.location.pathname
    })
    
    // Reset contador se entrou na página de chat
    if (!wasOnChatPage && this.isOnChatPage.value) {
      this.resetUnreadCount()
    }
    
    // ✅ NOVO: Atualizar sidebar quando mudar de página
    if (this.sidebarMounted.value) {
      // Emitir evento para o sidebar
      window.dispatchEvent(new CustomEvent('chat-page-changed', {
        detail: { 
          isOnChatPage: this.isOnChatPage.value,
          pathname: window.location.pathname
        }
      }))
    }
  }

  handleGlobalMessage(data) {
    if (!data || !this.shouldProcessMessage(data)) return

    const { conversation_id, message } = data
    if (!conversation_id || !message) return

    // ✅ INÍCIO DA CORREÇÃO: VERIFICAR MENSAGENS PRÓPRIAS ROBUSTAMENTE
    const currentUserId = window.Innoclapps?.scriptConfig?.('chat')?.user?.id
    const messageIsFromMe = message.is_from_me === true || message.whatsapp_key_from_me === true

    // Se a mensagem é marcada como "minha" pelo webhook, ignorar som e incremento de unread
    if (messageIsFromMe) {
      console.log('🚫 Mensagem enviada por mim ignorada para unread/som.')
      return
    }

    // Caso a flag acima seja falsa, mas o sender_id corresponda ao usuário do CRM, também ignorar
    const senderId = message.sender_id || message.sender?.id
    if (currentUserId && senderId && parseInt(currentUserId) === parseInt(senderId)) {
      console.log('🚫 Mensagem própria ignorada para unread/som')
      return
    }

    // ✅ USAR MÉTODO SEGURO PARA ATUALIZAR CONVERSA
    try {
      if (chatStore && typeof chatStore.updateConversationList === 'function') {
        chatStore.updateConversationList(conversation_id, message, false)
      } else if (chatStore && typeof chatStore.updateConversaComMensagem === 'function') {
        chatStore.updateConversaComMensagem(conversation_id, message, false)
      } else {
        this.updateConversationManually(conversation_id, message)
      }
    } catch (error) {
      console.warn('Erro ao atualizar conversa:', error)
      this.updateConversationManually(conversation_id, message)
    }

    // Incrementar contador
    this.incrementUnreadCount()
    
    // Som de notificação se não estiver na página de chat
    if (this.notifications && !this.isOnChatPage.value) {
      this.notifications.playNotificationSound?.()
    }
  }

  updateConversationManually(conversationId, message) {
    try {
      const conversations = chatStore.conversas.value || []
      const conversationIndex = conversations.findIndex(c => c.id === parseInt(conversationId))
      
      if (conversationIndex >= 0) {
        const updatedConversation = {
          ...conversations[conversationIndex],
          last_message: message.content || message.text || '',
          last_message_at: message.created_at || new Date().toISOString(),
          last_message_sender: message.sender_name || '',
          last_message_sender_id: message.sender_id,
          unread_count: (conversations[conversationIndex].unread_count || 0) + 1
        }
        
        const newConversations = [...conversations]
        newConversations[conversationIndex] = updatedConversation
        
        // Mover para o topo
        newConversations.sort((a, b) => {
          const aTime = new Date(a.last_message_at || 0).getTime()
          const bTime = new Date(b.last_message_at || 0).getTime()
          return bTime - aTime
        })
        
        chatStore.setConversas(newConversations)
        console.log('✅ Conversa atualizada manualmente:', conversationId)
        return true
      }
      return false
    } catch (error) {
      console.error('❌ Erro no updateConversationManually:', error)
      return false
    }
  }

  handleStoreUpdate(data) {
    if (data?.unreadTotal != null) {
      this.updateUnreadCount(data.unreadTotal)
    }
  }

  shouldProcessMessage(data) {
    // Filtros para evitar processamento desnecessário
    if (!data.conversation_id && !data.message) return false
    if (this.isOnChatPage.value) return false
    
    return true
  }

  incrementUnreadCount() {
    this.updateUnreadCount(this.unreadCount.value + 1)
  }

  updateUnreadCount(count) {
    const newCount = Math.max(0, parseInt(count) || 0)
    if (this.unreadCount.value === newCount) return
    
    console.log('🔄 ChatService: Atualizando unread count:', { de: this.unreadCount.value, para: newCount })
    
    this.unreadCount.value = newCount
    this.debouncedUpdateBadges()
    this.debouncedUpdateTitle()
    
    // ✅ CORREÇÃO: Sincronizar FORÇADAMENTE com chatStore
    try {
      if (chatStore && chatStore.unreadTotal) {
        console.log('🔄 Sincronizando ChatStore unreadTotal:', newCount)
        chatStore.unreadTotal.value = newCount
      }
      
      // ✅ NOVO: Também atualizar o método updateUnreadTotal do chatStore
      if (chatStore && typeof chatStore.updateUnreadTotal === 'function') {
        chatStore.updateUnreadTotal(newCount)
      }
    } catch (error) {
      console.warn('⚠️ Erro ao sincronizar com chatStore:', error)
    }
    
    // Emitir evento para outros componentes
    window.dispatchEvent(new CustomEvent('chat-unread-count-updated', {
      detail: { count: newCount }
    }))
  }

  _updateBadges() {
    const count = this.unreadCount.value
    const displayText = count > 99 ? '99+' : String(count)
    const shouldShow = count > 0

    // Atualizar badges existentes
    this.badgeElements.forEach(badge => {
      if (badge.isConnected) {
        badge.textContent = displayText
        badge.style.display = shouldShow ? 'inline' : 'none'
        badge.classList.toggle('show', shouldShow)
      } else {
        this.badgeElements.delete(badge)
      }
    })

    // Buscar novos badges
    const selectors = [
      '.chat-notification-badge',
      '[data-chat-badge]',
      '.notification-badge[data-type="chat"]'
    ]

    selectors.forEach(selector => {
      const elements = document.querySelectorAll(selector)
      elements.forEach(badge => {
        if (!this.badgeElements.has(badge)) {
          this.badgeElements.add(badge)
          badge.textContent = displayText
          badge.style.display = shouldShow ? 'inline' : 'none'
          badge.classList.toggle('show', shouldShow)
        }
      })
    })

    // PWA App Badge
    if ('setAppBadge' in navigator) {
      if (count > 0) {
        navigator.setAppBadge(count).catch(() => {})
      } else {
        navigator.clearAppBadge().catch(() => {})
      }
    }
  }

  _updatePageTitle() {
    const count = this.unreadCount.value
    const originalTitle = document.title.replace(/^\(\d+\)\s*/, '')
    const newTitle = count > 0 ? `(${count}) ${originalTitle}` : originalTitle
    
    if (this.lastTitleUpdate !== newTitle) {
      document.title = newTitle
      this.lastTitleUpdate = newTitle
    }
  }

  // Métodos públicos
  setOnChatPage(isOnChatPage) {
    this.isOnChatPage.value = isOnChatPage
    if (isOnChatPage) {
      this.resetUnreadCount()
    }
  }

  resetUnreadCount() {
    this.updateUnreadCount(0)
  }

  setUnreadCount(count) {
    this.updateUnreadCount(count)
  }

  getUnreadCount() {
    return this.unreadCount.value
  }

  getStats() {
    return {
      initialized: this.initialized.value,
      sidebarMounted: this.sidebarMounted.value,
      isOnChatPage: this.isOnChatPage.value,
      unreadCount: this.unreadCount.value,
      badgeElementsCount: this.badgeElements.size,
      listenersCount: this.eventListeners.length,
      conversationsLoaded: chatStore.conversas.value?.length || 0
    }
  }

  // ✅ MÉTODO PARA FORÇAR SINCRONIZAÇÃO (DEBUG)
  forceSyncUnreadCount() {
    try {
      console.log('🔧 Forçando sincronização de contadores...')
      
      // Pegar valor do chatService (fonte da verdade)
      const serviceCount = this.unreadCount.value
      
      // Forçar no chatStore
      if (chatStore && chatStore.updateUnreadTotal) {
        chatStore.updateUnreadTotal(serviceCount)
      }
      
      console.log('✅ Sincronização forçada:', {
        chatService: this.unreadCount.value,
        chatStore: chatStore.unreadTotal.value
      })
      
      return true
    } catch (error) {
      console.error('❌ Erro ao forçar sincronização:', error)
      return false
    }
  }

  // Utilitários
  debounce(func, wait) {
    let timeout
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(timeout)
        func(...args)
      }
      clearTimeout(timeout)
      timeout = setTimeout(later, wait)
    }
  }

  cleanup() {
    // Limpar sidebar
    if (this.sidebarApp && this.sidebarMounted.value) {
      try {
        this.sidebarApp.unmount()
        this.sidebarApp = null
      } catch (error) {
        console.warn('Erro ao destruir sidebar:', error)
      }
    }

    // Remover container
    if (this.sidebarContainer) {
      try {
        this.sidebarContainer.remove()
        this.sidebarContainer = null
      } catch (error) {
        console.warn('Erro ao remover container:', error)
      }
    }

    // ✅ NOVO: Cleanup do route observer
    if (this.routeObserver) {
      this.routeObserver.disconnect()
      this.routeObserver = null
    }

    // Remover listeners
    this.eventListeners.forEach(({ element, event, handler }) => {
      try {
        element.removeEventListener(event, handler)
      } catch (error) {
        console.warn('Erro ao remover listener:', error)
      }
    })
    this.eventListeners = []

    // Limpar caches
    this.badgeElements.clear()
    
    // Reset estado
    this.initialized.value = false
    this.sidebarMounted.value = false
    this.resetUnreadCount()
  }

  destroy() {
    this.cleanup()
  }

  // ✅ MÉTODO PÚBLICO PARA RECARREGAR DADOS
  async refreshData() {
    try {
      console.log('🔄 Atualizando dados do chat...')
      await this.loadInitialChatData()
      return true
    } catch (error) {
      console.error('❌ Erro ao atualizar dados:', error)
      return false
    }
  }
}

// Instância singleton
const chatService = new ChatService()

// Auto-inicializar de forma mais robusta
if (typeof window !== 'undefined') {
  const initChatService = () => {
    if (!chatService.initialized.value) {
      chatService.init().catch(error => {
        console.error('❌ Falha na inicialização do Chat Service:', error)
      })
    }
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      // Aguardar mais tempo para garantir que tudo esteja carregado
      setTimeout(initChatService, 2000)
    })
  } else {
    // DOM já carregado
    setTimeout(initChatService, 2000)
  }
}

// ✅ EXPOR MÉTODOS PARA DEBUG
window.debugChatCounters = {
  forceSyncUnreadCount: () => chatService.forceSyncUnreadCount(),
  debugUnreadCount: () => chatStore.debugUnreadCount(),
  recheckUnread: () => {
    if (chatStore) {
      chatStore.updateUnreadTotal()
    }
  }
}

// ✅ DEBUG TEMPORÁRIO
window.debugChatSidebar = function() {
  console.log('=== DEBUG CHAT SIDEBAR ===')
  console.log('ChatService Stats:', chatService.getStats())
  console.log('Sidebar Container:', document.getElementById('chat-sidebar-container'))
  console.log('Vue disponível:', !!window.Vue)
  console.log('Innoclapps disponível:', !!window.Innoclapps)
  console.log('ChatSidebar component:', document.querySelector('[data-chat-sidebar]'))
  console.log('ChatStore conversas:', chatStore.conversas.value?.length || 0)
  console.log('ChatStore unread:', chatStore.unreadTotal.value || 0)
}

// Auto-debug após 5 segundos
setTimeout(() => {
  if (window.debugChatSidebar) {
    window.debugChatSidebar()
  }
}, 5000)

export default chatService