import { ref, shallowRef, markRaw, watch } from 'vue'

class ChatStore {
  constructor() {
    this.conversas = shallowRef([])
    this.conversaSelecionada = ref(null)
    this.usuariosDigitando = shallowRef(new Map())
    this.unreadTotal = ref(0)
    this.isLoading = ref(false)

    this.setupWatchers()
  }

  setupWatchers() {
    // ✅ CORREÇÃO: Watch sem deep para evitar erro de iteração
    watch(
      () => this.conversas.value.length, 
      () => this.updateUnreadTotal(),
      { immediate: false }
    )
  }

  // ========= OPERAÇÕES DE ESTADO =========
  setConversas(conversas) {
    // ✅ CORREÇÃO: Validar entrada para evitar iteração de undefined
    const validConversas = Array.isArray(conversas) ? conversas : []
    this.conversas.value = markRaw(validConversas)
  }

  addConversa(conversa) {
    if (!conversa || this.conversas.value.find(c => c.id === conversa.id)) return
    this.conversas.value = markRaw([conversa, ...this.conversas.value])
  }

  updateConversa(conversaAtualizada) {
    const index = this.conversas.value.findIndex(c => c.id === conversaAtualizada.id)
    if (index === -1) return

    const array = [...this.conversas.value]
    array[index] = { ...array[index], ...conversaAtualizada }
    this.conversas.value = markRaw(array)

    if (this.conversaSelecionada.value?.id === conversaAtualizada.id) {
      this.conversaSelecionada.value = array[index]
    }
  }

  removeConversa(conversationId) {
    this.conversas.value = markRaw(
      this.conversas.value.filter(c => c.id !== parseInt(conversationId))
    )
    
    if (this.conversaSelecionada.value?.id === parseInt(conversationId)) {
      this.conversaSelecionada.value = null
    }
  }

  selectConversa(conversa) {
    this.conversaSelecionada.value = conversa
  }

  updateConversaComMensagem(conversationId, message, isOwnMessage = false) {
    const convId = parseInt(conversationId)
    const index = this.conversas.value.findIndex(c => c.id === convId)
    
    if (index === -1) {
      this.loadMissingConversation(convId, message)
      return
    }

    const array = [...this.conversas.value]
    const conversa = { ...array[index] }

    conversa.last_message = message.content || message.message || ''
    conversa.last_message_at = message.sent_at || message.created_at || new Date().toISOString()
    conversa.last_message_sender = message.sender_name || 'Usuário'
    conversa.updated_at = new Date().toISOString()

    if (!isOwnMessage && this.conversaSelecionada.value?.id !== convId) {
      conversa.unread_count = (conversa.unread_count || 0) + 1
    }

    array.splice(index, 1)
    array.unshift(conversa)
    
    this.conversas.value = markRaw(array)
  }

  async loadMissingConversation(conversationId, message) {
    try {
      const { default: servicoApi } = await import('./api.js')
      const response = await servicoApi.getConversation(conversationId)
      
      if (response && response.id) {
        const novaConversa = {
          ...response,
          type: response.type || (response.participants ? 'internal' : 'external'),
          unread_count: 1,
          contact_name: response.contact_name || response.subject || `Conversa #${response.id}`,
          last_message: message.content || message.message || '',
          last_message_at: message.sent_at || message.created_at || new Date().toISOString(),
          last_message_sender: message.sender_name || 'Usuário'
        }
        
        this.conversas.value = markRaw([novaConversa, ...this.conversas.value])
      }
    } catch (error) {
      console.error('Erro ao carregar conversa:', error)
    }
  }

  markConversaAsRead(conversationId) {
    const index = this.conversas.value.findIndex(c => c.id === parseInt(conversationId))
    if (index === -1) return

    const array = [...this.conversas.value]
    const conversa = { ...array[index] }

    if (conversa.unread_count > 0) {
      conversa.unread_count = 0
      conversa.read_at = new Date().toISOString()
      array[index] = conversa
      this.conversas.value = markRaw(array)
    }
  }
  
  getTopConversations(limit = 15) {
    return this.conversas.value
      .filter(c => {
        return c.unread_count > 0 || 
               c.last_message_at || 
               this.usuariosDigitando.value.has(c.id)
      })
      .sort((a, b) => {
        const aTyping = this.usuariosDigitando.value.has(a.id)
        const bTyping = this.usuariosDigitando.value.has(b.id)
        if (aTyping && !bTyping) return -1
        if (bTyping && !aTyping) return 1
        
        if (a.unread_count > 0 && b.unread_count === 0) return -1
        if (b.unread_count > 0 && a.unread_count === 0) return 1
        
        const aTime = new Date(a.last_message_at || 0).getTime()
        const bTime = new Date(b.last_message_at || 0).getTime()
        return bTime - aTime
      })
      .slice(0, limit)
  }

  setUsuarioDigitando(conversationId, userId, userName, isTyping) {
    const map = new Map(this.usuariosDigitando.value)
    const convId = parseInt(conversationId)

    if (!map.has(convId)) {
      map.set(convId, { users: [], timestamp: Date.now() })
    }

    const typing = map.get(convId)

    if (isTyping) {
      const exists = typing.users.some(u => u.id === userId)
      if (!exists) {
        typing.users.push({ id: userId, name: userName, startedAt: Date.now() })
        typing.timestamp = Date.now()
      }
    } else {
      typing.users = typing.users.filter(u => u.id !== userId)
      if (typing.users.length === 0) {
        map.delete(convId)
      }
    }

    this.usuariosDigitando.value = markRaw(map)
  }

  updateUnreadTotal(forceCount = null) {
    try {
      let newTotal
      
      if (forceCount !== null) {
        newTotal = Math.max(0, parseInt(forceCount) || 0)
      } else {
        // ✅ CORREÇÃO: Verificar se conversas é array válido
        if (!Array.isArray(this.conversas.value)) {
          newTotal = 0
        } else {
          newTotal = this.conversas.value.reduce(
            (sum, conv) => sum + (conv.unread_count || 0), 0
          )
        }
      }
      
      if (this.unreadTotal.value !== newTotal) {
        this.unreadTotal.value = newTotal
        
        // ✅ CORREÇÃO: Sincronização simplificada sem loops
        if (window.chatService?.unreadCount && 
            window.chatService.unreadCount.value !== newTotal) {
          setTimeout(() => {
            if (window.chatService?.unreadCount && 
                window.chatService.unreadCount.value !== newTotal) {
              window.chatService.unreadCount.value = newTotal
            }
          }, 50)
        }
      }
    } catch (error) {
      console.error('Erro em updateUnreadTotal:', error)
    }
  }

  getConversaById(id) {
    return this.conversas.value.find(c => c.id === parseInt(id))
  }

  // ========= MÉTODOS DE COMPATIBILIDADE =========
  selectConversation(conversationId) {
    const conversa = this.conversas.value.find(c => c.id === parseInt(conversationId))
    if (conversa) {
      this.selectConversa(conversa)
    }
  }

  markConversationAsReadGlobal(conversationId) {
    this.markConversaAsRead(conversationId)
  }

  updateConversationMeta(data) {
    this.updateConversa(data)
  }

  updateConversationList(conversationId, message, isOwnMessage = false) {
    this.updateConversaComMensagem(conversationId, message, isOwnMessage)
  }

  addConversation(conversa) {
    this.addConversa(conversa)
  }

  removeConversation(conversationId) {
    this.removeConversa(conversationId)
  }

  // ========= MÉTODOS ASYNC PARA API =========
  async markAsUnread(conversationId) {
    try {
      this.updateConversa({
        id: conversationId,
        unread_count: 1
      })
      
      const { default: servicoApi } = await import('./api.js')
      const response = await servicoApi.markAsUnread(conversationId)
      
      if (response.unread_count !== undefined) {
        this.updateConversa({
          id: conversationId,
          unread_count: response.unread_count
        })
      }
      
      return response
      
    } catch (error) {
      console.error('Erro ao marcar como não lida:', error)
      
      this.updateConversa({
        id: conversationId,
        unread_count: 0
      })
      
      throw error
    }
  }

  async markAsRead(conversationId) {
    try {
      this.markConversaAsRead(conversationId)
      
      const { default: servicoApi } = await import('./api.js')
      await servicoApi.markAsRead(conversationId)
      
    } catch (error) {
      console.error('Erro ao marcar como lida:', error)
      throw error
    }
  }

  // ========= HELPERS =========
  getStats() {
    return {
      totalConversas: this.conversas.value.length,
      naoLidas: this.conversas.value.filter(c => (c.unread_count || 0) > 0).length,
      unreadTotal: this.unreadTotal.value,
      conversaSelecionada: this.conversaSelecionada.value?.id || null,
      isLoading: this.isLoading.value
    }
  }

  reset() {
    this.conversas.value = markRaw([])
    this.conversaSelecionada.value = null
    this.usuariosDigitando.value = markRaw(new Map())
    this.unreadTotal.value = 0
    this.isLoading.value = false
  }
}

// ✅ INSTÂNCIA SINGLETON
const chatStore = new ChatStore()
window.chatStore = chatStore

export default chatStore