import { io } from 'socket.io-client'

const config = {
  url: null,
  debug: false,
  reconnectInterval: 5000,
  maxReconnectAttempts: 5,
  heartbeatInterval: 25000,
  connectionTimeout: 15000,
}

class WebSocketState {
  constructor() {
    this.reset()
  }

  reset() {
    this.socket = null
    this.isConnected = false
    this.isInitialized = false
    this.isDestroyed = false
    this.connectionState = 'disconnected'
    this.currentUser = null
    this.reconnectAttempts = 0
    this.lastError = null
    this.messageQueue = []
    this.heartbeatTimer = null
    this.reconnectTimer = null
    this.eventListeners = []
    this.eventCount = 0
    this.socketId = null
    
    // ✅ SISTEMA DE DEDUPLICAÇÃO SIMPLIFICADO
    this.processedEvents = new Map()
    this.processedLogicalMessages = new Map()
    this.eventTTL = 300000 // 5 minutos
    this.duplicateStats = { blocked: 0, allowed: 0 }
  }
}

const state = new WebSocketState()

function log(level, ...args) {
  if (!config.debug && level !== 'error') return
  const prefix = `[WebSocket] - ${level.toUpperCase()}:`
  console[level](prefix, ...args)
}

function emitGlobalEvent(eventName, data) {
  try {
    window.dispatchEvent(new CustomEvent(eventName, { detail: data }))
  } catch (e) {
    log('error', `Erro ao emitir evento ${eventName}:`, e)
  }
}

// ✅ DEDUPLICAÇÃO SIMPLIFICADA
function generateLogicalMessageKey(payload) {
  try {
    const message = payload.message || payload
    const messageId = message?.id
    const timestamp = message?.created_at || message?.sent_at || payload.timestamp
    
    if (messageId) {
      return `logical_msg_db_${messageId}_${timestamp || 'no_time'}`
    }
    
    const whatsappMessageId = message?.whatsapp_message_id
    if (whatsappMessageId) {
      return `logical_msg_wa_${whatsappMessageId}_${timestamp || 'no_time'}`
    }
    
    if (message?.content && payload.conversation_id) {
      const contentHash = message.content.substring(0, 50).replace(/\s/g, '')
      return `logical_msg_content_${payload.conversation_id}_${contentHash}_${timestamp || 'no_time'}`
    }
    
    return null
  } catch (error) {
    return null
  }
}

function isDuplicateEvent(payload) {
  try {
    const now = Date.now()
    cleanExpiredEvents()

    const logicalMessageKey = generateLogicalMessageKey(payload)
    if (logicalMessageKey) {
      if (state.processedLogicalMessages.has(logicalMessageKey)) {
        state.duplicateStats.blocked++
        return true
      }
      state.processedLogicalMessages.set(logicalMessageKey, now)
    }

    if (payload.event_id) {
      if (state.processedEvents.has(payload.event_id)) {
        state.duplicateStats.blocked++
        return true
      }
      state.processedEvents.set(payload.event_id, now)
    }
    
    state.duplicateStats.allowed++
    return false

  } catch (error) {
    state.duplicateStats.allowed++
    return false
  }
}

function cleanExpiredEvents() {
  const now = Date.now()
  const ttl = state.eventTTL
  
  for (const [key, timestamp] of state.processedEvents.entries()) {
    if (now - timestamp > ttl) {
      state.processedEvents.delete(key)
    }
  }
  
  for (const [key, timestamp] of state.processedLogicalMessages.entries()) {
    if (now - timestamp > ttl) {
      state.processedLogicalMessages.delete(key)
    }
  }
}

function getCurrentUser() {
  const chatConfig = window.Innoclapps?.scriptConfig('chat')

  if (chatConfig?.user_id) {
    return {
      id: parseInt(chatConfig.user_id),
      name: chatConfig.user_name || 'Usuário',
      email: chatConfig.user_email || null,
      super_admin: chatConfig.is_super_admin || false,
    }
  }

  return null
}

function getWebSocketUrl() {
  const chatConfig = window.Innoclapps?.scriptConfig('chat')

  return (
    chatConfig?.websocket_url?.replace(/\/socket\.io\/?$/, '') ||
    `${window.location.protocol}//${window.location.host}`
  )
}

function normalizeWsPayload(raw) {
  if (!raw) return {}

  if (typeof raw === 'object' && !Array.isArray(raw)) {
    return raw
  }

  if (Array.isArray(raw)) {
    if (raw.length === 2 && typeof raw[1] === 'object') {
      return raw[1] || {}
    }

    if (raw.length === 1 && typeof raw[0] === 'object') {
      return raw[0] || {}
    }

    const found = raw.find(item => item && typeof item === 'object')
    if (found) return found
  }

  return {}
}

function connect() {
  return new Promise((resolve, reject) => {
    if (state.isConnected) return resolve(true)
    if (state.isDestroyed) return reject(new Error('Serviço destruído.'))

    state.connectionState = 'connecting'
    config.url = config.url || getWebSocketUrl()

    // ✅ CORREÇÃO: Configuração Socket.IO otimizada
    const socket = io(config.url, {
      transports: ['websocket'],
      timeout: config.connectionTimeout,
      path: '/socket.io',
      reconnection: false,
      autoConnect: false,
      upgrade: true,
      rememberUpgrade: true
    })

    state.socket = socket

    const connectionTimeout = setTimeout(() => {
      socket.disconnect()
      reject(new Error('Timeout na conexão'))
    }, config.connectionTimeout)

    socket.on('connect', () => {
      clearTimeout(connectionTimeout)

      state.isConnected = true
      state.connectionState = 'connected'
      state.reconnectAttempts = 0
      state.socketId = socket.id

      // ✅ LIMPAR CACHE NA RECONEXÃO
      state.processedEvents.clear()
      state.processedLogicalMessages.clear()

      authenticateUser()
      setupEventListeners()
      startHeartbeat()
      processMessageQueue()

      emitGlobalEvent('websocket-connected', { socketId: socket.id })
      resolve(true)
    })

    socket.on('connect_error', error => {
      clearTimeout(connectionTimeout)
      state.lastError = error.message
      state.connectionState = 'error'
      reject(error)
    })

    socket.on('disconnect', reason => {
      state.isConnected = false
      state.connectionState = 'disconnected'
      state.socketId = null

      if (state.heartbeatTimer) {
        clearInterval(state.heartbeatTimer)
        state.heartbeatTimer = null
      }

      emitGlobalEvent('websocket-disconnected', { reason })

      if (!state.isDestroyed && reason !== 'io client disconnect') {
        scheduleReconnect()
      }
    })

    socket.connect()
  })
}

function setupEventListeners() {
  if (!state.socket || state.isDestroyed) return

  // ✅ CORREÇÃO: Limpar listeners existentes adequadamente
  if (state.eventListeners.length > 0) {
    state.eventListeners.forEach(({ event, handler }) =>
      state.socket.off(event, handler)
    )
    state.eventListeners = []
  }

  const addListener = (event, handler) => {
    const wrappedHandler = (...args) => {
      state.eventCount++
      handler(args.length === 1 ? args[0] : args)
    }
    
    state.socket.on(event, wrappedHandler)
    state.eventListeners.push({ event, handler: wrappedHandler })
  }

  // ✅ EVENTOS PRINCIPAIS COM DEDUPLICAÇÃO
  addListener('new_whatsapp_message', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-message-received', payload)
  })

  addListener('new_message', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-message-received', payload)
  })

  addListener('outbound_whatsapp_message', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-outbound-message-received', payload)
  })

  addListener('chat_message_received', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-message-received', payload)
  })

  addListener('whatsapp_status_update', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-message-status-updated', payload)
  })

  // ✅ EVENTOS DE DIGITAÇÃO (SEM DEDUPLICAÇÃO)
  addListener('user.typing_started', data => {
    const payload = normalizeWsPayload(data)
    emitGlobalEvent('chat-user-typing', { ...payload, is_typing: true })
  })

  addListener('user.typing_stopped', data => {
    const payload = normalizeWsPayload(data)
    emitGlobalEvent('chat-user-typing', { ...payload, is_typing: false })
  })

  // ✅ EVENTOS DE CONVERSA COM DEDUPLICAÇÃO
  addListener('chat.conversation.read', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-conversation-read', payload)
  })

  addListener('chat.conversation.updated', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-conversation-updated', payload)
  })

  addListener('new_conversation_created', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-new-conversation-created', payload)
  })

  addListener('conversation_finalized', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-conversation-finalized', payload)
  })

  addListener('conversation_reopened', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-conversation-reopened', payload)
  })

  addListener('conversation_auto_reopened', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-conversation-auto-reopened', payload)
  })

  addListener('conversation_status_updated', data => {
    const payload = normalizeWsPayload(data)
    if (isDuplicateEvent(payload)) return
    emitGlobalEvent('chat-conversation-status-updated', payload)
  })

  // ✅ EVENTOS INTERNOS (SEM DEDUPLICAÇÃO)
  addListener('conversation_access_denied', data => {
    const payload = normalizeWsPayload(data)
    emitGlobalEvent('chat-access-denied', payload)
  })

  addListener('auth_success', () => {
    log('info', 'Autenticado com sucesso')
  })

  addListener('auth_error', data => {
    log('error', 'Falha na autenticação:', data)
  })

  addListener('error', data => {
    log('error', 'Erro do servidor WebSocket:', data)
  })

  addListener('pong', () => {
    // Silencioso - heartbeat é interno
  })
}

function authenticateUser() {
  if (!state.socket?.connected) return

  state.currentUser = getCurrentUser()

  if (state.currentUser) {
    state.socket.emit('auth', {
      userId: state.currentUser.id,
      userName: state.currentUser.name,
      userEmail: state.currentUser.email,
    })
  }
}

function startHeartbeat() {
  if (state.heartbeatTimer) {
    clearInterval(state.heartbeatTimer)
    state.heartbeatTimer = null
  }

  state.heartbeatTimer = setInterval(() => {
    if (state.isConnected && state.socket?.connected) {
      state.socket.emit('ping', { timestamp: new Date().toISOString() })
    }
  }, config.heartbeatInterval)
}

function processMessageQueue() {
  if (!state.isConnected || state.messageQueue.length === 0) return

  while (state.messageQueue.length > 0) {
    const { event, data } = state.messageQueue.shift()
    sendEvent(event, data)
  }
}

function scheduleReconnect() {
  if (state.reconnectAttempts >= config.maxReconnectAttempts || state.isDestroyed) {
    state.connectionState = 'error'
    return
  }

  state.reconnectAttempts++
  const delay = Math.min(config.reconnectInterval * state.reconnectAttempts, 30000)

  if (state.reconnectTimer) {
    clearTimeout(state.reconnectTimer)
  }

  state.reconnectTimer = setTimeout(() => {
    if (!state.isDestroyed) {
      connect().catch(() => {
        // Erro já tratado no connect()
      })
    }
  }, delay)
}

function sendEvent(eventName, data = {}) {
  if (state.isDestroyed || !state.isInitialized) {
    return false
  }

  if (!state.isConnected || !state.socket?.connected) {
    state.messageQueue.push({ event: eventName, data })
    return false
  }

  state.socket.emit(eventName, data)
  return true
}

const websocketService = {
  async init(options = {}) {
    if (state.isInitialized) return state.isConnected

    Object.assign(config, options)
    state.isInitialized = true
    state.isDestroyed = false

    return await connect()
  },

  destroy() {
    if (state.isDestroyed) return

    state.isDestroyed = true

    if (state.reconnectTimer) {
      clearTimeout(state.reconnectTimer)
      state.reconnectTimer = null
    }

    if (state.heartbeatTimer) {
      clearInterval(state.heartbeatTimer)
      state.heartbeatTimer = null
    }

    if (state.socket) {
      state.socket.disconnect()
    }

    state.reset()
  },

  get isConnected() {
    return state.isConnected
  },

  get isInitialized() {
    return state.isInitialized
  },

  isValid() {
    return state.isInitialized && state.isConnected && !state.isDestroyed
  },

  getStatus() {
    return {
      isConnected: state.isConnected,
      isInitialized: state.isInitialized,
      connectionState: state.connectionState,
      currentUser: state.currentUser,
      reconnectAttempts: state.reconnectAttempts,
      maxReconnectAttempts: config.maxReconnectAttempts,
      lastError: state.lastError,
      eventCount: state.eventCount,
      socketId: state.socketId,
      queueSize: state.messageQueue.length,
      listenersCount: state.eventListeners.length,
      deduplication: {
        processedEvents: state.processedEvents.size,
        processedLogicalMessages: state.processedLogicalMessages.size,
        duplicatesBlocked: state.duplicateStats.blocked,
        eventsAllowed: state.duplicateStats.allowed,
        blockingRate: state.duplicateStats.blocked / (state.duplicateStats.blocked + state.duplicateStats.allowed) * 100 || 0
      }
    }
  },

  getCurrentUser: () => state.currentUser,

  identifyUser(user) {
    state.currentUser = user
    if (state.isConnected && state.socket) {
      state.socket.emit('auth', {
        userId: user.id,
        userName: user.name,
        userEmail: user.email
      })
    }
  },

  sendEvent,

  joinConversation: conversationId =>
    sendEvent('join_conversation', { conversation_id: conversationId }),

  leaveConversation: conversationId =>
    sendEvent('leave_conversation', { conversation_id: conversationId }),

  sendTypingIndicator: (conversationId, isTyping) =>
    sendEvent(isTyping ? 'typing_start' : 'typing_stop', {
      conversation_id: conversationId,
    }),

  sendConversationFinalized(conversationId, data = {}) {
    if (!websocketService.isConnected) return false

    try {
      const eventData = {
        conversation_id: conversationId,
        finalized_by: state.currentUser?.id,
        finalized_at: new Date().toISOString(),
        ...data
      }

      return sendEvent('conversation_finalized', eventData)

    } catch (error) {
      log('error', 'Erro ao enviar finalização:', error)
      return false
    }
  },

  sendConversationReopened(conversationId, reason = null) {
    if (!websocketService.isConnected) return false

    try {
      const eventData = {
        conversation_id: conversationId,
        reopened_by: state.currentUser?.id,
        reopened_at: new Date().toISOString(),
        reason: reason || 'Reabertura solicitada'
      }

      return sendEvent('conversation_reopened', eventData)

    } catch (error) {
      log('error', 'Erro ao enviar reabertura:', error)
      return false
    }
  },

  sendConversationStatusUpdated(conversationId, oldStatus, newStatus, data = {}) {
    if (!websocketService.isConnected) return false

    try {
      const eventData = {
        conversation_id: conversationId,
        old_status: oldStatus,
        new_status: newStatus,
        updated_by: state.currentUser?.id,
        updated_at: new Date().toISOString(),
        ...data
      }

      return sendEvent('conversation_status_updated', eventData)

    } catch (error) {
      log('error', 'Erro ao enviar status:', error)
      return false
    }
  },

  async restart() {
    websocketService.destroy()
    return await websocketService.init(config)
  },

  enableDebug() {
    config.debug = true
  },

  disableDebug() {
    config.debug = false
  },

  getEventListeners() {
    return state.eventListeners.map(({ event }) => event)
  },

  clearMessageQueue() {
    const queueSize = state.messageQueue.length
    state.messageQueue = []
    return queueSize
  },

  clearDeduplicationCache() {
    const totalCleared = state.processedEvents.size + state.processedLogicalMessages.size
    state.processedEvents.clear()
    state.processedLogicalMessages.clear()
    state.duplicateStats = { blocked: 0, allowed: 0 }
    return totalCleared
  },

  getDeduplicationStats() {
    return {
      processedEvents: state.processedEvents.size,
      processedLogicalMessages: state.processedLogicalMessages.size,
      eventTTL: state.eventTTL,
      duplicatesBlocked: state.duplicateStats.blocked,
      eventsAllowed: state.duplicateStats.allowed,
      blockingRate: state.duplicateStats.blocked / (state.duplicateStats.blocked + state.duplicateStats.allowed) * 100 || 0
    }
  },

  setDeduplicationTTL(ttl) {
    if (typeof ttl === 'number' && ttl > 0) {
      state.eventTTL = ttl
    }
  },

  forceCleanExpiredEvents() {
    cleanExpiredEvents()
  }
}

export default websocketService